<?php
/*
 Plugin Name: Internal Links Tool
 Description: Internal linking automation (scanner, keywords, linker, settings, CSV overrides).
 Version: 0.1.0
 Author: Shahar Barkai
 Author URI: https://bit.ly/ai-internal-links
*/

if (!defined('ABSPATH')) exit;

/**
 * Activation: create/upgrade DB tables.
 * Kept here because activation runs before plugins_loaded.
 */
function internallinkstool_activate() {
  $base = plugin_dir_path(__FILE__) . 'includes/';
  $db_file = $base . 'class-internallinkstool-db.php';

  if (file_exists($db_file)) require_once $db_file;

  if (class_exists('InternalLinksTool_DB') && method_exists('InternalLinksTool_DB', 'install')) {
    InternalLinksTool_DB::install();
  }
}
register_activation_hook(__FILE__, 'internallinkstool_activate');

final class InternalLinksTool_Plugin {

  public static function init() {
    add_action('plugins_loaded', [__CLASS__, 'boot']);
    add_action('admin_notices', [__CLASS__, 'render_support_banner']);
  }

  public static function boot() {
    // Require Yoast SEO to be active
    if (!defined('WPSEO_VERSION')) {
      add_action('admin_notices', [__CLASS__, 'render_yoast_missing_notice']);
      return;
    }

    add_action('admin_menu', [__CLASS__, 'admin_menu']);

    self::includes();

    // Init modules (only if class + method exist)
    foreach ([
      'InternalLinksTool_DB',
      'InternalLinksTool_Robots',
      'InternalLinksTool_Meta',
      'InternalLinksTool_Meta_Yoast',
      'InternalLinksTool_OpenAI',
      'InternalLinksTool_CSV',
      'InternalLinksTool_Scanner',
      'InternalLinksTool_Keywords',
      'InternalLinksTool_AnchorBanks',
      'InternalLinksTool_Linker',
      'InternalLinksTool_Admin',
      'InternalLinksTool_Strategy',
      // 'InternalLinksTool_Cron', // Background processing disabled for now
      'InternalLinksTool_Undo',
      'InternalLinksTool_Reports',
      'InternalLinksTool_ExistingLinks',
      'InternalLinksTool_Automation',
      'InternalLinksTool_AddLink',
    ] as $cls) {
      if (class_exists($cls) && method_exists($cls, 'init')) {
        $cls::init();
      }
    }
  }

  private static function includes() {
    $base = plugin_dir_path(__FILE__) . 'includes/';

    $files = [
      'class-internallinkstool-db.php',
      'class-internallinkstool-robots.php',
      'class-internallinkstool-meta.php',
      'class-internallinkstool-meta-yoast.php',
      'class-internallinkstool-openai.php',
      'class-internallinkstool-csv.php',
      'class-internallinkstool-scanner.php',
      'class-internallinkstool-keywords.php',
      'class-internallinkstool-anchorbanks.php',
      'class-internallinkstool-linker.php',
      'class-internallinkstool-admin.php',
      'class-internallinkstool-strategy.php',
      // 'class-internallinkstool-cron.php', // Background processing disabled for now
      'class-internallinkstool-undo.php',
      'class-internallinkstool-reports.php',
      'class-internallinkstool-existinglinks.php',
      'class-internallinkstool-automation.php',
      'class-internallinkstool-addlink.php',
    ];

    foreach ($files as $f) {
      $path = $base . $f;
      if (file_exists($path)) {
        require_once $path;
      }
    }
  }

  /**
   * MENUS: Only registered here (single source of truth).
   * Other classes must NOT add menus to avoid duplicates.
   */
  public static function admin_menu() {
    if (!current_user_can('manage_options')) return;

    $cap = 'manage_options';

    add_menu_page(
      'Internal Links Tool',
      'Internal Links Tool',
      $cap,
      'internallinkstool',
      [__CLASS__, 'render_dashboard'],
      'dashicons-admin-links',
      58
    );

    add_submenu_page('internallinkstool', 'Link Settings', '1. Link Settings', $cap, 'internallinkstool-settings', [__CLASS__, 'render_settings']);
    add_submenu_page('internallinkstool', 'Strategy', '2. Strategy', $cap, 'internallinkstool-strategy', [__CLASS__, 'render_strategy']);
    add_submenu_page('internallinkstool', 'Scanner', '3. Scanner', $cap, 'internallinkstool-scanner', [__CLASS__, 'render_scanner']);
    add_submenu_page('internallinkstool', 'Keywords', '4. Keywords', $cap, 'internallinkstool-keywords', [__CLASS__, 'render_keywords']);
    add_submenu_page('internallinkstool', 'AI Anchor Banks', '5. AI Anchor Banks', $cap, 'internallinkstool-anchor-banks', [__CLASS__, 'render_anchor_banks']);
    add_submenu_page('internallinkstool', 'Linker', '6. Linker', $cap, 'internallinkstool-linker', [__CLASS__, 'render_linker']);
    add_submenu_page('internallinkstool', 'Existing Links', 'Existing Links', $cap, 'internallinkstool-existing-links', [__CLASS__, 'render_existing_links']);
    add_submenu_page('internallinkstool', 'Add New Link', 'Add New Link', $cap, 'internallinkstool-addlink', [__CLASS__, 'render_addlink_page']);
  }

  public static function render_yoast_missing_notice() {
    if (!current_user_can('manage_options')) return;
    echo '<div class="notice notice-error"><p>';
    echo '<strong>Internal Links Tool:</strong> This plugin requires <a href="https://wordpress.org/plugins/wordpress-seo/" target="_blank" rel="noopener">Yoast SEO</a> to be installed and active. Please install and activate Yoast SEO, then reload this page.';
    echo '</p></div>';
  }

  public static function render_support_banner() {
    $screen = get_current_screen();
    if (!$screen || strpos($screen->id, 'internallinkstool') === false) return;

    echo '<div style="background:#fff;border:1px solid #ccd0d4;border-left:4px solid #f5c518;border-radius:4px;padding:12px 16px;margin:10px 0 0;display:flex;align-items:center;gap:16px;">';
    echo '<div>';
    echo '<p style="margin:0 0 6px;font-size:14px;"><strong>Help us improve the project and donate</strong></p>';
    echo '<a href="https://buymeacoffee.com/shahar.b" target="_blank" rel="noopener" class="button button-primary" style="background:#f5c518;border-color:#f5c518;color:#000;font-weight:bold;">Support this project</a>';
    echo '</div>';
    echo '</div>';
  }

  public static function render_dashboard() {
    if (!current_user_can('manage_options')) return;

    echo '<div class="wrap">';
    echo '<h1>Internal Links Tool</h1>';

    echo '<p>Automate your internal linking with AI-powered keyword extraction and smart link placement.</p>';

    // === HOW IT WORKS ===
    echo '<div style="background:#fff;border:1px solid #ccd0d4;border-radius:4px;padding:20px;margin:20px 0;">';
    echo '<h2 style="margin-top:0;">How It Works</h2>';
    echo '<p>Follow these steps in order to set up internal linking for your site:</p>';

    // Step 1
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:15px;margin:15px 0;">';
    echo '<h3 style="margin:0 0 10px 0;">Step 1: Link Settings</h3>';
    echo '<p style="margin:0;">Configure your basic settings before starting:</p>';
    echo '<ul style="margin:10px 0 0 20px;">';
    echo '<li>Enter your <strong>OpenAI API key</strong> (required for AI features)</li>';
    echo '<li>Choose which <strong>post types</strong> to include (posts, pages)</li>';
    echo '<li>Set <strong>max links per page</strong> and other limits</li>';
    echo '<li>Add any <strong>URLs or slugs to exclude</strong></li>';
    echo '<li>Choose <strong>automation mode</strong> for new posts (manual or auto)</li>';
    echo '</ul>';
    echo '<p style="margin:10px 0 0 0;"><a class="button" href="' . esc_url(admin_url('admin.php?page=internallinkstool-settings')) . '">Go to Link Settings</a></p>';
    echo '</div>';

    // Step 2
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:15px;margin:15px 0;">';
    echo '<h3 style="margin:0 0 10px 0;">Step 2: Strategy</h3>';
    echo '<p style="margin:0;">Define how links should be distributed:</p>';
    echo '<ul style="margin:10px 0 0 20px;">';
    echo '<li>Set <strong>tier budgets</strong> (how many links go to homepage vs categories vs articles)</li>';
    echo '<li>Set <strong>anchor text mix</strong> (exact match, partial, descriptive, etc.)</li>';
    echo '<li>Set <strong>minimum relevance score</strong> to ensure topical connections</li>';
    echo '<li>Run <strong>Auto-Tiering</strong> to classify your pages</li>';
    echo '</ul>';
    echo '<p style="margin:10px 0 0 0;"><a class="button" href="' . esc_url(admin_url('admin.php?page=internallinkstool-strategy')) . '">Go to Strategy</a></p>';
    echo '</div>';

    // Step 3
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:15px;margin:15px 0;">';
    echo '<h3 style="margin:0 0 10px 0;">Step 3: Scanner</h3>';
    echo '<p style="margin:0;">Map all your posts and pages:</p>';
    echo '<ul style="margin:10px 0 0 20px;">';
    echo '<li>Click <strong>Run Scanner</strong> to scan your content</li>';
    echo '<li>The scanner extracts: URL, title, H1, meta title, meta description</li>';
    echo '<li>Detects robots meta (noindex/nofollow)</li>';
    echo '<li>This is fast and uses <strong>no AI calls</strong></li>';
    echo '</ul>';
    echo '<p style="margin:10px 0 0 0;"><a class="button" href="' . esc_url(admin_url('admin.php?page=internallinkstool-scanner')) . '">Go to Scanner</a></p>';
    echo '</div>';

    // Step 4
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:15px;margin:15px 0;">';
    echo '<h3 style="margin:0 0 10px 0;">Step 4: Keywords</h3>';
    echo '<p style="margin:0;">Extract keywords for each page using AI:</p>';
    echo '<ul style="margin:10px 0 0 20px;">';
    echo '<li>Click <strong>Run Keywords</strong> to process pages</li>';
    echo '<li>AI analyzes: meta title, H1, URL slug, meta description</li>';
    echo '<li>Extracts <strong>primary keyword</strong> and <strong>secondary keywords</strong></li>';
    echo '<li>Includes brand names if found (Amazon, Walmart, etc.)</li>';
    echo '<li>Uses <strong>1 AI call per page</strong></li>';
    echo '</ul>';
    echo '<p style="margin:10px 0 0 0;"><a class="button" href="' . esc_url(admin_url('admin.php?page=internallinkstool-keywords')) . '">Go to Keywords</a></p>';
    echo '</div>';

    // Step 5
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:15px;margin:15px 0;">';
    echo '<h3 style="margin:0 0 10px 0;">Step 5: AI Anchor Banks</h3>';
    echo '<p style="margin:0;">Generate diverse anchor text variations for each page:</p>';
    echo '<ul style="margin:10px 0 0 20px;">';
    echo '<li>Click <strong>Generate Anchor Banks</strong> to process pages that have keywords</li>';
    echo '<li>AI creates 15-25 anchor text alternatives per page (exact, partial, descriptive, contextual, generic)</li>';
    echo '<li>Gives the Linker many more natural phrases to choose from</li>';
    echo '<li>Uses <strong>1 AI call per page</strong></li>';
    echo '</ul>';
    echo '<p style="margin:10px 0 0 0;"><a class="button" href="' . esc_url(admin_url('admin.php?page=internallinkstool-anchor-banks')) . '">Go to AI Anchor Banks</a></p>';
    echo '</div>';

    // Step 6
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:15px;margin:15px 0;">';
    echo '<h3 style="margin:0 0 10px 0;">Step 6: Linker</h3>';
    echo '<p style="margin:0;">Insert internal links into your content:</p>';
    echo '<ul style="margin:10px 0 0 20px;">';
    echo '<li>First, run a <strong>Dry Run</strong> to preview proposed links</li>';
    echo '<li>Review the preview: source page, target page, anchor text, relevance score</li>';
    echo '<li>When satisfied, run <strong>Insert Links</strong> to apply changes</li>';
    echo '<li>Uses <strong>no AI calls</strong> - just smart matching</li>';
    echo '</ul>';
    echo '<p style="margin:10px 0 0 0;"><a class="button button-primary" href="' . esc_url(admin_url('admin.php?page=internallinkstool-linker')) . '">Go to Linker</a></p>';
    echo '</div>';

    echo '</div>'; // End how it works box

    // === FEEDBACK SECTION ===
    echo '<div style="background:#fef8e7;border:1px solid #f0c33c;border-radius:4px;padding:20px;margin:20px 0;">';
    echo '<h2 style="margin-top:0;">We Would Love Your Feedback!</h2>';
    echo '<p>Help us improve this plugin by sharing your thoughts:</p>';
    echo '<ul style="margin:10px 0 10px 20px;">';
    echo '<li>Submit feedback</li>';
    echo '<li>Report a bug</li>';
    echo '<li>Request another feature</li>';
    echo '</ul>';
    echo '<p><a class="button button-primary" href="https://forms.gle/fRRovcFPu43fjMKd7" target="_blank" rel="noopener">Click Here to Submit Feedback</a></p>';
    echo '</div>';

    echo '</div>'; // End wrap
  }

  public static function render_scanner() {
    if (class_exists('InternalLinksTool_Scanner') && method_exists('InternalLinksTool_Scanner', 'render_scan_page')) {
      InternalLinksTool_Scanner::render_scan_page();
      return;
    }
    echo '<div class="wrap"><h1>Scanner</h1><p>Scanner renderer missing.</p></div>';
  }

  public static function render_keywords() {
    if (class_exists('InternalLinksTool_Keywords') && method_exists('InternalLinksTool_Keywords', 'render_page')) {
      InternalLinksTool_Keywords::render_page();
      return;
    }
    echo '<div class="wrap"><h1>Keywords</h1><p>Keywords renderer missing.</p></div>';
  }

  public static function render_anchor_banks() {
    if (class_exists('InternalLinksTool_AnchorBanks') && method_exists('InternalLinksTool_AnchorBanks', 'render_page')) {
      InternalLinksTool_AnchorBanks::render_page();
      return;
    }
    echo '<div class="wrap"><h1>AI Anchor Banks</h1><p>Anchor Banks renderer missing.</p></div>';
  }

  public static function render_strategy() {
    if (class_exists('InternalLinksTool_Strategy') && method_exists('InternalLinksTool_Strategy', 'render_page')) {
      InternalLinksTool_Strategy::render_page();
      return;
    }
    echo '<div class="wrap"><h1>Strategy</h1><p>Strategy renderer missing.</p></div>';
  }

  public static function render_linker() {
    if (class_exists('InternalLinksTool_Linker') && method_exists('InternalLinksTool_Linker', 'render_page')) {
      InternalLinksTool_Linker::render_page();
      return;
    }
    echo '<div class="wrap"><h1>Linker</h1><p>Linker renderer missing.</p></div>';
  }

  public static function render_settings() {
    if (class_exists('InternalLinksTool_Admin') && method_exists('InternalLinksTool_Admin', 'render_settings_page')) {
      InternalLinksTool_Admin::render_settings_page();
      return;
    }
    echo '<div class="wrap"><h1>Link Settings</h1><p>Settings renderer missing.</p></div>';
  }

  public static function render_addlink_page() {
    if (class_exists('InternalLinksTool_AddLink') && method_exists('InternalLinksTool_AddLink', 'render_page')) {
      InternalLinksTool_AddLink::render_page();
      return;
    }
    echo '<div class="wrap"><h1>Add New Link</h1><p>Add Link renderer missing.</p></div>';
  }

  public static function render_existing_links() {
    if (class_exists('InternalLinksTool_ExistingLinks') && method_exists('InternalLinksTool_ExistingLinks', 'render_page')) {
      InternalLinksTool_ExistingLinks::render_page();
      return;
    }
    echo '<div class="wrap"><h1>Existing Links</h1><p>Existing Links renderer missing.</p></div>';
  }

}

InternalLinksTool_Plugin::init();
