<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_Strategy {

  const OPT_STRATEGY = 'internallinkstool_strategy_settings';

  public static function init() {
    add_action('admin_post_internallinkstool_save_strategy', [__CLASS__, 'handle_save_strategy']);
    add_action('admin_post_internallinkstool_run_tiering', [__CLASS__, 'handle_run_tiering']);
    add_action('admin_post_internallinkstool_reset_tiering', [__CLASS__, 'handle_reset_tiering']);
    add_action('wp_ajax_internallinkstool_update_tier', [__CLASS__, 'ajax_update_tier']);
  }

  /**
   * Get strategy settings with defaults
   */
  public static function get_settings() {
    $defaults = [
      // Link budget by tier (must sum to 100)
      'budget_homepage' => 10,
      'budget_tier1'    => 50,
      'budget_tier2'    => 30,
      'budget_tier3'    => 10,

      // Anchor type mix (must sum to 100)
      'anchor_exact'       => 80,
      'anchor_partial'     => 10,
      'anchor_descriptive' => 5,
      'anchor_contextual'  => 5,
      'anchor_generic'     => 0,

      // Guardrails
      'max_links_per_source'  => 5,
      'max_links_per_run'     => 3,

      // Topic relevance
      'min_relevance_score'   => 20,

      // Custom tier criteria (optional, free-text)
      'tier1_criteria' => '',
      'tier2_criteria' => '',
      'tier3_criteria' => '',

      // Ignore tiering entirely
      'ignore_tiering' => 0,
    ];

    $saved = get_option(self::OPT_STRATEGY, []);
    if (!is_array($saved)) $saved = [];

    // Migrate old anchor mix defaults (20/30/20/20/10) to new primary-heavy defaults (80/10/5/5/0)
    if (
      isset($saved['anchor_exact']) &&
      (int)$saved['anchor_exact'] === 20 &&
      (int)($saved['anchor_partial'] ?? 30) === 30 &&
      (int)($saved['anchor_descriptive'] ?? 20) === 20 &&
      (int)($saved['anchor_contextual'] ?? 20) === 20 &&
      (int)($saved['anchor_generic'] ?? 10) === 10
    ) {
      $saved['anchor_exact']       = 80;
      $saved['anchor_partial']     = 10;
      $saved['anchor_descriptive'] = 5;
      $saved['anchor_contextual']  = 5;
      $saved['anchor_generic']     = 0;
      update_option(self::OPT_STRATEGY, $saved);
    }

    return array_merge($defaults, $saved);
  }

  /**
   * Render the Strategy settings page
   */
  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $s = self::get_settings();
    $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : '';
    $err = isset($_GET['err']) ? sanitize_text_field($_GET['err']) : '';

    // Get tiering stats
    $stats = self::get_tiering_stats();

    echo '<div class="wrap">';
    echo '<h1>Link Strategy</h1>';

    echo '<p>Configure your internal linking strategy. This determines how links are distributed across page tiers and what anchor text types are used.</p>';

    if ($err) echo '<div class="notice notice-error"><p>' . esc_html($err) . '</p></div>';
    if ($msg) echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';

    // === STRATEGY SETTINGS FORM ===
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
    echo '<input type="hidden" name="action" value="internallinkstool_save_strategy" />';
    wp_nonce_field('internallinkstool_save_strategy');

    // --- Link Budget by Tier ---
    echo '<div id="budget-by-tier-section"' . (!empty($s['ignore_tiering']) ? ' style="opacity:0.4;pointer-events:none;"' : '') . '>';
    echo '<h2>Link Budget by Tier</h2>';
    echo '<p class="description">Percentage of links each tier should receive. Must sum to 100%.</p>';
    echo '<table class="form-table">';

    echo '<tr><th>Homepage</th><td>';
    echo '<input type="number" name="budget_homepage" value="' . esc_attr($s['budget_homepage']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Links to homepage</p></td></tr>';

    echo '<tr><th>Tier 1 (Main Categories)</th><td>';
    echo '<input type="number" name="budget_tier1" value="' . esc_attr($s['budget_tier1']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Short keywords (1-3 words), main navigation pages</p></td></tr>';

    echo '<tr><th>Tier 2 (Subcategories)</th><td>';
    echo '<input type="number" name="budget_tier2" value="' . esc_attr($s['budget_tier2']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Medium keywords (2-4 words), subcategory pages</p></td></tr>';

    echo '<tr><th>Tier 3 (Articles)</th><td>';
    echo '<input type="number" name="budget_tier3" value="' . esc_attr($s['budget_tier3']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Long-tail keywords (3+ words), article/blog pages</p></td></tr>';

    echo '</table>';
    echo '</div>';

    // --- Anchor Type Mix ---
    echo '<h2>Anchor Text Type Mix</h2>';
    echo '<p class="description">Distribution of anchor text types when linking to a target. Must sum to 100%.</p>';
    echo '<table class="form-table">';

    echo '<tr><th>Exact Match</th><td>';
    echo '<input type="number" name="anchor_exact" value="' . esc_attr($s['anchor_exact']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Primary keyword exactly as defined</p></td></tr>';

    echo '<tr><th>Partial / Semantic</th><td>';
    echo '<input type="number" name="anchor_partial" value="' . esc_attr($s['anchor_partial']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Variations and synonyms of the primary keyword</p></td></tr>';

    echo '<tr><th>Descriptive</th><td>';
    echo '<input type="number" name="anchor_descriptive" value="' . esc_attr($s['anchor_descriptive']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Topic expansion phrases (e.g., "learn more about X")</p></td></tr>';

    echo '<tr><th>Contextual</th><td>';
    echo '<input type="number" name="anchor_contextual" value="' . esc_attr($s['anchor_contextual']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">Natural phrases that fit the surrounding content</p></td></tr>';

    echo '<tr><th>Generic Relevant</th><td>';
    echo '<input type="number" name="anchor_generic" value="' . esc_attr($s['anchor_generic']) . '" min="0" max="100" style="width:70px;" /> %';
    echo '<p class="description">General terms like "this guide", "our review"</p></td></tr>';

    echo '</table>';

    // --- Guardrails ---
    echo '<h2>Guardrails</h2>';
    echo '<table class="form-table">';

    echo '<tr><th>Max total links per source page</th><td>';
    echo '<input type="number" name="max_links_per_source" value="' . esc_attr($s['max_links_per_source']) . '" min="1" max="50" style="width:70px;" />';
    echo '<p class="description">Maximum internal links allowed on a single page</p></td></tr>';

    echo '<tr><th>Max new links per source per run</th><td>';
    echo '<input type="number" name="max_links_per_run" value="' . esc_attr($s['max_links_per_run']) . '" min="1" max="20" style="width:70px;" />';
    echo '<p class="description">How many new links to add per page in each Linker run</p></td></tr>';

    echo '</table>';

    // --- Topic Relevance ---
    echo '<h2>Topic Relevance</h2>';
    echo '<p class="description">Ensure source pages link to topically related target pages.</p>';
    echo '<table class="form-table">';

    echo '<tr><th>Minimum relevance score</th><td>';
    echo '<input type="number" name="min_relevance_score" value="' . esc_attr($s['min_relevance_score']) . '" min="0" max="100" style="width:70px;" />';
    echo '<p class="description">0-100. Only create links when source and target are topically related. Higher = stricter filtering.<br>';
    echo '<strong>0</strong> = No filtering (any keyword match)<br>';
    echo '<strong>20</strong> = Basic relevance (recommended)<br>';
    echo '<strong>50+</strong> = Strong topic overlap required</p></td></tr>';

    echo '</table>';

    // === PAGE TIERING CRITERIA (inside form) ===
    echo '<hr>';
    echo '<h2>Page Tiering Criteria</h2>';

    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:14px 18px;margin:10px 0 16px;">';
    echo '<p style="margin:0 0 8px;font-size:13px;"><strong>Page Tiering</strong> classifies every page on your site into 3 importance tiers (plus homepage). Once pages are tiered, you assign each tier a <strong>link budget</strong> &mdash; a percentage that controls how many of the total internal links go to pages in that tier.</p>';
    echo '<p style="margin:0;font-size:13px;">For example: Tier 1 pages (broad, high-value category pages) might get 50% of all links, Tier 2 (subcategory pages) 30%, and Tier 3 (individual articles) 20%. This ensures your most important pages receive the most link equity.</p>';
    echo '</div>';

    echo '<table class="form-table">';
    echo '<tr><th>Ignore page tiering</th><td>';
    echo '<input type="hidden" name="ignore_tiering" value="0" />';
    echo '<label><input type="checkbox" name="ignore_tiering" value="1" ' . checked($s['ignore_tiering'], 1, false) . '> <strong>Ignore page tiering</strong></label>';
    echo '<p class="description">When enabled, the tool will not classify pages into tiers and will ignore all tier-based link budget rules. All pages are treated equally &mdash; links are distributed purely by relevance score.</p>';
    echo '</td></tr>';
    echo '</table>';

    echo '<div id="tiering-criteria-section"' . (!empty($s['ignore_tiering']) ? ' style="opacity:0.4;pointer-events:none;"' : '') . '>';

    echo '<p>You can optionally override the default tier definitions below. Leave blank to keep the defaults.</p>';

    echo '<table class="form-table">';

    echo '<tr><th>Tier 1 criteria</th><td>';
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:8px 12px;margin-bottom:8px;">';
    echo '<strong>Current Tier 1 criteria is:</strong> Primary keyword has 1-2 words, slug has 1-2 words, and URL depth is 1 segment or less (e.g., <code>/shoes/</code>).';
    echo '</div>';
    echo '<textarea rows="3" style="width:520px" name="tier1_criteria">' . esc_textarea($s['tier1_criteria']) . '</textarea>';
    echo '<p class="description">Optional. Override with your own definition (e.g., "Top-level category pages targeting head terms like shoes, dresses, bags").</p>';
    echo '</td></tr>';

    echo '<tr><th>Tier 2 criteria</th><td>';
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:8px 12px;margin-bottom:8px;">';
    echo '<strong>Current Tier 2 criteria is:</strong> Primary keyword has up to 3 words, slug has up to 4 words, and URL depth is up to 2 segments (e.g., <code>/shoes/running-shoes/</code>).';
    echo '</div>';
    echo '<textarea rows="3" style="width:520px" name="tier2_criteria">' . esc_textarea($s['tier2_criteria']) . '</textarea>';
    echo '<p class="description">Optional. Override with your own definition (e.g., "Subcategory pages targeting mid-tail terms like running shoes, evening dresses").</p>';
    echo '</td></tr>';

    echo '<tr><th>Tier 3 criteria</th><td>';
    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:8px 12px;margin-bottom:8px;">';
    echo '<strong>Current Tier 3 criteria is:</strong> Everything else — longer keywords, deeper URLs, blog posts and articles (e.g., <code>/blog/best-running-shoes-2025/</code>).';
    echo '</div>';
    echo '<textarea rows="3" style="width:520px" name="tier3_criteria">' . esc_textarea($s['tier3_criteria']) . '</textarea>';
    echo '<p class="description">Optional. Override with your own definition (e.g., "Blog posts and guides targeting long-tail keywords").</p>';
    echo '</td></tr>';

    echo '</table>';

    echo '<div class="notice notice-warning" style="border-left-color:#dba617;background:#fef8ee;padding:10px 12px;">';
    echo '<p><strong>When do criteria changes take effect?</strong></p>';
    echo '<p style="margin:4px 0;">Changing tier criteria does <strong>not</strong> retroactively re-classify pages that have already been scanned, tiered, and linked. ';
    echo 'New criteria will apply to:</p>';
    echo '<ul style="margin:5px 0 5px 20px;list-style:disc;">';
    echo '<li>Newly published pages processed by automation.</li>';
    echo '<li>All pages — if you click <strong>"Reset All Tiers"</strong> below and then re-run <strong>"Run Auto-Tiering"</strong>.</li>';
    echo '</ul>';
    echo '<p style="margin:4px 0;">To fully re-classify your site: Reset All Tiers → re-run the Scanner → Keywords → Auto-Tiering → Linker.</p>';
    echo '</div>';

    echo '</div>'; // end #tiering-criteria-section

    echo '<script>
      (function() {
        var cb = document.querySelector("input[name=\'ignore_tiering\'][type=\'checkbox\']");
        var sections = [
          document.getElementById("budget-by-tier-section"),
          document.getElementById("tiering-criteria-section")
        ];
        if (cb) {
          cb.addEventListener("change", function() {
            sections.forEach(function(s) {
              if (!s) return;
              s.style.opacity = cb.checked ? "0.4" : "1";
              s.style.pointerEvents = cb.checked ? "none" : "auto";
            });
          });
        }
      })();
    </script>';

    submit_button('Save Strategy Settings');
    echo '</form>';

    // === PAGE TIERING SECTION ===
    echo '<hr>';
    echo '<h2>Page Tiering</h2>';
    echo '<p>Automatically classify your pages into tiers based on keyword length and page type. Run this after the Scanner and Keywords steps.</p>';

    // Show current stats
    echo '<div class="notice notice-info" style="border-left-color:#2271b1;background:#f0f6fc;">';
    echo '<p><strong>Current Tiering Status:</strong></p>';
    echo '<ul style="margin:5px 0 5px 20px;">';
    echo '<li><strong>Homepage:</strong> ' . (int)$stats['homepage'] . ' pages</li>';
    echo '<li><strong>Tier 1 (Main Categories):</strong> ' . (int)$stats['tier1'] . ' pages</li>';
    echo '<li><strong>Tier 2 (Subcategories):</strong> ' . (int)$stats['tier2'] . ' pages</li>';
    echo '<li><strong>Tier 3 (Articles):</strong> ' . (int)$stats['tier3'] . ' pages</li>';
    echo '<li><strong>Unassigned:</strong> ' . (int)$stats['unassigned'] . ' pages</li>';
    echo '</ul>';
    echo '</div>';

    // Run tiering button
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;">';
    echo '<input type="hidden" name="action" value="internallinkstool_run_tiering" />';
    wp_nonce_field('internallinkstool_run_tiering');
    submit_button('Run Auto-Tiering', 'primary', 'submit', false);
    echo '</form>';

    // Reset tiering button
    echo ' <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;margin-left:10px;">';
    echo '<input type="hidden" name="action" value="internallinkstool_reset_tiering" />';
    wp_nonce_field('internallinkstool_reset_tiering');
    submit_button('Reset All Tiers', 'secondary', 'submit', false);
    echo '</form>';

    // === TIERED PAGES TABLE ===
    echo '<hr>';
    echo '<h2>Tiered Pages</h2>';

    $tp_per_page = isset($_GET['tp_per_page']) ? max(1, min(500, (int)$_GET['tp_per_page'])) : 50;
    $tp_paged    = isset($_GET['tp_paged']) ? max(1, (int)$_GET['tp_paged']) : 1;
    self::render_tiered_pages_table($tp_per_page, $tp_paged);

    echo '</div>';
  }

  /**
   * Render table of tiered pages with pagination
   */
  private static function render_tiered_pages_table($per_page = 50, $paged = 1) {
    if (!class_exists('InternalLinksTool_DB')) {
      echo '<p>Database class not available.</p>';
      return;
    }

    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    // Check if tier column exists
    $col_exists = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if (!$col_exists) {
      echo '<p><em>Tier column not yet created. Run tiering first.</em></p>';
      return;
    }

    // Count total tiered pages
    $total = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs} WHERE tier IS NOT NULL AND tier != ''");

    if ($total === 0) {
      echo '<p><em>No tiered pages yet. Run auto-tiering above.</em></p>';
      return;
    }

    $total_pages = max(1, (int)ceil($total / $per_page));
    if ($paged > $total_pages) $paged = $total_pages;
    $offset = ($paged - 1) * $per_page;

    $base_url = admin_url('admin.php?page=internallinkstool-strategy');

    // Per-page form
    echo '<form method="get" action="' . esc_url($base_url) . '" style="margin:8px 0;">';
    echo '<input type="hidden" name="page" value="internallinkstool-strategy" />';
    echo '<label>Show <input type="number" name="tp_per_page" value="' . esc_attr($per_page) . '" min="1" max="500" style="width:70px;" /> pages per page</label> ';
    echo '<input type="submit" class="button" value="Apply" />';
    echo '<span class="description" style="margin-left:12px;">Showing ' . esc_html($total) . ' tiered pages total. Page ' . esc_html($paged) . ' of ' . esc_html($total_pages) . '.</span>';
    echo '</form>';

    $sql = $wpdb->prepare(
      "SELECT d.id, d.post_id, d.url, d.meta_title, d.tier,
              k.primary_keyword,
              pm.meta_value as yoast_focus_kw
       FROM {$docs} d
       LEFT JOIN {$keywords} k ON k.document_id = d.id
       LEFT JOIN {$wpdb->postmeta} pm ON pm.post_id = d.post_id AND pm.meta_key = '_yoast_wpseo_focuskw'
       WHERE d.tier IS NOT NULL AND d.tier != ''
       ORDER BY
         CASE d.tier
           WHEN 'homepage' THEN 1
           WHEN 'tier1' THEN 2
           WHEN 'tier2' THEN 3
           WHEN 'tier3' THEN 4
           ELSE 5
         END,
         d.id ASC
       LIMIT %d OFFSET %d",
      $per_page, $offset
    );

    $rows = $wpdb->get_results($sql, ARRAY_A);

    $tier_options = ['homepage' => 'Homepage', 'tier1' => 'Tier 1', 'tier2' => 'Tier 2', 'tier3' => 'Tier 3'];
    $nonce = wp_create_nonce('internallinkstool_update_tier');

    echo '<table class="widefat fixed striped">';
    echo '<thead><tr>';
    echo '<th style="width:5%;">ID</th>';
    echo '<th style="width:25%;">URL</th>';
    echo '<th style="width:18%;">Meta Title</th>';
    echo '<th style="width:15%;">Yoast Focus KW</th>';
    echo '<th style="width:15%;">Primary Keyword</th>';
    echo '<th style="width:15%;">Tier</th>';
    echo '</tr></thead><tbody>';

    if (empty($rows)) {
      echo '<tr><td colspan="6"><em>No results on this page.</em></td></tr>';
    } else {
      foreach ($rows as $r) {
        $doc_id = (int)$r['id'];
        $tier = $r['tier'] ?? '';

        echo '<tr>';
        echo '<td>' . (int)$r['post_id'] . '</td>';
        echo '<td><a href="' . esc_url($r['url']) . '" target="_blank">' . esc_html($r['url']) . '</a></td>';
        echo '<td>' . esc_html($r['meta_title'] ?? '') . '</td>';
        echo '<td><code>' . esc_html($r['yoast_focus_kw'] ?? '-') . '</code></td>';
        echo '<td><code>' . esc_html($r['primary_keyword'] ?? '-') . '</code></td>';
        echo '<td>';
        echo '<select class="ilt-tier-select" data-doc-id="' . esc_attr($doc_id) . '" style="width:100px;">';
        foreach ($tier_options as $val => $label) {
          echo '<option value="' . esc_attr($val) . '" ' . selected($tier, $val, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<span class="ilt-tier-status" data-doc-id="' . esc_attr($doc_id) . '" style="margin-left:6px;font-size:11px;"></span>';
        echo '</td>';
        echo '</tr>';
      }
    }

    echo '</tbody></table>';

    // Prev / Next navigation
    if ($total_pages > 1) {
      echo '<div style="margin:10px 0;">';
      if ($paged > 1) {
        $prev_url = add_query_arg(['tp_paged' => $paged - 1, 'tp_per_page' => $per_page], $base_url);
        echo '<a class="button" href="' . esc_url($prev_url) . '">&laquo; Previous</a> ';
      }
      if ($paged < $total_pages) {
        $next_url = add_query_arg(['tp_paged' => $paged + 1, 'tp_per_page' => $per_page], $base_url);
        echo '<a class="button" href="' . esc_url($next_url) . '">Next &raquo;</a>';
      }
      echo '</div>';
    }

    // AJAX for tier change
    ?>
    <script type="text/javascript">
    (function() {
      var nonce = <?php echo wp_json_encode($nonce); ?>;
      document.querySelectorAll('.ilt-tier-select').forEach(function(sel) {
        sel.addEventListener('change', function() {
          var docId = this.getAttribute('data-doc-id');
          var newTier = this.value;
          var status = document.querySelector('.ilt-tier-status[data-doc-id="' + docId + '"]');
          status.textContent = 'Saving...';
          status.style.color = '#666';

          var fd = new FormData();
          fd.append('action', 'internallinkstool_update_tier');
          fd.append('doc_id', docId);
          fd.append('tier', newTier);
          fd.append('_ajax_nonce', nonce);

          fetch(ajaxurl, { method: 'POST', body: fd })
            .then(function(r) { return r.json(); })
            .then(function(data) {
              if (data.success) {
                status.textContent = 'Saved';
                status.style.color = '#2e7d32';
              } else {
                status.textContent = 'Error';
                status.style.color = '#d63638';
              }
              setTimeout(function() { status.textContent = ''; }, 2000);
            })
            .catch(function() {
              status.textContent = 'Failed';
              status.style.color = '#d63638';
              setTimeout(function() { status.textContent = ''; }, 2000);
            });
        });
      });
    })();
    </script>
    <?php
  }

  private static function get_tier_label($tier) {
    $labels = [
      'homepage' => 'Homepage',
      'tier1' => 'Tier 1',
      'tier2' => 'Tier 2',
      'tier3' => 'Tier 3',
    ];
    return $labels[$tier] ?? $tier;
  }

  private static function get_tier_color($tier) {
    $colors = [
      'homepage' => '#9c27b0',
      'tier1' => '#1976d2',
      'tier2' => '#388e3c',
      'tier3' => '#f57c00',
    ];
    return $colors[$tier] ?? '#666';
  }

  /**
   * Get tiering statistics
   */
  public static function get_tiering_stats() {
    $stats = [
      'homepage' => 0,
      'tier1' => 0,
      'tier2' => 0,
      'tier3' => 0,
      'unassigned' => 0,
    ];

    if (!class_exists('InternalLinksTool_DB')) return $stats;

    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');

    // Check if tier column exists
    $col_exists = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if (!$col_exists) {
      $stats['unassigned'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs}");
      return $stats;
    }

    $stats['homepage'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs} WHERE tier = 'homepage'");
    $stats['tier1'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs} WHERE tier = 'tier1'");
    $stats['tier2'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs} WHERE tier = 'tier2'");
    $stats['tier3'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs} WHERE tier = 'tier3'");
    $stats['unassigned'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs} WHERE tier IS NULL OR tier = ''");

    return $stats;
  }

  /**
   * Handle save strategy settings
   */
  public static function handle_save_strategy() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_save_strategy')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    $s = [
      'budget_homepage' => isset($_POST['budget_homepage']) ? (int)$_POST['budget_homepage'] : 10,
      'budget_tier1'    => isset($_POST['budget_tier1']) ? (int)$_POST['budget_tier1'] : 50,
      'budget_tier2'    => isset($_POST['budget_tier2']) ? (int)$_POST['budget_tier2'] : 30,
      'budget_tier3'    => isset($_POST['budget_tier3']) ? (int)$_POST['budget_tier3'] : 10,

      'anchor_exact'       => isset($_POST['anchor_exact']) ? (int)$_POST['anchor_exact'] : 80,
      'anchor_partial'     => isset($_POST['anchor_partial']) ? (int)$_POST['anchor_partial'] : 10,
      'anchor_descriptive' => isset($_POST['anchor_descriptive']) ? (int)$_POST['anchor_descriptive'] : 5,
      'anchor_contextual'  => isset($_POST['anchor_contextual']) ? (int)$_POST['anchor_contextual'] : 5,
      'anchor_generic'     => isset($_POST['anchor_generic']) ? (int)$_POST['anchor_generic'] : 0,

      'max_links_per_source'  => isset($_POST['max_links_per_source']) ? (int)$_POST['max_links_per_source'] : 5,
      'max_links_per_run'     => isset($_POST['max_links_per_run']) ? (int)$_POST['max_links_per_run'] : 3,

      'min_relevance_score'   => isset($_POST['min_relevance_score']) ? max(0, min(100, (int)$_POST['min_relevance_score'])) : 20,

      'tier1_criteria' => isset($_POST['tier1_criteria']) ? sanitize_textarea_field($_POST['tier1_criteria']) : '',
      'tier2_criteria' => isset($_POST['tier2_criteria']) ? sanitize_textarea_field($_POST['tier2_criteria']) : '',
      'tier3_criteria' => isset($_POST['tier3_criteria']) ? sanitize_textarea_field($_POST['tier3_criteria']) : '',

      'ignore_tiering' => (int)!empty($_POST['ignore_tiering']),
    ];

    // Validate budget sums
    $budget_sum = $s['budget_homepage'] + $s['budget_tier1'] + $s['budget_tier2'] + $s['budget_tier3'];
    $anchor_sum = $s['anchor_exact'] + $s['anchor_partial'] + $s['anchor_descriptive'] + $s['anchor_contextual'] + $s['anchor_generic'];

    if (empty($s['ignore_tiering']) && $budget_sum !== 100) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&err=' . rawurlencode('Link budget must sum to 100%. Currently: ' . $budget_sum . '%')));
      exit;
    }

    if ($anchor_sum !== 100) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&err=' . rawurlencode('Anchor type mix must sum to 100%. Currently: ' . $anchor_sum . '%')));
      exit;
    }

    update_option(self::OPT_STRATEGY, $s, false);

    wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&msg=' . rawurlencode('Strategy settings saved.')));
    exit;
  }

  /**
   * Handle run tiering
   */
  public static function handle_run_tiering() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_run_tiering')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    $result = self::run_auto_tiering();

    wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&msg=' . rawurlencode('Tiering complete. ' . $result['message'])));
    exit;
  }

  /**
   * Handle reset tiering
   */
  public static function handle_reset_tiering() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_reset_tiering')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    if (!class_exists('InternalLinksTool_DB')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&err=' . rawurlencode('DB class not available.')));
      exit;
    }

    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');

    // Check if tier column exists
    $col_exists = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if ($col_exists) {
      $wpdb->query("UPDATE {$docs} SET tier = NULL, page_type = NULL");
    }

    wp_redirect(admin_url('admin.php?page=internallinkstool-strategy&msg=' . rawurlencode('All tiers have been reset.')));
    exit;
  }

  /**
   * AJAX handler for manual tier update
   */
  public static function ajax_update_tier() {
    check_ajax_referer('internallinkstool_update_tier');
    if (!current_user_can('manage_options')) {
      wp_send_json_error('No permission');
    }

    $doc_id = isset($_POST['doc_id']) ? (int)$_POST['doc_id'] : 0;
    $tier = isset($_POST['tier']) ? sanitize_key($_POST['tier']) : '';

    if ($doc_id <= 0 || !in_array($tier, ['homepage', 'tier1', 'tier2', 'tier3'], true)) {
      wp_send_json_error('Invalid parameters');
    }

    if (!class_exists('InternalLinksTool_DB')) {
      wp_send_json_error('DB not available');
    }

    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');
    $wpdb->update($docs, ['tier' => $tier], ['id' => $doc_id]);

    wp_send_json_success(['doc_id' => $doc_id, 'tier' => $tier]);
  }

  /**
   * Run auto-tiering logic
   */
  public static function run_auto_tiering() {
    if (!class_exists('InternalLinksTool_DB')) {
      return ['success' => false, 'message' => 'DB class not available.'];
    }

    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    // Ensure tier and page_type columns exist
    self::ensure_tier_columns();

    // Get site URL for homepage detection
    $site_url = trailingslashit(home_url());

    // Get all documents with their keywords
    $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, k.primary_keyword
            FROM {$docs} d
            LEFT JOIN {$keywords} k ON k.document_id = d.id";
    $rows = $wpdb->get_results($sql, ARRAY_A);

    if (empty($rows)) {
      return ['success' => false, 'message' => 'No documents found. Run Scanner first.'];
    }

    $counts = ['homepage' => 0, 'tier1' => 0, 'tier2' => 0, 'tier3' => 0];

    foreach ($rows as $r) {
      $doc_id = (int)$r['id'];
      $url = trim((string)($r['url'] ?? ''));
      $primary_kw = trim((string)($r['primary_keyword'] ?? ''));

      // Determine tier
      $tier = self::determine_tier($url, $primary_kw, $site_url);
      $page_type = self::determine_page_type($url, $primary_kw, $site_url);

      // Update document
      $wpdb->update(
        $docs,
        ['tier' => $tier, 'page_type' => $page_type],
        ['id' => $doc_id]
      );

      if (isset($counts[$tier])) {
        $counts[$tier]++;
      }
    }

    $message = sprintf(
      'Homepage: %d, Tier 1: %d, Tier 2: %d, Tier 3: %d',
      $counts['homepage'], $counts['tier1'], $counts['tier2'], $counts['tier3']
    );

    return ['success' => true, 'message' => $message];
  }

  /**
   * Determine tier for a page
   */
  private static function determine_tier($url, $primary_keyword, $site_url) {
    // Homepage check
    $url_normalized = trailingslashit(strtolower($url));
    $site_normalized = trailingslashit(strtolower($site_url));
    if ($url_normalized === $site_normalized) {
      return 'homepage';
    }

    // Count words in primary keyword
    $kw_words = 0;
    if ($primary_keyword !== '') {
      $kw_words = count(preg_split('/\s+/', trim($primary_keyword)));
    }

    // Count slug segments
    $path = wp_parse_url($url, PHP_URL_PATH);
    $path = trim($path, '/');
    $segments = $path !== '' ? explode('/', $path) : [];
    $segment_count = count($segments);

    // Get last slug
    $slug = end($segments) ?: '';
    $slug_words = 0;
    if ($slug !== '') {
      $slug_clean = str_replace(['-', '_'], ' ', $slug);
      $slug_words = count(preg_split('/\s+/', trim($slug_clean)));
    }

    // Tiering logic:
    // Tier 1: Short keyword (1-2 words) OR short slug (1-2 words) OR shallow depth (1 segment)
    // Tier 2: Medium keyword (2-3 words) OR medium slug (2-4 words) OR medium depth (2 segments)
    // Tier 3: Long keyword (4+ words) OR long slug (4+ words) OR deep (3+ segments)

    if ($kw_words <= 2 && $slug_words <= 2 && $segment_count <= 1) {
      return 'tier1';
    }

    if ($kw_words <= 3 && $slug_words <= 4 && $segment_count <= 2) {
      return 'tier2';
    }

    return 'tier3';
  }

  /**
   * Determine page type
   */
  private static function determine_page_type($url, $primary_keyword, $site_url) {
    $url_normalized = trailingslashit(strtolower($url));
    $site_normalized = trailingslashit(strtolower($site_url));

    if ($url_normalized === $site_normalized) {
      return 'homepage';
    }

    $path = wp_parse_url($url, PHP_URL_PATH);
    $path = trim($path, '/');
    $segments = $path !== '' ? explode('/', $path) : [];
    $segment_count = count($segments);

    // Simple heuristics for page type
    if ($segment_count === 1) {
      return 'category';
    }

    if ($segment_count === 2) {
      return 'subcategory';
    }

    // Check for blog/article indicators
    $slug = end($segments) ?: '';
    $slug_lower = strtolower($slug);

    // Long slugs or containing certain patterns suggest articles
    if (str_contains($path, 'blog') || str_contains($path, 'article') || str_contains($path, 'post')) {
      return 'article';
    }

    if ($segment_count >= 3) {
      return 'article';
    }

    return 'page';
  }

  /**
   * Ensure tier columns exist in documents table
   */
  private static function ensure_tier_columns() {
    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');

    // Add tier column if not exists
    $col_tier = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if (!$col_tier) {
      $wpdb->query("ALTER TABLE {$docs} ADD COLUMN tier VARCHAR(20) NULL");
    }

    // Add page_type column if not exists
    $col_type = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'page_type'");
    if (!$col_type) {
      $wpdb->query("ALTER TABLE {$docs} ADD COLUMN page_type VARCHAR(50) NULL");
    }
  }

  /* =====================================================
   * ANCHOR BANKS METHODS
   * ===================================================== */

  /**
   * Get anchor bank statistics
   */
  public static function get_anchor_bank_stats() {
    $stats = [
      'pages_with_banks' => 0,
      'total_anchors' => 0,
      'pages_needing_banks' => 0,
    ];

    if (!class_exists('InternalLinksTool_DB')) return $stats;

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');
    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    // Check if anchor_banks table exists
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) {
      // Count pages with keywords as needing banks
      $stats['pages_needing_banks'] = (int)$wpdb->get_var(
        "SELECT COUNT(DISTINCT k.document_id) FROM {$keywords} k WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''"
      );
      return $stats;
    }

    $stats['pages_with_banks'] = (int)$wpdb->get_var("SELECT COUNT(DISTINCT document_id) FROM {$banks}");
    $stats['total_anchors'] = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$banks}");

    // Pages with keywords but no anchor banks
    $stats['pages_needing_banks'] = (int)$wpdb->get_var(
      "SELECT COUNT(DISTINCT k.document_id)
       FROM {$keywords} k
       LEFT JOIN {$banks} b ON b.document_id = k.document_id
       WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
         AND b.id IS NULL"
    );

    return $stats;
  }

  /**
   * Render anchor banks table
   */
  public static function render_anchor_banks_table() {
    if (!class_exists('InternalLinksTool_DB')) {
      echo '<p>Database class not available.</p>';
      return;
    }

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');
    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    // Check if table exists
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) {
      echo '<p><em>Anchor banks table not yet created. Generate anchor banks first.</em></p>';
      return;
    }

    // Get sample of anchor banks grouped by document
    $sql = "SELECT b.document_id, d.url, d.post_id, k.primary_keyword,
                   pm.meta_value as yoast_focus_kw,
                   GROUP_CONCAT(CASE WHEN b.anchor_type = 'exact' THEN b.anchor_text END SEPARATOR ', ') as exact_anchors,
                   GROUP_CONCAT(CASE WHEN b.anchor_type = 'partial' THEN b.anchor_text END SEPARATOR ', ') as partial_anchors,
                   GROUP_CONCAT(CASE WHEN b.anchor_type = 'descriptive' THEN b.anchor_text END SEPARATOR ', ') as descriptive_anchors,
                   COUNT(*) as total_anchors
            FROM {$banks} b
            INNER JOIN {$docs} d ON d.id = b.document_id
            LEFT JOIN {$keywords} k ON k.document_id = b.document_id
            LEFT JOIN {$wpdb->postmeta} pm ON pm.post_id = d.post_id AND pm.meta_key = '_yoast_wpseo_focuskw'
            GROUP BY b.document_id, d.url, d.post_id, k.primary_keyword, pm.meta_value
            ORDER BY b.document_id DESC
            LIMIT 20";

    $rows = $wpdb->get_results($sql, ARRAY_A);

    if (empty($rows)) {
      echo '<p><em>No anchor banks generated yet.</em></p>';
      return;
    }

    echo '<p class="description">Showing up to 20 pages with anchor banks.</p>';
    echo '<table class="widefat fixed striped">';
    echo '<thead><tr>';
    echo '<th style="width:18%;">URL</th>';
    echo '<th style="width:10%;">Yoast Focus KW</th>';
    echo '<th style="width:10%;">Primary KW</th>';
    echo '<th style="width:18%;">Exact Anchors</th>';
    echo '<th style="width:22%;">Partial/Semantic</th>';
    echo '<th style="width:17%;">Descriptive</th>';
    echo '<th style="width:5%;">Total</th>';
    echo '</tr></thead><tbody>';

    foreach ($rows as $r) {
      echo '<tr>';
      echo '<td><a href="' . esc_url($r['url']) . '" target="_blank">' . esc_html($r['url']) . '</a></td>';
      echo '<td><code>' . esc_html($r['yoast_focus_kw'] ?? '-') . '</code></td>';
      echo '<td><code>' . esc_html($r['primary_keyword'] ?? '-') . '</code></td>';
      echo '<td style="font-size:11px;">' . esc_html($r['exact_anchors'] ?? '') . '</td>';
      echo '<td style="font-size:11px;">' . esc_html($r['partial_anchors'] ?? '') . '</td>';
      echo '<td style="font-size:11px;">' . esc_html($r['descriptive_anchors'] ?? '') . '</td>';
      echo '<td>' . (int)$r['total_anchors'] . '</td>';
      echo '</tr>';
    }

    echo '</tbody></table>';
  }

  /**
   * Generate anchor bank for a single page
   *
   * @param int $doc_id Document ID
   * @param string $primary_kw Primary keyword
   * @param string $secondary_kw Secondary keywords (comma-separated)
   * @param string $meta_title Meta title
   * @param string $meta_desc Meta description
   * @return int Number of anchors generated (0 on error)
   */
  public static function generate_anchor_bank_for_page($doc_id, $primary_kw, $secondary_kw, $meta_title, $meta_desc) {
    if (!class_exists('InternalLinksTool_DB')) return 0;
    if (!class_exists('InternalLinksTool_OpenAI') || !method_exists('InternalLinksTool_OpenAI', 'generate_anchor_bank')) return 0;

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');

    // Ensure table exists
    self::ensure_anchor_banks_table();

    try {
      $result = InternalLinksTool_OpenAI::generate_anchor_bank($primary_kw, $secondary_kw, $meta_title, $meta_desc);

      if (!empty($result['error'])) {
        return 0;
      }

      $count = 0;
      $anchor_types = ['exact', 'partial', 'descriptive', 'contextual', 'generic'];
      foreach ($anchor_types as $type) {
        $anchors = isset($result[$type]) && is_array($result[$type]) ? $result[$type] : [];
        foreach ($anchors as $anchor) {
          $anchor = trim((string)$anchor);
          if ($anchor === '') continue;

          $wpdb->insert($banks, [
            'document_id' => (int)$doc_id,
            'anchor_type' => $type,
            'anchor_text' => $anchor,
            'used_count' => 0,
            'created_at' => current_time('mysql'),
          ]);
          $count++;
        }
      }

      return $count;
    } catch (Exception $e) {
      return 0;
    }
  }

  /**
   * Ensure anchor_banks table exists
   */
  private static function ensure_anchor_banks_table() {
    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) {
      $charset_collate = $wpdb->get_charset_collate();
      $wpdb->query("CREATE TABLE {$banks} (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        document_id BIGINT(20) UNSIGNED NOT NULL,
        anchor_type VARCHAR(30) NOT NULL,
        anchor_text VARCHAR(255) NOT NULL,
        used_count INT UNSIGNED NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY document_id_idx (document_id),
        KEY anchor_type_idx (anchor_type),
        UNIQUE KEY doc_type_text (document_id, anchor_type, anchor_text(100))
      ) {$charset_collate};");
    }
  }

  /**
   * Get an anchor from the bank for a document
   * Used by the Linker to pick anchors deterministically
   *
   * @param int $document_id Target document ID
   * @param string $anchor_type Type: exact, partial, descriptive, contextual, generic
   * @param array $exclude_anchors Anchors already used (to avoid duplicates)
   * @return string|null Anchor text or null if none available
   */
  public static function get_anchor_from_bank($document_id, $anchor_type, $exclude_anchors = []) {
    if (!class_exists('InternalLinksTool_DB')) return null;

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) return null;

    $exclude_sql = '';
    if (!empty($exclude_anchors)) {
      $placeholders = implode(',', array_fill(0, count($exclude_anchors), '%s'));
      $exclude_sql = $wpdb->prepare(" AND anchor_text NOT IN ({$placeholders})", $exclude_anchors);
    }

    // Get least-used anchor of this type
    $sql = "SELECT anchor_text FROM {$banks}
            WHERE document_id = %d AND anchor_type = %s
            {$exclude_sql}
            ORDER BY used_count ASC, RAND()
            LIMIT 1";

    $anchor = $wpdb->get_var($wpdb->prepare($sql, $document_id, $anchor_type));

    if ($anchor) {
      // Increment used count
      $wpdb->query($wpdb->prepare(
        "UPDATE {$banks} SET used_count = used_count + 1 WHERE document_id = %d AND anchor_type = %s AND anchor_text = %s",
        $document_id, $anchor_type, $anchor
      ));
    }

    return $anchor;
  }

  /**
   * Get all anchors for a document from the bank
   *
   * @param int $document_id Target document ID
   * @return array Anchors grouped by type
   */
  public static function get_all_anchors_for_document($document_id) {
    $result = [
      'exact' => [],
      'partial' => [],
      'descriptive' => [],
      'contextual' => [],
      'generic' => [],
    ];

    if (!class_exists('InternalLinksTool_DB')) return $result;

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) return $result;

    $rows = $wpdb->get_results($wpdb->prepare(
      "SELECT anchor_type, anchor_text FROM {$banks} WHERE document_id = %d ORDER BY used_count ASC",
      $document_id
    ), ARRAY_A);

    foreach ($rows as $r) {
      $type = $r['anchor_type'];
      if (isset($result[$type])) {
        $result[$type][] = $r['anchor_text'];
      }
    }

    return $result;
  }
}
