<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_Meta {

  public static function extract($post_id) {
    $meta_title  = class_exists('InternalLinksTool_Meta_Yoast') ? InternalLinksTool_Meta_Yoast::get_meta_title($post_id) : '';
    $meta_desc   = class_exists('InternalLinksTool_Meta_Yoast') ? InternalLinksTool_Meta_Yoast::get_meta_description($post_id) : '';
    $meta_robots = class_exists('InternalLinksTool_Meta_Yoast') ? InternalLinksTool_Meta_Yoast::get_meta_robots($post_id) : '';

    $post = get_post($post_id);
    $h1 = '';
    $content = '';
    $post_title = '';

    if ($post) {
      $post_title = is_string($post->post_title) ? trim($post->post_title) : '';
      $content = is_string($post->post_content) ? $post->post_content : '';
      if ($content !== '') {
        $h1 = self::extract_first_h1($content);
      }
    }

    // Fallbacks if Yoast meta missing
    if ($meta_title === '' && $post_title !== '') {
      $meta_title = $post_title;
    }

    if ($meta_desc === '' && $content !== '') {
      $meta_desc = self::content_snippet($content, 180);
    }

    return [
      'meta_title'        => $meta_title,
      'meta_desc'         => $meta_desc,         // ✅ preferred DB field name
      'meta_description'  => $meta_desc,         // ✅ back-compat (some code used this)
      'meta_robots'       => $meta_robots,
      'h1'                => $h1,
      'post_title'        => $post_title,        // helpful for keyword module
      'content_snippet'   => ($content !== '' ? self::content_snippet($content, 240) : ''),
    ];
  }

  private static function extract_first_h1($html) {
    $html = (string)$html;
    if ($html === '') return '';

    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    $wrapped = '<!DOCTYPE html><html><body>' . $html . '</body></html>';
    $ok = $dom->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();
    if (!$ok) return '';

    $h1s = $dom->getElementsByTagName('h1');
    if ($h1s && $h1s->length > 0) {
      return trim($h1s->item(0)->textContent);
    }
    return '';
  }

  private static function content_snippet($html, $max_len = 180) {
    $txt = wp_strip_all_tags((string)$html, true);
    $txt = preg_replace('/\s+/u', ' ', $txt);
    $txt = trim((string)$txt);
    if ($txt === '') return '';
    if (mb_strlen($txt) <= $max_len) return $txt;
    return mb_substr($txt, 0, $max_len) . '…';
  }
}
