<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_Linker {

  private static $ptr_key = 'internallinkstool_linker_ptr';

  private static $bg_key = 'internallinkstool_bg_linker';

  public static function init() {
    add_action('admin_post_internallinkstool_run_linker', [__CLASS__, 'handle_run_linker']);
    add_action('admin_post_internallinkstool_reset_linker_ptr', [__CLASS__, 'handle_reset_ptr']);

    // AJAX handler for redo anchor
    add_action('wp_ajax_internallinkstool_redo_anchor', [__CLASS__, 'ajax_redo_anchor']);

    // Background Run All
    add_action('admin_post_internallinkstool_bg_start', [__CLASS__, 'handle_bg_start']);
    add_action('admin_post_internallinkstool_bg_stop', [__CLASS__, 'handle_bg_stop']);
    add_action('wp_ajax_internallinkstool_bg_status', [__CLASS__, 'ajax_bg_status']);
    add_action('internallinkstool_bg_linker_batch', [__CLASS__, 'process_bg_batch']);
  }

  /* -------------------------
   * Eligibility helpers (match Scanner/Keywords)
   * ------------------------- */
  private static function get_allowed_statuses($settings) {
    if (!empty($settings['scan_statuses']) && is_array($settings['scan_statuses'])) {
      $statuses = array_values(array_unique(array_map('sanitize_key', $settings['scan_statuses'])));
      $statuses = array_values(array_intersect($statuses, ['publish','draft']));
      if (!empty($statuses)) return $statuses;
    }

    $legacy = get_option('internallinkstool_scan_statuses', null);
    if (is_array($legacy) && !empty($legacy)) {
      $legacy = array_values(array_unique(array_map('sanitize_key', $legacy)));
      $legacy = array_values(array_intersect($legacy, ['publish','draft']));
      if (!empty($legacy)) return $legacy;
    }

    return ['publish'];
  }

  private static function get_allowed_types($settings) {
    $types = [];
    if (!empty($settings['include_posts'])) $types[] = 'post';
    if (!empty($settings['include_pages'])) $types[] = 'page';
    if (empty($types)) $types = ['post','page'];

    $types = array_values(array_unique(array_map('sanitize_key', $types)));
    $types = array_values(array_intersect($types, ['post','page']));
    return !empty($types) ? $types : ['post','page'];
  }

  private static function get_allowed_target_types($settings, $source_type) {
    $all_allowed = self::get_allowed_types($settings);
    $key = ($source_type === 'page') ? 'page_link_targets' : 'post_link_targets';
    $val = $settings[$key] ?? 'all';
    if ($val === 'none') return [];
    if ($val === 'all') return $all_allowed;
    return in_array($val, $all_allowed, true) ? [$val] : [];
  }

  private static function in_placeholders($arr) {
    $arr = is_array($arr) ? array_values($arr) : [];
    if (empty($arr)) return ['()', []];
    return ['(' . implode(',', array_fill(0, count($arr), '%s')) . ')', $arr];
  }

  private static function get_tables() {
    global $wpdb;
    return [
      'docs' => $wpdb->prefix . 'internallinkstool_documents',
      'kws'  => $wpdb->prefix . 'internallinkstool_keywords',
    ];
  }

  /* -------------------------
   * Admin UI
   * ------------------------- */
  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $batch_size = isset($_GET['batch_size']) ? (int)$_GET['batch_size'] : 10;
    $batch_size = max(1, min(100, $batch_size));

    $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : '';
    $err = isset($_GET['err']) ? sanitize_text_field($_GET['err']) : '';

    $preview_key = isset($_GET['preview_key']) ? sanitize_text_field($_GET['preview_key']) : '';
    $preview = [];
    $debug_rows = [];
    $is_bg_preview = false;

    if ($preview_key) {
      $preview = get_transient('internallinkstool_linker_preview_' . $preview_key);
      if (!is_array($preview)) $preview = [];

      $debug_rows = get_transient('internallinkstool_linker_debug_' . $preview_key);
      if (!is_array($debug_rows)) $debug_rows = [];
    }

    // Load accumulated preview from completed background run
    if (empty($preview) && empty($debug_rows)) {
      $bg_check = get_option(self::$bg_key, []);
      if (($bg_check['status'] ?? '') === 'done') {
        $bg_preview = get_transient('internallinkstool_bg_linker_preview');
        if (is_array($bg_preview) && !empty($bg_preview)) { $preview = $bg_preview; $is_bg_preview = true; }
        $bg_debug = get_transient('internallinkstool_bg_linker_debug');
        if (is_array($bg_debug) && !empty($bg_debug)) { $debug_rows = $bg_debug; $is_bg_preview = true; }
      }
    }

    $settings = class_exists('InternalLinksTool_Admin')
      ? InternalLinksTool_Admin::get_settings()
      : [
        'max_links_per_page'=>3,'max_links_per_target'=>1,'skip_sentences'=>2,'spread_mode'=>'spread',
        'include_posts'=>1,'include_pages'=>1,'same_category_only'=>0,'exclude_slug_words'=>'','exclude_urls'=>'','respect_robots'=>1,
        'scan_statuses'=>['publish']
      ];

    $statuses = self::get_allowed_statuses($settings);
    $types    = self::get_allowed_types($settings);

    $progress = self::get_progress_counts($settings);

    echo '<div class="wrap"><h1>Linker</h1>';

    echo '<p>The Linker inserts internal links into your content by matching keywords from target pages. ';
    echo 'Use <strong>Dry Run</strong> first to preview proposed links without modifying your posts. ';
    echo 'Results and linked pages appear in the tables below after each run.</p>';

    if ($err) echo '<div class="notice notice-error"><p>' . esc_html($err) . '</p></div>';
    if ($msg) echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';

    echo '<div class="notice notice-info"><p><strong>Dry Run note:</strong> Dry run does NOT modify posts. Preview + debug tables appear below after a run.</p></div>';

    // Build display labels
    $types_list = [];
    if (in_array('post', $types)) $types_list[] = 'Posts';
    if (in_array('page', $types)) $types_list[] = 'Pages';
    if (empty($types_list)) $types_list = ['Posts', 'Pages'];

    $statuses_list = [];
    if (in_array('publish', $statuses)) $statuses_list[] = 'Published';
    if (in_array('draft', $statuses)) $statuses_list[] = 'Draft';
    if (empty($statuses_list)) $statuses_list = ['Published'];

    $respect_robots = !empty($settings['respect_robots']) ? 'Yes (excluding noindex/nofollow)' : 'No';

    // Get Strategy settings
    $strategy = class_exists('InternalLinksTool_Strategy')
      ? InternalLinksTool_Strategy::get_settings()
      : [];

    // Current linker settings summary
    echo '<div class="notice notice-warning" style="border-left-color:#2271b1;background:#f0f6fc;">';
    echo '<p><strong>Current Linker Settings:</strong></p>';
    echo '<ul style="margin:5px 0 5px 20px;">';
    echo '<li><strong>Post Types:</strong> ' . esc_html(implode(', ', $types_list)) . '</li>';
    echo '<li><strong>Statuses:</strong> ' . esc_html(implode(', ', $statuses_list)) . '</li>';
    echo '<li><strong>Respect Robots:</strong> ' . esc_html($respect_robots) . '</li>';
    echo '<li><strong>Max Links/Page:</strong> ' . (int)($settings['max_links_per_page'] ?? 3) . '</li>';
    echo '<li><strong>Max Links/Target:</strong> ' . (int)($settings['max_links_per_target'] ?? 1) . '</li>';
    echo '<li><strong>Skip Sentences:</strong> ' . (int)($settings['skip_sentences'] ?? 2) . '</li>';
    echo '<li><strong>Spread Mode:</strong> ' . esc_html((string)($settings['spread_mode'] ?? 'spread')) . '</li>';
    if (!empty($settings['anchor_rewriting'])) {
      echo '<li><strong>AI Anchor Rewriting:</strong> <span style="background:#e7f5e7;color:#2e7d32;padding:2px 6px;">ON</span></li>';
    }
    echo '</ul>';
    echo '<p class="description" style="margin-top:5px;">Change these in <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-settings')) . '">Link Settings</a>.</p>';
    echo '</div>';

    // Strategy settings summary
    echo '<div class="notice notice-info" style="border-left-color:#9c27b0;background:#f9f0fc;">';
    echo '<p><strong>Strategy Settings (from Strategy page):</strong></p>';
    echo '<ul style="margin:5px 0 5px 20px;">';
    echo '<li><strong>Tier Budgets:</strong> Homepage ' . (int)($strategy['budget_homepage'] ?? 10) . '%, ';
    echo 'Tier1 ' . (int)($strategy['budget_tier1'] ?? 50) . '%, ';
    echo 'Tier2 ' . (int)($strategy['budget_tier2'] ?? 30) . '%, ';
    echo 'Tier3 ' . (int)($strategy['budget_tier3'] ?? 10) . '%</li>';
    echo '<li><strong>Anchor Mix:</strong> Exact ' . (int)($strategy['anchor_exact'] ?? 80) . '%, ';
    echo 'Partial ' . (int)($strategy['anchor_partial'] ?? 10) . '%, ';
    echo 'Descriptive ' . (int)($strategy['anchor_descriptive'] ?? 5) . '%, ';
    echo 'Contextual ' . (int)($strategy['anchor_contextual'] ?? 5) . '%, ';
    echo 'Generic ' . (int)($strategy['anchor_generic'] ?? 0) . '%</li>';
    echo '<li><strong>Max Links/Run:</strong> ' . (int)($strategy['max_links_per_run'] ?? 3) . '</li>';
    echo '<li><strong>Min Relevance Score:</strong> ' . (int)($strategy['min_relevance_score'] ?? 20) . ' <span style="color:#666;">(0=off, 20=basic, 50+=strict)</span></li>';
    echo '</ul>';
    echo '<p class="description" style="margin-top:5px;">Change these in <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-strategy')) . '">Strategy</a>.</p>';
    echo '</div>';

    echo '<p><strong>Eligible docs (with keywords):</strong> <code>' . (int)$progress['total'] . '</code>. ';
    echo 'Pointer: <code>' . (int)get_option(self::$ptr_key, 0) . '</code>';
    echo '</p>';

    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
    echo '<input type="hidden" name="action" value="internallinkstool_run_linker" />';
    wp_nonce_field('internallinkstool_run_linker');

    echo '<table class="form-table"><tr>';
    echo '<th scope="row">Batch size</th>';
    echo '<td><input type="number" name="batch_size" value="' . esc_attr($batch_size) . '" min="1" max="100" />';
    echo '<p class="description">How many documents to process per click (avoid timeouts).</p>';
    echo '</td></tr>';

    echo '<tr><th scope="row">Mode</th><td>';
    $dry_checked = isset($_GET['dry']) ? ((int)$_GET['dry'] === 1) : 1;
    echo '<label><input type="checkbox" name="dry_run" value="1" ' . checked($dry_checked, 1, false) . ' /> <strong>Dry run (preview only)</strong></label>';
    echo '<p class="description">Recommended first: shows what would be inserted without modifying post content.</p>';
    echo '</td></tr>';

    echo '</table>';

    submit_button('Run Linker Batch');
    echo '</form>';

    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:10px;">';
    echo '<input type="hidden" name="action" value="internallinkstool_reset_linker_ptr" />';
    wp_nonce_field('internallinkstool_reset_linker_ptr');
    submit_button('Reset Linker Pointer (start from beginning)', 'secondary');
    echo '</form>';

    // === BACKGROUND RUN ALL ===
    echo '<hr>';
    echo '<h2>Run All (Background)</h2>';

    echo '<div style="background:#f0f6fc;border-left:4px solid #2271b1;padding:14px 18px;margin:10px 0 16px;">';
    echo '<p style="margin:0 0 6px;font-size:14px;"><strong>This is the main action &mdash; it does everything automatically:</strong></p>';
    echo '<ol style="margin:0 0 0 18px;padding:0;font-size:13px;line-height:1.7;">';
    echo '<li>Scans all pages according to your Link Settings (post types, statuses, exclusions)</li>';
    echo '<li>Extracts primary &amp; secondary keywords for each page using AI</li>';
    echo '<li>Generates AI Anchor Banks (diverse anchor text variations per page)</li>';
    echo '<li>Runs the linker to find and insert internal links across your content</li>';
    echo '</ol>';
    echo '<p style="margin:8px 0 0;font-size:12px;color:#555;">The Scanner, Keywords, and AI Anchor Banks pages are available separately for reviewing and debugging results.</p>';
    echo '</div>';

    $bg = get_option(self::$bg_key, []);
    $bg_status = $bg['status'] ?? 'idle';
    $bg_dry_run = !empty($bg['dry_run']);
    $bg_links_label = $bg_dry_run ? 'Links proposed' : 'Links inserted';

    if ($bg_status === 'running') {
      $bg_processed = (int)($bg['processed'] ?? 0);
      $bg_total = (int)($bg['total'] ?? 0);
      $bg_links = (int)($bg['inserted_links'] ?? 0);
      $bg_pct = $bg_total > 0 ? round(($bg_processed / $bg_total) * 100) : 0;

      $bg_phase = $bg['phase'] ?? 'linking';
      $phase_labels = [
        'scanning'         => 'Scanning pages...',
        'keywords'         => 'Extracting keywords (AI)...',
        'keyword_planning' => 'Planning unique keywords (AI)...',
        'anchor_banks'     => 'Generating anchor banks (AI)...',
        'linking'          => $bg_dry_run ? 'Linking (Dry Run)...' : 'Inserting links...',
      ];
      $bg_phase_label = $phase_labels[$bg_phase] ?? 'Processing...';

      echo '<div id="ilt-bg-progress" style="background:#f0f6fc;border:1px solid #2271b1;border-radius:4px;padding:15px;margin:10px 0;">';
      echo '<p style="margin:0 0 8px;"><strong>Background run is active' . ($bg_dry_run ? ' (Dry Run)' : '') . '</strong> &mdash; <span id="ilt-bg-phase">' . esc_html($bg_phase_label) . '</span></p>';
      echo '<div style="background:#ddd;border-radius:4px;height:24px;overflow:hidden;margin-bottom:8px;">';
      echo '<div id="ilt-bg-bar" style="background:#2271b1;height:100%;width:' . $bg_pct . '%;transition:width 0.3s;border-radius:4px;"></div>';
      echo '</div>';
      echo '<p style="margin:0;" id="ilt-bg-text">Processed: <strong>' . $bg_processed . '</strong> / ' . $bg_total . ' pages (' . $bg_pct . '%). ' . $bg_links_label . ': <strong>' . $bg_links . '</strong></p>';
      echo '</div>';

      echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:8px;">';
      echo '<input type="hidden" name="action" value="internallinkstool_bg_stop" />';
      wp_nonce_field('internallinkstool_bg_stop');
      submit_button('Stop Background Run', 'secondary', 'submit', false);
      echo '</form>';

      // Auto-poll for progress
      ?>
      <script type="text/javascript">
      (function() {
        var polling = setInterval(function() {
          fetch(ajaxurl + '?action=internallinkstool_bg_status&_ajax_nonce=<?php echo wp_create_nonce("internallinkstool_bg_status"); ?>')
            .then(function(r) { return r.json(); })
            .then(function(data) {
              if (!data.success) return;
              var d = data.data;
              var bar = document.getElementById('ilt-bg-bar');
              var text = document.getElementById('ilt-bg-text');
              if (bar) bar.style.width = d.pct + '%';
              var phaseEl = document.getElementById('ilt-bg-phase');
              if (phaseEl && d.phase_label) phaseEl.textContent = d.phase_label;
              var lbl = d.links_label || 'Links inserted';
              if (d.phase && d.phase !== 'linking') {
                var phaseProgress = '';
                if (d.phase_total > 0) {
                  phaseProgress = ' &mdash; ' + d.phase_done + ' / ' + d.phase_total + ' (' + d.phase_pct + '%)';
                  if (bar) bar.style.width = d.phase_pct + '%';
                }
                if (text) text.innerHTML = '<em>' + d.phase_label + '</em>' + phaseProgress;
              } else {
                if (text) text.innerHTML = 'Processed: <strong>' + d.processed + '</strong> / ' + d.total + ' pages (' + d.pct + '%). ' + lbl + ': <strong>' + d.inserted_links + '</strong>';
              }
              if (d.status !== 'running') {
                clearInterval(polling);
                location.reload();
              }
            })
            .catch(function() {});
        }, 3000);
      })();
      </script>
      <?php

    } elseif ($bg_status === 'done') {
      $bg_processed = (int)($bg['processed'] ?? 0);
      $bg_links = (int)($bg['inserted_links'] ?? 0);
      echo '<div class="notice notice-success inline" style="margin:8px 0;padding:8px 12px;">';
      echo '<strong>Background run complete' . ($bg_dry_run ? ' (Dry Run)' : '') . '.</strong> Processed: ' . $bg_processed . ' pages. ' . $bg_links_label . ': ' . $bg_links . '.';
      if ($bg_dry_run) {
        echo ' <em>No links were actually inserted. See preview below.</em>';
      }
      echo '</div>';

      echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:8px;display:inline-block;">';
      echo '<input type="hidden" name="action" value="internallinkstool_bg_start" />';
      echo '<input type="hidden" name="bg_dry_run" value="0" />';
      wp_nonce_field('internallinkstool_bg_start');
      submit_button('Run All Again (Insert Links)', 'primary', 'submit', false);
      echo '</form>';

      echo ' <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:8px;display:inline-block;margin-left:8px;">';
      echo '<input type="hidden" name="action" value="internallinkstool_bg_start" />';
      echo '<input type="hidden" name="bg_dry_run" value="1" />';
      wp_nonce_field('internallinkstool_bg_start');
      submit_button('Run All Again (Dry Run)', 'secondary', 'submit', false);
      echo '</form>';

    } else {
      // idle or stopped
      if ($bg_status === 'stopped') {
        $bg_processed = (int)($bg['processed'] ?? 0);
        $bg_links = (int)($bg['inserted_links'] ?? 0);
        echo '<div class="notice notice-warning inline" style="margin:8px 0;padding:8px 12px;">';
        echo '<strong>Background run was stopped' . ($bg_dry_run ? ' (Dry Run)' : '') . '.</strong> Processed: ' . $bg_processed . ' pages. ' . $bg_links_label . ': ' . $bg_links . '.';
        echo '</div>';
      }

      echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:8px;display:inline-block;">';
      echo '<input type="hidden" name="action" value="internallinkstool_bg_start" />';
      echo '<input type="hidden" name="bg_dry_run" value="0" />';
      wp_nonce_field('internallinkstool_bg_start');
      submit_button('Run All (Insert Links)', 'primary', 'submit', false);
      echo '</form>';

      echo ' <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin-top:8px;display:inline-block;margin-left:8px;">';
      echo '<input type="hidden" name="action" value="internallinkstool_bg_start" />';
      echo '<input type="hidden" name="bg_dry_run" value="1" />';
      wp_nonce_field('internallinkstool_bg_start');
      submit_button('Run All (Dry Run)', 'secondary', 'submit', false);
      echo '</form>';
    }

    // ✅ Always show debug table if we have it, even if preview is empty
    if (!empty($debug_rows)) {
      echo '<hr>';
      echo '<div class="ilt-accordion">';
      echo '<h2 class="ilt-accordion-header" style="cursor:pointer;user-select:none;display:flex;align-items:center;gap:8px;" onclick="this.nextElementSibling.style.display = this.nextElementSibling.style.display === \'none\' ? \'block\' : \'none\'; this.querySelector(\'.ilt-accordion-arrow\').textContent = this.nextElementSibling.style.display === \'none\' ? \'▶\' : \'▼\';">';
      echo '<span class="ilt-accordion-arrow" style="font-size:14px;color:#666;">▶</span>';
      echo '<span>Batch Debug: SOURCE Pages Processed (' . count($debug_rows) . ' pages)</span>';
      echo '</h2>';
      echo '<div class="ilt-accordion-content" style="display:none;">';
      echo '<p class="description">These are the <strong>source pages</strong> (pages that CONTAIN the links). The Primary/Secondary KW columns show each source page\'s own keywords (used for relevance scoring, NOT as anchor text). Anchor text comes from the <strong>target page\'s</strong> keywords — see the Dry Run Preview below.</p>';
      echo '<table class="widefat fixed striped">';
      echo '<thead><tr>';
      echo '<th style="width:5%;">Post ID</th>';
      echo '<th style="width:14%;">Title</th>';
      echo '<th style="width:14%;">URL</th>';
      echo '<th style="width:5%;">Type</th>';
      echo '<th style="width:5%;">Status</th>';
      echo '<th style="width:12%;">Meta Title</th>';
      echo '<th style="width:12%;">Meta Desc</th>';
      echo '<th style="width:10%;">Yoast Focus KW</th>';
      echo '<th style="width:10%;">Primary KW</th>';
      echo '<th style="width:10%;">Secondary KW</th>';
      echo '<th style="width:5%;">Proposals</th>';
      echo '</tr></thead><tbody>';

      foreach ($debug_rows as $d) {
        $pid = (int)($d['post_id'] ?? 0);
        $title = (string)($d['title'] ?? '');
        $edit = $pid ? get_edit_post_link($pid) : '';
        $url = (string)($d['url'] ?? '');
        $type = (string)($d['type'] ?? '');
        $status = (string)($d['status'] ?? '');
        $meta_title = (string)($d['meta_title'] ?? '');
        $meta_desc = (string)($d['meta_desc'] ?? '');
        $yoast_fkw = (string)($d['yoast_focus_kw'] ?? '');
        $pk = (string)($d['primary_keyword'] ?? '');
        $sk = (string)($d['secondary_keywords'] ?? '');
        $prop = (int)($d['proposals'] ?? 0);

        $meta_desc_short = mb_substr($meta_desc, 0, 120);
        $sk_short = mb_substr($sk, 0, 90);

        echo '<tr>';
        echo '<td><code>' . $pid . '</code></td>';
        echo '<td>' . ($edit ? '<a href="' . esc_url($edit) . '" target="_blank" rel="noopener">' . esc_html($title) . '</a>' : esc_html($title)) . '</td>';
        echo '<td>' . ($url ? '<a href="' . esc_url($url) . '" target="_blank" rel="noopener">' . esc_html($url) . '</a>' : '') . '</td>';
        echo '<td><code>' . esc_html($type) . '</code></td>';
        echo '<td><code>' . esc_html($status) . '</code></td>';
        echo '<td>' . esc_html(mb_substr($meta_title, 0, 80)) . '</td>';
        echo '<td>' . esc_html($meta_desc_short) . '</td>';
        echo '<td><code>' . esc_html($yoast_fkw) . '</code></td>';
        echo '<td><code>' . esc_html($pk) . '</code></td>';
        echo '<td>' . esc_html($sk_short) . '</td>';
        echo '<td><strong>' . $prop . '</strong></td>';
        echo '</tr>';
      }

      echo '</tbody></table>';
      echo '</div>'; // end accordion content
      echo '</div>'; // end accordion wrapper
    } else if ($preview_key) {
      echo '<hr><p><em>No debug rows stored for this run.</em></p>';
    }

    // Preview table
    if (!empty($preview)) {
      global $wpdb;
      echo '<hr>';
      echo '<h2>' . ($is_bg_preview ? 'Background Run Preview (all batches)' : 'Dry Run Preview (this batch)') . '</h2>';

      // Explanation about the linking approach
      echo '<div class="notice notice-info" style="background:#f0f6fc;border-left-color:#2271b1;padding:10px 15px;margin:10px 0 15px 0;">';
      echo '<p style="margin:0;"><strong>How anchor selection works:</strong></p>';
      echo '<p style="margin:5px 0 0 0;">Each row = a link FROM a source page TO a target page. The <strong>"Target Keywords"</strong> column shows the target page\'s keywords (from the Keywords step — same data as the Keywords page). ';
      echo 'The <strong>"Anchor Used"</strong> column shows which keyword or variation was actually found in the source content and will become the clickable link text.</p>';
      echo '<p style="margin:5px 0 0 0;"><strong>Selection priority (per target):</strong> Primary keyword (exact) → Bank exact anchors → Primary variations → Long secondary keywords (3+ words) → Bank partial anchors → Short secondary keywords (1-2 words) → Bank descriptive → Bank contextual → Bank generic.</p>';
      echo '<p style="margin:5px 0 0 0;"><strong>Target order:</strong> Follows Strategy tier budgets (Tier 1 first) and relevance scoring.</p>';
      echo '<p style="margin:5px 0 0 0;"><strong>Type column:</strong> Shows anchor type + source. <strong>&#9733; bank</strong> = from Strategy anchor bank (AI-generated). <strong>kw</strong> = from Keywords step.</p>';
      echo '<p style="margin:5px 0 0 0;"><strong>Tip:</strong> If Anchor Used differs from Target Keywords, it means the exact keyword wasn\'t found verbatim in the source content. Plural/singular matching and anchor bank variations are tried automatically.</p>';
      echo '</div>';

      // Check if any row has anchor rewriting data
      $has_rewriting = false;
      foreach ($preview as $row) {
        if (!empty($row['rewrite_changed']) || !empty($row['rewrite_error'])) {
          $has_rewriting = true;
          break;
        }
      }

      // Batch-fetch Yoast focus keywords for all target post IDs (single query)
      $target_ids = array_unique(array_filter(array_map(function($r) {
        return (int)($r['target_post_id'] ?? 0);
      }, $preview)));
      $yoast_map = [];
      if (!empty($target_ids)) {
        $ids_placeholder = implode(',', array_map('intval', $target_ids));
        $yoast_rows = $wpdb->get_results(
          "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE post_id IN ({$ids_placeholder}) AND meta_key = '_yoast_wpseo_focuskw'",
          ARRAY_A
        );
        if ($yoast_rows) {
          foreach ($yoast_rows as $yr) {
            $yoast_map[(int)$yr['post_id']] = (string)$yr['meta_value'];
          }
        }
      }

      echo '<table class="widefat fixed striped" id="linker-preview-table">';
      echo '<thead><tr>';
      echo '<th style="width:13%;cursor:pointer;" data-sort-col="0" title="Click to sort">Source URL <span class="sort-arrow">&#x25B4;&#x25BE;</span></th>';
      echo '<th style="width:13%;cursor:pointer;" data-sort-col="1" title="Click to sort">Target URL <span class="sort-arrow">&#x25B4;&#x25BE;</span></th>';
      echo '<th style="width:10%;">Target Keywords</th>';
      echo '<th style="width:9%;cursor:pointer;" data-sort-col="3" title="Click to sort">Anchor Used <span class="sort-arrow">&#x25B4;&#x25BE;</span></th>';
      echo '<th style="width:5%;">Type</th>';
      echo '<th style="width:4%;cursor:pointer;" data-sort-col="5" title="Click to sort">Tier <span class="sort-arrow">&#x25B4;&#x25BE;</span></th>';
      echo '<th style="width:4%;cursor:pointer;" data-sort-col="6" title="Click to sort">Rel <span class="sort-arrow">&#x25B4;&#x25BE;</span></th>';
      echo '<th style="width:14%;">AI Anchor Bank</th>';
      echo '<th style="width:12%;">Context</th>';
      echo '<th style="width:7%;">AI Suggest</th>';
      echo '<th style="width:5%;">Redo</th>';
      echo '</tr></thead><tbody>';

      $row_idx = 0;
      foreach ($preview as $row) {
        $row_idx++;
        $src_id = (int)($row['source_post_id'] ?? 0);
        $src_url = (string)($row['source_url'] ?? '');
        $src_edit = $src_id ? get_edit_post_link($src_id) : '';

        $target_post_id = (int)($row['target_post_id'] ?? 0);
        $target_url = (string)($row['target_url'] ?? '');
        $target_edit = $target_post_id ? get_edit_post_link($target_post_id) : '';
        $target_meta_title = (string)($row['target_meta_title'] ?? '');
        $target_meta_desc = (string)($row['target_meta_desc'] ?? '');
        $target_tier = (string)($row['target_tier'] ?? '');

        $anchor = (string)($row['anchor'] ?? '');
        $anchor_type = (string)($row['anchor_type'] ?? 'exact');
        $rewritten_anchor = (string)($row['rewritten_anchor'] ?? $anchor);
        $rewrite_changed = !empty($row['rewrite_changed']);
        $rewrite_reason = (string)($row['rewrite_reason'] ?? '');
        $rewrite_error = $row['rewrite_error'] ?? null;
        $excerpt = (string)($row['excerpt'] ?? '');

        // Tier badge colors
        $tier_colors = [
          'homepage' => '#9c27b0',
          'tier1' => '#1976d2',
          'tier2' => '#388e3c',
          'tier3' => '#f57c00',
        ];
        $tier_color = $tier_colors[$target_tier] ?? '#666';
        $tier_label = $target_tier === 'homepage' ? 'Home' : ucfirst($target_tier);

        // Anchor type badge colors
        $type_colors = [
          'exact' => '#d32f2f',
          'partial' => '#1976d2',
          'descriptive' => '#388e3c',
          'contextual' => '#7b1fa2',
          'generic' => '#666',
        ];
        $type_color = $type_colors[$anchor_type] ?? '#666';

        echo '<tr>';

        // Source URL column (full URL)
        echo '<td class="ilt-source-url" style="word-break:break-all;font-size:12px;">';
        if ($src_edit) {
          echo '<a href="' . esc_url($src_edit) . '" target="_blank" rel="noopener">' . esc_html($src_url) . '</a>';
        } else {
          echo '<a href="' . esc_url($src_url) . '" target="_blank" rel="noopener">' . esc_html($src_url) . '</a>';
        }
        echo '</td>';

        // Target URL column (full URL)
        echo '<td class="ilt-target-url" style="word-break:break-all;font-size:12px;">';
        if ($target_edit) {
          echo '<a href="' . esc_url($target_edit) . '" target="_blank" rel="noopener">' . esc_html($target_url) . '</a>';
        } else {
          echo '<a href="' . esc_url($target_url) . '" target="_blank" rel="noopener">' . esc_html($target_url) . '</a>';
        }
        echo '</td>';

        // Target Keywords column (from Keywords step)
        $target_keywords = isset($row['target_keywords']) && is_array($row['target_keywords']) ? $row['target_keywords'] : [];
        echo '<td class="ilt-target-kwds" style="font-size:11px;">';
        if (!empty($target_keywords)) {
          $kw_id = 'tkw-' . $row_idx;
          foreach ($target_keywords as $idx => $tkw) {
            $label = ($idx === 0) ? 'P:' : 'S:';
            $bg = ($idx === 0) ? '#e3f2fd' : '#f5f5f5';
            if ($idx <= 2) {
              echo '<div style="background:' . $bg . ';padding:2px 4px;margin:1px 0;border-radius:2px;"><strong>' . $label . '</strong> ' . esc_html($tkw) . '</div>';
            }
          }
          $extra = count($target_keywords) - 3;
          if ($extra > 0) {
            echo '<a href="#" class="tkw-toggle" data-target="' . esc_attr($kw_id) . '" style="color:#2271b1;font-size:10px;text-decoration:none;cursor:pointer;">+' . $extra . ' more</a>';
            echo '<div id="' . esc_attr($kw_id) . '" style="display:none;">';
            foreach ($target_keywords as $idx => $tkw) {
              if ($idx < 3) continue;
              echo '<div style="background:#f5f5f5;padding:2px 4px;margin:1px 0;border-radius:2px;"><strong>S:</strong> ' . esc_html($tkw) . '</div>';
            }
            echo '</div>';
          }
        } else {
          echo '<span style="color:#999;">No keywords</span>';
        }
        echo '</td>';

        // Anchor Text column (prominent) - shows which keyword/anchor was actually matched
        echo '<td class="ilt-anchor" data-anchor="' . esc_attr($anchor) . '">';
        echo '<code class="ilt-anchor-text" style="background:#fff3cd;color:#856404;padding:4px 8px;font-size:12px;font-weight:bold;display:inline-block;">' . esc_html($anchor) . '</code>';
        echo '</td>';

        // Anchor Type column (with source indicator: bank = from Strategy anchor bank, kw = from Keywords step)
        $anchor_source = (string)($row['anchor_source'] ?? 'keyword');
        $source_badge = $anchor_source === 'bank'
          ? '<div style="font-size:9px;color:#7b1fa2;margin-top:2px;" title="From Strategy anchor bank">&#9733; bank</div>'
          : '<div style="font-size:9px;color:#999;margin-top:2px;" title="From Keywords step">kw</div>';
        echo '<td class="ilt-anchor-type" data-type="' . esc_attr($anchor_type) . '" data-source="' . esc_attr($anchor_source) . '"><span class="ilt-type-label" style="background:' . $type_color . ';color:#fff;padding:2px 5px;border-radius:3px;font-size:10px;">' . esc_html($anchor_type) . '</span>' . $source_badge . '</td>';

        // Tier column
        echo '<td class="ilt-tier" data-tier="' . esc_attr($tier_label) . '"><span class="ilt-tier-label" style="background:' . $tier_color . ';color:#fff;padding:2px 6px;border-radius:3px;font-size:10px;">' . esc_html($tier_label) . '</span></td>';

        // Relevance score column
        $rel_score = (int)($row['relevance_score'] ?? 0);
        $rel_color = $rel_score >= 50 ? '#2e7d32' : ($rel_score >= 20 ? '#f57c00' : '#c62828');
        echo '<td class="ilt-relevance" data-score="' . $rel_score . '"><span class="ilt-rel-score" style="background:' . $rel_color . ';color:#fff;padding:2px 6px;border-radius:3px;font-size:10px;">' . $rel_score . '</span></td>';

        // AI Anchor Bank column — show all bank anchors grouped by type
        $target_bank = isset($row['target_anchor_bank']) && is_array($row['target_anchor_bank']) ? $row['target_anchor_bank'] : [];
        echo '<td class="ilt-anchor-bank" style="font-size:10px;line-height:1.4;">';
        if (!empty($target_bank)) {
          $bank_id = 'tab-' . $row_idx;
          $bank_type_colors = [
            'exact' => '#d32f2f',
            'partial' => '#1976d2',
            'descriptive' => '#388e3c',
            'contextual' => '#7b1fa2',
            'generic' => '#666',
          ];
          $bank_type_order = ['exact', 'partial', 'descriptive', 'contextual', 'generic'];
          $shown = 0;
          $hidden_items = [];
          foreach ($bank_type_order as $btype) {
            if (empty($target_bank[$btype])) continue;
            foreach ($target_bank[$btype] as $btext) {
              $bc = $bank_type_colors[$btype] ?? '#666';
              $pill = '<span style="display:inline-block;background:' . $bc . ';color:#fff;padding:1px 4px;border-radius:3px;margin:1px;font-size:9px;white-space:nowrap;">' . esc_html($btype) . '</span> <span style="font-size:10px;">' . esc_html($btext) . '</span>';
              if ($shown < 3) {
                echo '<div>' . $pill . '</div>';
                $shown++;
              } else {
                $hidden_items[] = $pill;
              }
            }
          }
          if (!empty($hidden_items)) {
            echo '<a href="#" class="tab-toggle" data-target="' . esc_attr($bank_id) . '" style="color:#2271b1;font-size:9px;text-decoration:none;cursor:pointer;">+' . count($hidden_items) . ' more</a>';
            echo '<div id="' . esc_attr($bank_id) . '" style="display:none;">';
            foreach ($hidden_items as $hi) {
              echo '<div>' . $hi . '</div>';
            }
            echo '</div>';
          }
        } else {
          echo '<span style="color:#999;">No bank</span>';
        }
        echo '</td>';

        // Context column (excerpt showing where link appears)
        echo '<td class="ilt-context" style="font-size:11px;color:#666;">';
        if ($excerpt) {
          // Highlight the anchor in the excerpt
          $highlighted = str_ireplace($anchor, '<mark style="background:#fff3cd;">' . esc_html($anchor) . '</mark>', esc_html($excerpt));
          echo $highlighted;
        } else {
          echo '-';
        }
        echo '</td>';

        // AI Suggestion column
        $row_id = 'anchor-row-' . $src_id . '-' . $target_post_id;
        echo '<td id="' . esc_attr($row_id) . '-result">';
        if ($rewrite_error) {
          echo '<span style="color:#999;" title="' . esc_attr($rewrite_error) . '">-</span>';
        } elseif ($rewrite_changed) {
          echo '<code style="background:#e7f5e7;color:#2e7d32;">' . esc_html($rewritten_anchor) . '</code>';
        } else {
          echo '<span style="color:#999;">-</span>';
        }
        echo '</td>';

        // Redo button column
        echo '<td>';
        echo '<button type="button" class="button button-small redo-anchor-btn" ';
        echo 'data-row-id="' . esc_attr($row_id) . '" ';
        echo 'data-anchor="' . esc_attr($anchor) . '" ';
        echo 'data-sentence="' . esc_attr($excerpt) . '" ';
        echo 'data-target-title="' . esc_attr($target_meta_title) . '" ';
        echo 'data-target-desc="' . esc_attr($target_meta_desc) . '">';
        echo 'Redo</button>';
        echo '</td>';

        echo '</tr>';
      }

      echo '</tbody></table>';

      // Anchor type + source distribution summary
      $type_dist = ['exact' => 0, 'partial' => 0, 'descriptive' => 0, 'contextual' => 0, 'generic' => 0];
      $source_dist = ['bank' => 0, 'keyword' => 0];
      foreach ($preview as $row) {
        $atype = $row['anchor_type'] ?? 'exact';
        if (isset($type_dist[$atype])) {
          $type_dist[$atype]++;
        }
        $asrc = ($row['anchor_source'] ?? 'keyword');
        $source_dist[$asrc] = ($source_dist[$asrc] ?? 0) + 1;
      }
      $total_links = array_sum($type_dist);

      echo '<div style="background:#f9f9f9;border:1px solid #ddd;border-radius:4px;padding:15px;margin-top:15px;">';
      echo '<h3 style="margin:0 0 10px 0;font-size:14px;">Batch Anchor Type Distribution</h3>';
      echo '<div style="display:flex;gap:15px;flex-wrap:wrap;">';

      $type_labels = ['exact' => 'Exact Match', 'partial' => 'Partial/Semantic', 'descriptive' => 'Descriptive', 'contextual' => 'Contextual', 'generic' => 'Generic'];
      $type_colors_dist = ['exact' => '#d32f2f', 'partial' => '#1976d2', 'descriptive' => '#388e3c', 'contextual' => '#7b1fa2', 'generic' => '#666'];

      foreach ($type_dist as $atype => $count) {
        $pct = $total_links > 0 ? round(($count / $total_links) * 100) : 0;
        $label = $type_labels[$atype] ?? ucfirst($atype);
        $color = $type_colors_dist[$atype] ?? '#666';
        echo '<div style="text-align:center;">';
        echo '<div style="background:' . $color . ';color:#fff;padding:8px 12px;border-radius:4px;font-weight:bold;">' . $count . '</div>';
        echo '<div style="font-size:11px;color:#666;margin-top:4px;">' . esc_html($label) . '</div>';
        echo '<div style="font-size:10px;color:#999;">' . $pct . '%</div>';
        echo '</div>';
      }

      echo '</div>';

      // Anchor source distribution (bank vs keyword)
      $bank_count = (int)($source_dist['bank'] ?? 0);
      $kw_count = (int)($source_dist['keyword'] ?? 0);
      $bank_pct = $total_links > 0 ? round(($bank_count / $total_links) * 100) : 0;
      $kw_pct = $total_links > 0 ? round(($kw_count / $total_links) * 100) : 0;
      echo '<div style="margin-top:12px;padding-top:12px;border-top:1px solid #ddd;">';
      echo '<strong style="font-size:12px;">Anchor Source:</strong> ';
      echo '<span style="background:#7b1fa2;color:#fff;padding:2px 6px;border-radius:3px;font-size:11px;margin-left:8px;">&#9733; Bank: ' . $bank_count . ' (' . $bank_pct . '%)</span> ';
      echo '<span style="background:#999;color:#fff;padding:2px 6px;border-radius:3px;font-size:11px;margin-left:4px;">Keywords: ' . $kw_count . ' (' . $kw_pct . '%)</span>';
      echo '</div>';

      echo '<p style="margin:10px 0 0 0;font-size:12px;color:#666;"><strong>How it works:</strong> The Linker enforces your Strategy anchor mix by assigning link slots per type (e.g. 80% exact = 4 of 5 links must be exact). For each slot, only candidates matching the needed type are tried. If no match is found, that slot stays empty rather than using the wrong type. Within each type, priority: Primary keyword → Bank entries → Variations → Secondary keywords.</p>';
      echo '</div>';

      // ══════════════════════════════════════════════════════════════════════
      // LINK DISTRIBUTION SUMMARY
      // ══════════════════════════════════════════════════════════════════════

      // Gather unique source and target URLs with their link counts
      $source_urls = [];
      $target_urls = [];

      foreach ($preview as $row) {
        $src_url = (string)($row['source_url'] ?? '');
        $tgt_url = (string)($row['target_url'] ?? '');
        $anchor  = (string)($row['anchor'] ?? '');

        // Count links FROM each source (with target->anchor mapping)
        if ($src_url !== '') {
          if (!isset($source_urls[$src_url])) {
            $source_urls[$src_url] = ['count' => 0, 'targets' => [], 'target_anchors' => []];
          }
          $source_urls[$src_url]['count']++;
          $source_urls[$src_url]['targets'][] = $tgt_url;
          $source_urls[$src_url]['target_anchors'][$tgt_url] = $anchor;
        }

        // Count links TO each target with anchors
        if ($tgt_url !== '') {
          if (!isset($target_urls[$tgt_url])) {
            $target_urls[$tgt_url] = ['count' => 0, 'anchors' => [], 'sources' => []];
          }
          $target_urls[$tgt_url]['count']++;
          $target_urls[$tgt_url]['anchors'][] = $anchor;
          $target_urls[$tgt_url]['sources'][] = $src_url;
        }
      }

      $num_sources = count($source_urls);
      $num_targets = count($target_urls);

      // Sort targets by link count descending
      uasort($target_urls, function($a, $b) {
        return $b['count'] - $a['count'];
      });

      // Sort sources by link count descending
      uasort($source_urls, function($a, $b) {
        return $b['count'] - $a['count'];
      });

      echo '<div style="background:#f0f6fc;border:1px solid #c3d9ed;border-radius:4px;padding:15px;margin-top:15px;">';
      echo '<h3 style="margin:0 0 12px 0;font-size:14px;color:#1d4ed8;">📊 Link Distribution Summary</h3>';

      // Overall stats
      echo '<div style="display:flex;gap:30px;margin-bottom:15px;">';
      echo '<div style="text-align:center;padding:10px 20px;background:#fff;border-radius:4px;border:1px solid #ddd;">';
      echo '<div style="font-size:28px;font-weight:bold;color:#1976d2;">' . $num_sources . '</div>';
      echo '<div style="font-size:11px;color:#666;">Source Pages<br>(pages linking out)</div>';
      echo '</div>';
      echo '<div style="text-align:center;padding:10px 20px;background:#fff;border-radius:4px;border:1px solid #ddd;">';
      echo '<div style="font-size:28px;font-weight:bold;color:#388e3c;">' . $num_targets . '</div>';
      echo '<div style="font-size:11px;color:#666;">Target Pages<br>(pages receiving links)</div>';
      echo '</div>';
      echo '<div style="text-align:center;padding:10px 20px;background:#fff;border-radius:4px;border:1px solid #ddd;">';
      echo '<div style="font-size:28px;font-weight:bold;color:#7b1fa2;">' . $total_links . '</div>';
      echo '<div style="font-size:11px;color:#666;">Total Links<br>(anchor insertions)</div>';
      echo '</div>';
      echo '</div>';

      // Target URLs breakdown table (collapsible)
      echo '<div class="ilt-accordion" style="margin-top:10px;">';
      echo '<h4 class="ilt-accordion-header" style="cursor:pointer;user-select:none;display:flex;align-items:center;gap:8px;margin:0;padding:8px;background:#e3f2fd;border-radius:4px;" onclick="this.nextElementSibling.style.display = this.nextElementSibling.style.display === \'none\' ? \'block\' : \'none\'; this.querySelector(\'.ilt-accordion-arrow\').textContent = this.nextElementSibling.style.display === \'none\' ? \'▶\' : \'▼\';">';
      echo '<span class="ilt-accordion-arrow" style="font-size:12px;color:#666;">▶</span>';
      echo '<span style="font-size:13px;">Links Received Per Target URL (' . $num_targets . ' URLs)</span>';
      echo '</h4>';
      echo '<div class="ilt-accordion-content" style="display:none;margin-top:8px;">';

      echo '<table class="widefat fixed striped" style="font-size:12px;">';
      echo '<thead><tr>';
      echo '<th style="width:40%;">Target URL</th>';
      echo '<th style="width:8%;text-align:center;">Links</th>';
      echo '<th style="width:52%;">Anchors Used</th>';
      echo '</tr></thead><tbody>';

      foreach ($target_urls as $url => $data) {
        $url_short = strlen($url) > 60 ? substr($url, 0, 57) . '...' : $url;
        $anchors_unique = array_unique($data['anchors']);
        $anchors_display = array_slice($anchors_unique, 0, 5);
        $more_count = count($anchors_unique) - 5;

        echo '<tr>';
        echo '<td><a href="' . esc_url($url) . '" target="_blank" rel="noopener" title="' . esc_attr($url) . '">' . esc_html($url_short) . '</a></td>';
        echo '<td style="text-align:center;"><strong>' . (int)$data['count'] . '</strong></td>';
        echo '<td>';
        foreach ($anchors_display as $anch) {
          echo '<code style="background:#f5f5f5;padding:1px 4px;margin-right:4px;font-size:11px;">' . esc_html($anch) . '</code>';
        }
        if ($more_count > 0) {
          echo '<span style="color:#666;font-size:10px;">+' . $more_count . ' more</span>';
        }
        echo '</td>';
        echo '</tr>';
      }

      echo '</tbody></table>';
      echo '</div>'; // end accordion content
      echo '</div>'; // end accordion

      // Source URLs breakdown table (collapsible)
      echo '<div class="ilt-accordion" style="margin-top:10px;">';
      echo '<h4 class="ilt-accordion-header" style="cursor:pointer;user-select:none;display:flex;align-items:center;gap:8px;margin:0;padding:8px;background:#e8f5e9;border-radius:4px;" onclick="this.nextElementSibling.style.display = this.nextElementSibling.style.display === \'none\' ? \'block\' : \'none\'; this.querySelector(\'.ilt-accordion-arrow\').textContent = this.nextElementSibling.style.display === \'none\' ? \'▶\' : \'▼\';">';
      echo '<span class="ilt-accordion-arrow" style="font-size:12px;color:#666;">▶</span>';
      echo '<span style="font-size:13px;">Links Sent Per Source URL (' . $num_sources . ' URLs)</span>';
      echo '</h4>';
      echo '<div class="ilt-accordion-content" style="display:none;margin-top:8px;">';

      echo '<table class="widefat fixed striped" style="font-size:12px;">';
      echo '<thead><tr>';
      echo '<th style="width:50%;">Source URL</th>';
      echo '<th style="width:8%;text-align:center;">Links Out</th>';
      echo '<th style="width:42%;">Linked To (with anchor)</th>';
      echo '</tr></thead><tbody>';

      foreach ($source_urls as $url => $data) {
        $url_short = strlen($url) > 70 ? substr($url, 0, 67) . '...' : $url;
        $targets_unique = array_unique($data['targets']);
        $targets_display = array_slice($targets_unique, 0, 4);
        $more_count = count($targets_unique) - 4;

        echo '<tr>';
        echo '<td><a href="' . esc_url($url) . '" target="_blank" rel="noopener" title="' . esc_attr($url) . '">' . esc_html($url_short) . '</a></td>';
        echo '<td style="text-align:center;"><strong>' . (int)$data['count'] . '</strong></td>';
        echo '<td style="font-size:10px;">';
        foreach ($targets_display as $tgt) {
          // Show just the path portion for cleaner display
          $tgt_path = parse_url($tgt, PHP_URL_PATH) ?: $tgt;
          $tgt_short = strlen($tgt_path) > 35 ? substr($tgt_path, 0, 32) . '...' : $tgt_path;
          $anchor_used = isset($data['target_anchors'][$tgt]) ? $data['target_anchors'][$tgt] : '';
          echo '<div style="margin-bottom:3px;display:flex;align-items:center;gap:6px;">';
          echo '<a href="' . esc_url($tgt) . '" target="_blank" rel="noopener" style="color:#1976d2;" title="' . esc_attr($tgt) . '">' . esc_html($tgt_short) . '</a>';
          if ($anchor_used !== '') {
            echo '<code style="background:#fff3cd;color:#856404;padding:1px 4px;font-size:9px;white-space:nowrap;">' . esc_html($anchor_used) . '</code>';
          }
          echo '</div>';
        }
        if ($more_count > 0) {
          echo '<span style="color:#999;">+' . $more_count . ' more</span>';
        }
        echo '</td>';
        echo '</tr>';
      }

      echo '</tbody></table>';
      echo '</div>'; // end accordion content
      echo '</div>'; // end accordion

      echo '</div>'; // end link distribution summary box

      // JavaScript for Redo Anchor AJAX
      ?>
      <script type="text/javascript">
      (function() {
        var buttons = document.querySelectorAll('.redo-anchor-btn');
        buttons.forEach(function(btn) {
          btn.addEventListener('click', function() {
            var rowId = this.getAttribute('data-row-id');
            var anchor = this.getAttribute('data-anchor');
            var sentence = this.getAttribute('data-sentence');
            var targetTitle = this.getAttribute('data-target-title');
            var targetDesc = this.getAttribute('data-target-desc');
            var resultCell = document.getElementById(rowId + '-result');
            var button = this;

            // Disable button and show loading
            button.disabled = true;
            button.textContent = 'Loading...';
            resultCell.innerHTML = '<span style="color:#666;">Generating...</span>';

            // AJAX request
            var formData = new FormData();
            formData.append('action', 'internallinkstool_redo_anchor');
            formData.append('anchor', anchor);
            formData.append('sentence', sentence);
            formData.append('target_title', targetTitle);
            formData.append('target_desc', targetDesc);
            formData.append('_ajax_nonce', '<?php echo wp_create_nonce("internallinkstool_redo_anchor"); ?>');

            fetch(ajaxurl, {
              method: 'POST',
              body: formData
            })
            .then(function(response) { return response.json(); })
            .then(function(data) {
              button.disabled = false;
              button.textContent = 'Redo';

              if (data.success) {
                var html = '<code style="background:#e7f5e7;color:#2e7d32;font-weight:bold;">' + escapeHtml(data.data.rewritten) + '</code>';

                // Show alternatives if available
                if (data.data.alternatives && data.data.alternatives.length > 0) {
                  html += '<div style="color:#666;font-size:10px;margin-top:3px;">Alt: ';
                  data.data.alternatives.forEach(function(alt, i) {
                    if (i > 0) html += ', ';
                    html += '<code style="background:#f0f0f0;">' + escapeHtml(alt) + '</code>';
                  });
                  html += '</div>';
                }

                // Show reason
                if (data.data.reason && data.data.reason.indexOf('Alt:') === -1) {
                  html += '<div style="color:#888;font-size:10px;margin-top:2px;">' + escapeHtml(data.data.reason) + '</div>';
                }
                resultCell.innerHTML = html;
              } else {
                resultCell.innerHTML = '<span style="color:#d63638;">Error: ' + escapeHtml(data.data || 'Unknown error') + '</span>';
              }
            })
            .catch(function(err) {
              button.disabled = false;
              button.textContent = 'Redo';
              resultCell.innerHTML = '<span style="color:#d63638;">Request failed</span>';
            });
          });
        });

        function escapeHtml(text) {
          var div = document.createElement('div');
          div.appendChild(document.createTextNode(text));
          return div.innerHTML;
        }
      })();

      // ── Sortable preview table ──
      (function() {
        var table = document.getElementById('linker-preview-table');
        if (!table) return;
        var thead = table.querySelector('thead');
        var tbody = table.querySelector('tbody');
        if (!thead || !tbody) return;

        var sortState = {}; // col → 'asc' | 'desc'

        thead.addEventListener('click', function(e) {
          var th = e.target.closest('th[data-sort-col]');
          if (!th) return;
          var col = parseInt(th.getAttribute('data-sort-col'), 10);
          if (isNaN(col)) return;

          // Toggle direction
          var dir = sortState[col] === 'asc' ? 'desc' : 'asc';
          sortState[col] = dir;

          // Update arrow indicators
          thead.querySelectorAll('.sort-arrow').forEach(function(s) { s.innerHTML = '\u25B4\u25BE'; });
          var arrow = th.querySelector('.sort-arrow');
          if (arrow) arrow.innerHTML = dir === 'asc' ? '\u25B4' : '\u25BE';

          // Sort rows
          var rows = Array.prototype.slice.call(tbody.querySelectorAll('tr'));
          rows.sort(function(a, b) {
            var cellA = a.children[col];
            var cellB = b.children[col];
            if (!cellA || !cellB) return 0;
            var valA = (cellA.textContent || '').trim().toLowerCase();
            var valB = (cellB.textContent || '').trim().toLowerCase();
            // Try numeric sort for Rel column
            var numA = parseFloat(valA);
            var numB = parseFloat(valB);
            if (!isNaN(numA) && !isNaN(numB)) {
              return dir === 'asc' ? numA - numB : numB - numA;
            }
            if (valA < valB) return dir === 'asc' ? -1 : 1;
            if (valA > valB) return dir === 'asc' ? 1 : -1;
            return 0;
          });

          // Re-append sorted rows
          rows.forEach(function(row) { tbody.appendChild(row); });
        });
      })();
      </script>
      <script type="text/javascript">
      (function() {
        document.querySelectorAll('.tkw-toggle, .tab-toggle').forEach(function(link) {
          link.addEventListener('click', function(e) {
            e.preventDefault();
            var target = document.getElementById(this.getAttribute('data-target'));
            if (!target) return;
            if (target.style.display === 'none') {
              target.style.display = 'block';
              this.textContent = this.textContent.replace('+', '−').replace('more', 'less');
            } else {
              target.style.display = 'none';
              this.textContent = this.textContent.replace('−', '+').replace('less', 'more');
            }
          });
        });
      })();
      </script>
      <style>
        #linker-preview-table thead th[data-sort-col]:hover { background: #e8e8e8; }
        #linker-preview-table thead .sort-arrow { color: #999; font-size: 10px; margin-left: 3px; }
      </style>
      <?php
    } else if ($preview_key) {
      echo '<hr><p><em>No proposals were generated in this batch.</em></p>';
    }

    // === ALWAYS-VISIBLE: All Scanned Pages ===
    self::render_scanned_pages_table($settings);

    echo '</div>';
  }

  /**
   * Render table of all scanned pages with their keywords (always visible on Linker page)
   */
  private static function render_scanned_pages_table($settings) {
    global $wpdb;

    $tables = self::get_tables();
    $docs = $tables['docs'];
    $kws  = $tables['kws'];

    $lp_per_page = isset($_GET['lp_per_page']) ? max(1, min(500, (int)$_GET['lp_per_page'])) : 50;
    $lp_paged    = isset($_GET['lp_paged']) ? max(1, (int)$_GET['lp_paged']) : 1;

    $total = (int)$wpdb->get_var(
      "SELECT COUNT(*) FROM {$docs} d
       INNER JOIN {$kws} k ON k.document_id = d.id
       WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''"
    );

    if ($total === 0) {
      echo '<hr>';
      echo '<h2>Scanned Pages</h2>';
      echo '<p><em>No scanned pages with keywords yet. Run the Scanner and Keywords steps first.</em></p>';
      return;
    }

    $total_pages = max(1, (int)ceil($total / $lp_per_page));
    if ($lp_paged > $total_pages) $lp_paged = $total_pages;
    $offset = ($lp_paged - 1) * $lp_per_page;

    // Check if tier column exists
    $tier_col = '';
    $tier_exists = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if ($tier_exists) {
      $tier_col = ', d.tier';
    }

    $rows = $wpdb->get_results($wpdb->prepare(
      "SELECT d.id AS doc_id, d.post_id, d.url, d.meta_title, d.type, d.status,
              k.primary_keyword, k.secondary_keywords {$tier_col}
       FROM {$docs} d
       INNER JOIN {$kws} k ON k.document_id = d.id
       WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
       ORDER BY d.id ASC
       LIMIT %d OFFSET %d",
      $lp_per_page, $offset
    ), ARRAY_A);

    $base_url = admin_url('admin.php?page=internallinkstool-linker');

    echo '<hr>';
    echo '<h2>Scanned Pages</h2>';
    echo '<p class="description">All pages that have been scanned and have keywords assigned. These are eligible for linking.</p>';

    echo '<form method="get" action="' . esc_url($base_url) . '" style="margin:8px 0;">';
    echo '<input type="hidden" name="page" value="internallinkstool-linker" />';
    echo '<label>Show <input type="number" name="lp_per_page" value="' . esc_attr($lp_per_page) . '" min="1" max="500" style="width:70px;" /> per page</label> ';
    echo '<input type="submit" class="button" value="Apply" />';
    echo '<span class="description" style="margin-left:12px;">' . esc_html($total) . ' pages total. Page ' . esc_html($lp_paged) . ' of ' . esc_html($total_pages) . '.</span>';
    echo '</form>';

    echo '<table class="widefat fixed striped">';
    echo '<thead><tr>';
    echo '<th style="width:5%;">ID</th>';
    echo '<th style="width:22%;">URL</th>';
    echo '<th style="width:15%;">Meta Title</th>';
    echo '<th style="width:5%;">Type</th>';
    echo '<th style="width:15%;">Primary Keyword</th>';
    echo '<th style="width:20%;">Secondary Keywords</th>';
    if ($tier_exists) echo '<th style="width:8%;">Tier</th>';
    echo '</tr></thead><tbody>';

    $tier_colors = [
      'homepage' => '#9c27b0', 'tier1' => '#1976d2',
      'tier2' => '#388e3c', 'tier3' => '#f57c00',
    ];
    $tier_labels = [
      'homepage' => 'Home', 'tier1' => 'Tier 1',
      'tier2' => 'Tier 2', 'tier3' => 'Tier 3',
    ];

    foreach ($rows as $r) {
      $pid = (int)$r['post_id'];
      $edit = $pid ? get_edit_post_link($pid) : '';
      $sk = (string)($r['secondary_keywords'] ?? '');
      $sk_short = mb_strlen($sk) > 100 ? mb_substr($sk, 0, 100) . '...' : $sk;

      echo '<tr>';
      echo '<td><code>' . $pid . '</code></td>';
      echo '<td style="word-break:break-all;font-size:12px;">';
      if ($edit) {
        echo '<a href="' . esc_url($edit) . '" target="_blank">' . esc_html($r['url']) . '</a>';
      } else {
        echo '<a href="' . esc_url($r['url']) . '" target="_blank">' . esc_html($r['url']) . '</a>';
      }
      echo '</td>';
      echo '<td>' . esc_html(mb_substr((string)($r['meta_title'] ?? ''), 0, 80)) . '</td>';
      echo '<td><code>' . esc_html($r['type'] ?? '') . '</code></td>';
      echo '<td><code>' . esc_html($r['primary_keyword'] ?? '') . '</code></td>';
      echo '<td style="font-size:12px;">' . esc_html($sk_short) . '</td>';

      if ($tier_exists) {
        $t = (string)($r['tier'] ?? '');
        $tc = $tier_colors[$t] ?? '#999';
        $tl = $tier_labels[$t] ?? ($t ?: '-');
        echo '<td>';
        if ($t) {
          echo '<span style="background:' . $tc . ';color:#fff;padding:2px 8px;border-radius:3px;font-size:11px;">' . esc_html($tl) . '</span>';
        } else {
          echo '<span style="color:#999;">-</span>';
        }
        echo '</td>';
      }

      echo '</tr>';
    }

    echo '</tbody></table>';

    if ($total_pages > 1) {
      echo '<div style="margin:10px 0;">';
      if ($lp_paged > 1) {
        $prev_url = add_query_arg(['lp_paged' => $lp_paged - 1, 'lp_per_page' => $lp_per_page], $base_url);
        echo '<a class="button" href="' . esc_url($prev_url) . '">&laquo; Previous</a> ';
      }
      if ($lp_paged < $total_pages) {
        $next_url = add_query_arg(['lp_paged' => $lp_paged + 1, 'lp_per_page' => $lp_per_page], $base_url);
        echo '<a class="button" href="' . esc_url($next_url) . '">Next &raquo;</a>';
      }
      echo '</div>';
    }
  }

  public static function handle_reset_ptr() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_reset_linker_ptr')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-linker&err=' . rawurlencode('Nonce failed (reset).')));
      exit;
    }

    update_option(self::$ptr_key, 0, false);
    wp_redirect(admin_url('admin.php?page=internallinkstool-linker&msg=' . rawurlencode('Linker pointer reset to 0.')));
    exit;
  }

  public static function handle_run_linker() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_run_linker')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-linker&err=' . rawurlencode('Nonce failed. Refresh and try again.')));
      exit;
    }

    $batch_size = isset($_POST['batch_size']) ? (int)$_POST['batch_size'] : 10;
    $batch_size = max(1, min(100, $batch_size));
    $dry_run = !empty($_POST['dry_run']);

    try {
      $res = self::run_batch($batch_size, $dry_run);

      $dedup_count = (int)($res['dedup_count'] ?? 0);
      $dedup_suffix = $dedup_count > 0 ? (', Deduped anchors: ' . $dedup_count) : '';

      $msg = $dry_run
        ? ('Dry run complete. Processed: ' . (int)$res['processed'] . ', Proposed links: ' . (int)$res['inserted_links'] . $dedup_suffix . '.')
        : ('Linker run complete. Processed: ' . (int)$res['processed'] . ', Updated: ' . (int)$res['updated'] . ', Inserted links: ' . (int)$res['inserted_links'] . $dedup_suffix . '.');

      $qs = 'page=internallinkstool-linker&batch_size=' . $batch_size . '&dry=' . ($dry_run ? 1 : 0) . '&msg=' . rawurlencode($msg);
      if (!empty($res['preview_key'])) $qs .= '&preview_key=' . rawurlencode($res['preview_key']);

      wp_redirect(admin_url('admin.php?' . $qs));
      exit;

    } catch (Exception $e) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-linker&err=' . rawurlencode('Linker error: ' . $e->getMessage())));
      exit;
    }
  }

  /* -------------------------
   * Background Run All
   * ------------------------- */
  public static function handle_bg_start() {
    if (!current_user_can('manage_options')) wp_die('No permission');
    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_bg_start')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-linker&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    $dry_run = !empty($_POST['bg_dry_run']);

    // Reset pointer to 0
    update_option(self::$ptr_key, 0, false);

    // Clear any previous accumulated preview data
    delete_transient('internallinkstool_bg_linker_preview');
    delete_transient('internallinkstool_bg_linker_debug');

    // Store background state — start with scanning phase
    update_option(self::$bg_key, [
      'status'         => 'running',
      'phase'          => 'scanning',
      'dry_run'        => $dry_run,
      'total'          => 0,
      'processed'      => 0,
      'updated'        => 0,
      'inserted_links' => 0,
      'batch_size'     => 20,
      'started_at'     => current_time('mysql'),
    ], false);

    // Schedule first batch immediately
    wp_schedule_single_event(time(), 'internallinkstool_bg_linker_batch');
    spawn_cron();

    wp_redirect(admin_url('admin.php?page=internallinkstool-linker&msg=' . rawurlencode('Background run started — scanning, extracting keywords, generating anchor banks, then linking.' . ($dry_run ? ' (Dry Run)' : ''))));
    exit;
  }

  public static function handle_bg_stop() {
    if (!current_user_can('manage_options')) wp_die('No permission');
    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_bg_stop')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-linker&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    $bg = get_option(self::$bg_key, []);
    $bg['status'] = 'stopped';
    update_option(self::$bg_key, $bg, false);

    // Clear any scheduled events
    wp_clear_scheduled_hook('internallinkstool_bg_linker_batch');

    wp_redirect(admin_url('admin.php?page=internallinkstool-linker&msg=' . rawurlencode('Background linker stopped.')));
    exit;
  }

  public static function ajax_bg_status() {
    check_ajax_referer('internallinkstool_bg_status');
    $bg = get_option(self::$bg_key, []);

    // ── Cron kick: if process is running but seems stuck, trigger it ──
    if (($bg['status'] ?? 'idle') === 'running') {
      $last_update = (int)($bg['last_update'] ?? 0);
      $now = time();
      // If no update in last 10 seconds, try to kick the cron
      if ($last_update === 0 || ($now - $last_update) > 10) {
        // Check if cron event is scheduled
        $next_scheduled = wp_next_scheduled('internallinkstool_bg_linker_batch');
        if ($next_scheduled === false || $next_scheduled <= $now) {
          // No scheduled event or it's overdue - reschedule and spawn
          wp_clear_scheduled_hook('internallinkstool_bg_linker_batch');
          wp_schedule_single_event($now, 'internallinkstool_bg_linker_batch');
          spawn_cron();
        }

        // If still stuck after 30 seconds, directly process one batch via AJAX
        // This handles hosts where loopback/cron is completely blocked
        if ($last_update > 0 && ($now - $last_update) > 30) {
          // Set a short time limit and process directly
          @set_time_limit(60);
          self::process_bg_batch();
        }
      }
    }

    $total = (int)($bg['total'] ?? 0);
    $processed = (int)($bg['processed'] ?? 0);
    $pct = $total > 0 ? round(($processed / $total) * 100) : 0;

    $is_dry = !empty($bg['dry_run']);
    $phase = $bg['phase'] ?? 'linking';
    $phase_labels = [
      'scanning'         => 'Scanning pages...',
      'keywords'         => 'Extracting keywords (AI)...',
      'keyword_planning' => 'Planning unique keywords (AI)...',
      'anchor_banks'     => 'Generating anchor banks (AI)...',
      'linking'          => $is_dry ? 'Linking (Dry Run)...' : 'Inserting links...',
    ];
    $phase_total = (int)($bg['phase_total'] ?? 0);
    $phase_done  = (int)($bg['phase_done'] ?? 0);
    $phase_pct   = $phase_total > 0 ? round(($phase_done / $phase_total) * 100) : 0;

    wp_send_json_success([
      'status'         => $bg['status'] ?? 'idle',
      'total'          => $total,
      'processed'      => $processed,
      'updated'        => (int)($bg['updated'] ?? 0),
      'inserted_links' => (int)($bg['inserted_links'] ?? 0),
      'pct'            => $pct,
      'dry_run'        => $is_dry,
      'links_label'    => $is_dry ? 'Links proposed' : 'Links inserted',
      'phase'          => $phase,
      'phase_label'    => $phase_labels[$phase] ?? 'Processing...',
      'phase_total'    => $phase_total,
      'phase_done'     => $phase_done,
      'phase_pct'      => $phase_pct,
    ]);
  }

  public static function process_bg_batch() {
    $bg = get_option(self::$bg_key, []);
    if (($bg['status'] ?? 'idle') !== 'running') return;

    // Track last update time for stall detection
    $bg['last_update'] = time();
    update_option(self::$bg_key, $bg, false);

    $phase = $bg['phase'] ?? 'linking';
    $batch_size = (int)($bg['batch_size'] ?? 20);
    $dry_run = !empty($bg['dry_run']);
    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];

    // ── Phase: Scanning ──
    if ($phase === 'scanning') {
      $scanned = 0;
      $types = [];
      if (!empty($settings['include_posts'])) $types[] = 'post';
      if (!empty($settings['include_pages'])) $types[] = 'page';
      if (empty($types)) $types = ['post', 'page'];
      $statuses = self::get_allowed_statuses($settings);

      if (class_exists('InternalLinksTool_DB') && method_exists('InternalLinksTool_DB', 'get_unmapped_post_ids')
          && method_exists('InternalLinksTool_DB', 'upsert_document_from_post')) {
        $ids = InternalLinksTool_DB::get_unmapped_post_ids($types, $statuses, $batch_size);
        foreach ($ids as $pid) {
          InternalLinksTool_DB::upsert_document_from_post((int)$pid);
          $scanned++;
        }
      }

      if ($scanned === 0) {
        // All pages scanned — move to keywords phase
        $bg['phase'] = 'keywords';
        $bg['phase_total'] = 0;
        $bg['phase_done']  = 0;
      } else {
        // Track scanning progress: total eligible WP posts vs already scanned
        global $wpdb;
        $type_ph = implode(',', array_fill(0, count($types), '%s'));
        $status_ph = implode(',', array_fill(0, count($statuses), '%s'));
        $total_posts = (int)$wpdb->get_var($wpdb->prepare(
          "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type IN ({$type_ph}) AND post_status IN ({$status_ph})",
          array_merge($types, $statuses)
        ));
        $scanned_docs = class_exists('InternalLinksTool_DB') && method_exists('InternalLinksTool_DB', 'count_documents_by_types_and_statuses')
          ? (int)InternalLinksTool_DB::count_documents_by_types_and_statuses($types, $statuses)
          : 0;
        $bg['phase_total'] = $total_posts;
        $bg['phase_done']  = $scanned_docs;
      }
      update_option(self::$bg_key, $bg, false);
      wp_schedule_single_event(time() + 1, 'internallinkstool_bg_linker_batch');
      spawn_cron();
      return;
    }

    // ── Phase: Keywords extraction ──
    // Use small batch for API-heavy phases to avoid PHP timeout
    if ($phase === 'keywords') {
      $api_batch = min($batch_size, 5);
      $kw_done = 0;
      if (class_exists('InternalLinksTool_Keywords') && method_exists('InternalLinksTool_Keywords', 'run_batch')) {
        try {
          $kw_res = InternalLinksTool_Keywords::run_batch($api_batch);
          $kw_done = (int)($kw_res['processed'] ?? 0);
        } catch (Exception $e) {
          $kw_done = 0;
        }
      }

      if ($kw_done === 0) {
        // All keywords extracted — move to keyword planning phase
        $bg['phase'] = 'keyword_planning';
        $bg['phase_done'] = 0;
        $bg['phase_total'] = 0;
        $bg['phase_pct'] = 0;
      }
      // Track progress for UI
      if (class_exists('InternalLinksTool_Keywords') && method_exists('InternalLinksTool_Keywords', 'get_progress_counts')) {
        $kw_progress = InternalLinksTool_Keywords::get_progress_counts();
        $bg['phase_total'] = (int)($kw_progress['total'] ?? 0);
        $bg['phase_done']  = (int)($kw_progress['done'] ?? 0);
      }
      update_option(self::$bg_key, $bg, false);
      wp_schedule_single_event(time() + 1, 'internallinkstool_bg_linker_batch');
      spawn_cron();
      return;
    }

    // ── Phase: Keyword planning (deduplication) ──
    if ($phase === 'keyword_planning') {
      $result = InternalLinksTool_Keywords::plan_unique_keywords(3);

      if ($result['status'] === 'done') {
        $bg['phase'] = 'anchor_banks';
        $bg['phase_done'] = 0;
        $bg['phase_total'] = 0;
        $bg['phase_pct'] = 0;
        update_option(self::$bg_key, $bg, false);
        wp_schedule_single_event(time(), 'internallinkstool_bg_linker_batch');
      } else {
        $bg['phase_done']  = $result['resolved'];
        $bg['phase_total'] = $result['total_conflicts'];
        $bg['phase_pct']   = $result['total_conflicts'] > 0
          ? round(100 * $result['resolved'] / $result['total_conflicts'])
          : 100;
        update_option(self::$bg_key, $bg, false);
        wp_schedule_single_event(time() + 5, 'internallinkstool_bg_linker_batch');
      }
      spawn_cron();
      return;
    }

    // ── Phase: Anchor banks generation ──
    // Use small batch for API-heavy phases to avoid PHP timeout
    if ($phase === 'anchor_banks') {
      $api_batch = min($batch_size, 5);
      $ab_done = 0;
      if (class_exists('InternalLinksTool_AnchorBanks') && method_exists('InternalLinksTool_AnchorBanks', 'run_batch')) {
        try {
          $ab_res = InternalLinksTool_AnchorBanks::run_batch($api_batch);
          $ab_done = (int)($ab_res['processed'] ?? 0);
        } catch (Exception $e) {
          $ab_done = 0;
        }
      }

      if ($ab_done === 0) {
        // All anchor banks generated — move to linking phase
        $bg['phase'] = 'linking';
        $bg['phase_total'] = 0;
        $bg['phase_done']  = 0;
        // Now compute total for the linker progress bar
        $progress = self::get_progress_counts($settings);
        $bg['total'] = (int)$progress['total'];
        // Reset pointer for linking
        update_option(self::$ptr_key, 0, false);
      } else {
        // Track progress for UI
        if (class_exists('InternalLinksTool_AnchorBanks') && method_exists('InternalLinksTool_AnchorBanks', 'get_progress_counts')) {
          $ab_progress = InternalLinksTool_AnchorBanks::get_progress_counts();
          $bg['phase_total'] = (int)($ab_progress['total'] ?? 0);
          $bg['phase_done']  = (int)($ab_progress['done'] ?? 0);
        }
      }
      update_option(self::$bg_key, $bg, false);
      wp_schedule_single_event(time() + 1, 'internallinkstool_bg_linker_batch');
      spawn_cron();
      return;
    }

    // ── Phase: Linking (original logic) ──
    try {
      $res = self::run_batch($batch_size, $dry_run);
    } catch (Exception $e) {
      $bg['status'] = 'stopped';
      $bg['error'] = $e->getMessage();
      update_option(self::$bg_key, $bg, false);
      return;
    }

    // Re-read in case it was stopped during run_batch
    $bg = get_option(self::$bg_key, []);
    if (($bg['status'] ?? 'idle') !== 'running') return;

    $bg['processed']      += (int)($res['processed'] ?? 0);
    $bg['updated']        += (int)($res['updated'] ?? 0);
    $bg['inserted_links'] += (int)($res['inserted_links'] ?? 0);

    // Accumulate preview/debug rows from this batch
    if (!empty($res['preview_key'])) {
      $batch_preview = get_transient('internallinkstool_linker_preview_' . $res['preview_key']);
      $batch_debug   = get_transient('internallinkstool_linker_debug_' . $res['preview_key']);

      $acc_preview = get_transient('internallinkstool_bg_linker_preview');
      if (!is_array($acc_preview)) $acc_preview = [];
      $acc_debug = get_transient('internallinkstool_bg_linker_debug');
      if (!is_array($acc_debug)) $acc_debug = [];

      if (is_array($batch_preview)) {
        $acc_preview = array_merge($acc_preview, $batch_preview);
        if (count($acc_preview) > 1000) $acc_preview = array_slice($acc_preview, 0, 1000);
      }
      if (is_array($batch_debug)) {
        $acc_debug = array_merge($acc_debug, $batch_debug);
        if (count($acc_debug) > 1000) $acc_debug = array_slice($acc_debug, 0, 1000);
      }

      set_transient('internallinkstool_bg_linker_preview', $acc_preview, HOUR_IN_SECONDS);
      set_transient('internallinkstool_bg_linker_debug', $acc_debug, HOUR_IN_SECONDS);

      // Cleanup individual batch transients
      delete_transient('internallinkstool_linker_preview_' . $res['preview_key']);
      delete_transient('internallinkstool_linker_debug_' . $res['preview_key']);
    }

    // Check if done (no pages processed = pointer past all docs)
    if ((int)($res['processed'] ?? 0) === 0) {
      $bg['status'] = 'done';
      update_option(self::$bg_key, $bg, false);
      return;
    }

    update_option(self::$bg_key, $bg, false);

    // Schedule next batch
    wp_schedule_single_event(time() + 2, 'internallinkstool_bg_linker_batch');
    spawn_cron();
  }

  /* -------------------------
   * Core batch runner
   * ------------------------- */
  public static function run_batch($batch_size = 10, $dry_run = true) {
    global $wpdb;

    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];
    $max_links_page = (int)($settings['max_links_per_page'] ?? 3);
    if ($max_links_page <= 0) return ['processed'=>0,'updated'=>0,'inserted_links'=>0,'preview_key'=>''];

    $types    = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    [$types_ph, $types_params] = self::in_placeholders($types);
    [$st_ph, $st_params]       = self::in_placeholders($statuses);

    $tables = self::get_tables();
    $docs = $tables['docs'];
    $kws  = $tables['kws'];

    $ptr = (int)get_option(self::$ptr_key, 0);

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Fetch SOURCE docs eligible + with keywords + include meta fields for debug
    $sql = "
      SELECT d.id AS doc_id, d.post_id, d.url, d.type, d.status, d.meta_title, d.meta_desc,
             k.primary_keyword, k.secondary_keywords
      FROM {$docs} d
      INNER JOIN {$kws} k ON k.document_id = d.id
      WHERE d.id > %d
        AND d.type IN {$types_ph}
        AND d.status IN {$st_ph}
        {$robots_sql}
        AND k.primary_keyword IS NOT NULL AND k.primary_keyword <> ''
      ORDER BY d.id ASC
      LIMIT %d
    ";

    $params = array_merge([$ptr], $types_params, $st_params, [(int)$batch_size]);
    $rows = $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);
    if (!is_array($rows)) $rows = [];

    if (empty($rows)) {
      return ['processed'=>0,'updated'=>0,'inserted_links'=>0,'preview_key'=>''];
    }

    $targets = self::get_targets_pool($settings, 1200);

    $processed = 0;
    $updated = 0;
    $inserted_links = 0;
    $preview_rows = [];
    $debug_rows = [];

    foreach ($rows as $r) {
      $doc_id  = (int)$r['doc_id'];
      $post_id = (int)$r['post_id'];
      $processed++;
      $ptr = $doc_id; // always advance pointer

      $post = get_post($post_id);
      if (!$post) continue;

      $permalink = get_permalink($post_id);
      if (!$permalink) continue;

      if (self::is_excluded_url($permalink, $settings)) continue;

      $source_cats = [];
      if (!empty($settings['same_category_only']) && get_post_type($post_id) === 'post') {
        $source_cats = wp_get_post_categories($post_id);
        if (!is_array($source_cats)) $source_cats = [];
      }

      // Fetch Yoast focus keyword for source page (used in preview)
      $source_yoast_fkw = $wpdb->get_var($wpdb->prepare(
        "SELECT meta_value FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key = '_yoast_wpseo_focuskw' LIMIT 1",
        $post_id
      ));

      $source_meta = [
        'meta_title'     => (string)($r['meta_title'] ?? ''),
        'meta_desc'      => (string)($r['meta_desc'] ?? ''),
        'yoast_focus_kw' => (string)($source_yoast_fkw ?: ''),
      ];

      $source_keywords = [
        'primary' => (string)($r['primary_keyword'] ?? ''),
        'secondary' => (string)($r['secondary_keywords'] ?? ''),
      ];

      // Filter targets by source post type
      $source_type = $post->post_type;
      $allowed_target_types = self::get_allowed_target_types($settings, $source_type);
      $filtered_targets = array_values(array_filter($targets, fn($t) => in_array($t['post_type'], $allowed_target_types, true)));

      $unused_counts = [];
      $res = self::insert_links_into_content(
        (string)$post->post_content,
        $permalink,
        $post_id,
        $filtered_targets,
        $settings,
        $source_cats,
        $dry_run,
        $source_meta,
        $source_keywords,
        $unused_counts,
        false, // tentative = false
        []     // no cross-page anchor blocking
      );

      // Debug row (always) - use already-fetched source meta
      $debug_rows[] = [
        'post_id' => $post_id,
        'title' => get_the_title($post_id),
        'url' => $permalink,
        'type' => (string)($r['type'] ?? ''),
        'status' => (string)($r['status'] ?? ''),
        'meta_title' => $source_meta['meta_title'],
        'meta_desc' => $source_meta['meta_desc'],
        'yoast_focus_kw' => $source_meta['yoast_focus_kw'],
        'primary_keyword' => (string)($r['primary_keyword'] ?? ''),
        'secondary_keywords' => (string)($r['secondary_keywords'] ?? ''),
        'proposals' => (int)($res['inserted'] ?? 0),
      ];

      if (!empty($res['preview']) && is_array($res['preview'])) {
        foreach ($res['preview'] as $p) $preview_rows[] = $p;
      }

      if (!$dry_run && !empty($res['changed'])) {
        wp_update_post(['ID'=>$post_id,'post_content'=>$res['html']]);
        $updated++;
      }

      $inserted_links += (int)($res['inserted'] ?? 0);
    }

    update_option(self::$ptr_key, $ptr, false);

    $preview_key = wp_generate_password(10, false, false);
    set_transient('internallinkstool_linker_preview_' . $preview_key, $preview_rows, 15 * MINUTE_IN_SECONDS);
    set_transient('internallinkstool_linker_debug_' . $preview_key, $debug_rows, 15 * MINUTE_IN_SECONDS);

    return [
      'processed'=>$processed,
      'updated'=>$updated,
      'inserted_links'=>$inserted_links,
      'preview_key'=>$preview_key,
      'dedup_count'=>0,
    ];
  }

  private static function get_targets_pool($settings, $limit = 1200) {
    global $wpdb;

    $types    = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);
    [$types_ph, $types_params] = self::in_placeholders($types);
    [$st_ph, $st_params]       = self::in_placeholders($statuses);

    $tables = self::get_tables();
    $docs = $tables['docs'];
    $kws  = $tables['kws'];
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Include tier column if it exists
    $tier_col = '';
    $tier_exists = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if ($tier_exists) {
      $tier_col = ', d.tier';
    }

    $sql = "
      SELECT d.id AS doc_id, d.post_id, d.url, d.meta_title, d.meta_desc,
             k.primary_keyword, k.secondary_keywords {$tier_col}
      FROM {$docs} d
      INNER JOIN {$kws} k ON k.document_id = d.id
      WHERE d.type IN {$types_ph}
        AND d.status IN {$st_ph}
        {$robots_sql}
        AND d.url IS NOT NULL AND d.url <> ''
        AND k.primary_keyword IS NOT NULL AND k.primary_keyword <> ''
      ORDER BY d.id ASC
      LIMIT %d
    ";

    $params = array_merge($types_params, $st_params, [(int)$limit]);
    $rows = $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);
    if (!is_array($rows)) $rows = [];

    // Check if anchor_banks table exists
    $banks_exist = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");

    $targets = [];
    foreach ($rows as $r) {
      $pid = (int)$r['post_id'];
      $doc_id = (int)$r['doc_id'];
      $url = trim((string)$r['url']);
      if ($pid <= 0 || $url === '') continue;

      if (self::is_excluded_url($url, $settings)) continue;
      if (!self::is_included_url($url, $settings)) continue;

      $pt = get_post_type($pid);
      if ($pt === 'post' && empty($settings['include_posts'])) continue;
      if ($pt === 'page' && empty($settings['include_pages'])) continue;

      $cats = [];
      if (!empty($settings['same_category_only']) && $pt === 'post') {
        $cats = wp_get_post_categories($pid);
        if (!is_array($cats)) $cats = [];
      }

      // Parse primary keywords (now comma-separated for multiple primaries)
      $primary_raw = trim((string)($r['primary_keyword'] ?? ''));
      if ($primary_raw === '') continue;

      $primary_arr = array_values(array_filter(array_map('trim', preg_split('/\s*,\s*/', $primary_raw))));
      if (empty($primary_arr)) continue;

      $primary = $primary_arr[0]; // First primary for backwards compatibility
      $primary_count = count($primary_arr); // Track how many primaries for 80/20 logic

      $tier = isset($r['tier']) ? (string)$r['tier'] : 'tier3';
      if ($tier === '') $tier = 'tier3';

      // Try to get anchors from anchor_banks first
      $anchor_bank = [];
      if ($banks_exist) {
        $bank_rows = $wpdb->get_results($wpdb->prepare(
          "SELECT anchor_type, anchor_text, used_count FROM {$banks} WHERE document_id = %d ORDER BY used_count ASC",
          $doc_id
        ), ARRAY_A);

        if (!empty($bank_rows)) {
          foreach ($bank_rows as $br) {
            $atype = (string)$br['anchor_type'];
            $atext = (string)$br['anchor_text'];
            if (!isset($anchor_bank[$atype])) $anchor_bank[$atype] = [];
            $anchor_bank[$atype][] = $atext;
          }
        }
      }

      // Build primary_secondary array: all primaries first, then secondaries
      // This structure allows 80/20 distribution logic to know which are primary vs secondary
      $primary_secondary = [];
      foreach ($primary_arr as $p) {
        if ($p !== '') $primary_secondary[] = $p;
      }

      $secondary_raw = trim((string)($r['secondary_keywords'] ?? ''));
      if ($secondary_raw !== '') {
        $secondary_arr = array_values(array_filter(array_map('trim', preg_split('/\s*,\s*/', $secondary_raw))));
        foreach ($secondary_arr as $sec) {
          if ($sec !== '' && !in_array($sec, $primary_arr)) {
            $primary_secondary[] = $sec;
          }
        }
      }

      $targets[] = [
        'doc_id'     => $doc_id,
        'post_id'    => $pid,
        'post_type'  => $pt,
        'url'        => $url,
        'meta_title' => trim((string)($r['meta_title'] ?? '')),
        'meta_desc'  => trim((string)($r['meta_desc'] ?? '')),
        'tier'       => $tier,
        'anchor_bank'     => $anchor_bank,
        'primary_secondary' => $primary_secondary, // All primaries first, then secondaries
        'primary_count'     => $primary_count,     // How many are primaries (for 80/20 distribution)
        'cats'       => $cats,
      ];
    }

    return $targets;
  }

  private static function is_excluded_url($url, $settings) {
    $url = (string)$url;

    if (!empty($settings['exclude_urls'])) {
      $raw = (string)$settings['exclude_urls'];
      $parts = preg_split('/\r\n|\r|\n|,/', $raw);
      foreach ($parts as $p) {
        $p = trim($p);
        if ($p === '') continue;
        if (strpos($url, $p) !== false) return true;
      }
    }

    if (!empty($settings['exclude_slug_words'])) {
      $parts = preg_split('/[,\n\r]+/', (string)$settings['exclude_slug_words']);
      $parts = array_values(array_filter(array_map('trim', $parts)));
      $slug = strtolower((string)wp_parse_url($url, PHP_URL_PATH));
      foreach ($parts as $w) {
        $w = strtolower(trim($w));
        if ($w === '') continue;
        if (strpos($slug, $w) !== false) return true;
      }
    }

    return false;
  }

  /**
   * Check if URL passes the include-URLs whitelist.
   * Returns true if no whitelist is set, or if the URL matches any entry.
   */
  private static function is_included_url($url, $settings) {
    if (empty($settings['include_urls'])) return true;

    $raw = trim((string)$settings['include_urls']);
    if ($raw === '') return true;

    $url = (string)$url;
    $parts = preg_split('/\r\n|\r|\n|,/', $raw);
    foreach ($parts as $p) {
      $p = trim($p);
      if ($p === '') continue;
      if (strpos($url, $p) !== false) return true;
    }
    return false;
  }

  /**
   * Normalize a URL for self-link comparison.
   * Strips protocol, www, trailing slash so that different representations
   * of the same page are treated as equal.
   */
  private static function normalize_url_for_compare($url) {
    $url = strtolower(trim((string)$url));
    if ($url === '') return '';
    // Strip protocol
    $url = preg_replace('#^https?://#', '', $url);
    // Strip www.
    $url = preg_replace('#^www\.#', '', $url);
    // Strip trailing slash
    $url = rtrim($url, '/');
    return $url;
  }

  /**
   * Calculate topic relevance score between source and target (0-100)
   */
  private static function calculate_relevance_score($source_keywords, $target, $source_cats = []) {
    $score = 0;

    // Get target keywords
    $target_primary = '';
    $target_keywords = [];

    // From anchor bank
    if (!empty($target['anchor_bank'])) {
      foreach ($target['anchor_bank'] as $type => $anchors) {
        foreach ($anchors as $anchor) {
          $target_keywords[] = strtolower(trim($anchor));
          $target_keywords = array_merge($target_keywords, self::extract_significant_words($anchor));
        }
      }
    }

    // From primary/secondary keywords
    if (!empty($target['primary_secondary'])) {
      foreach ($target['primary_secondary'] as $idx => $anchor) {
        $anchor_lower = strtolower(trim($anchor));
        if ($idx === 0) $target_primary = $anchor_lower;
        $target_keywords[] = $anchor_lower;
        $target_keywords = array_merge($target_keywords, self::extract_significant_words($anchor));
      }
    }

    // From meta title and description
    if (!empty($target['meta_title'])) {
      $target_keywords = array_merge($target_keywords, self::extract_significant_words($target['meta_title']));
    }
    if (!empty($target['meta_desc'])) {
      $target_keywords = array_merge($target_keywords, self::extract_significant_words($target['meta_desc']));
    }

    $target_keywords = array_unique(array_filter($target_keywords));

    if (empty($source_keywords) || empty($target_keywords)) {
      return 10; // Minimal score if no keywords to compare
    }

    // 1. Exact keyword match (highest weight)
    $exact_matches = array_intersect($source_keywords, $target_keywords);
    $score += count($exact_matches) * 15; // 15 points per exact match

    // 2. Partial word overlap
    foreach ($source_keywords as $sk) {
      foreach ($target_keywords as $tk) {
        if ($sk === $tk) continue; // Already counted
        // Check if one contains the other
        if (strlen($sk) >= 4 && strlen($tk) >= 4) {
          if (strpos($sk, $tk) !== false || strpos($tk, $sk) !== false) {
            $score += 8; // Partial overlap
          }
        }
      }
    }

    // 3. Category match bonus
    if (!empty($source_cats) && !empty($target['cats'])) {
      $cat_overlap = array_intersect($source_cats, $target['cats']);
      $score += count($cat_overlap) * 20; // 20 points per shared category
    }

    // 4. Same topic cluster (check for common topic words)
    $topic_words = ['guide', 'review', 'best', 'how', 'what', 'why', 'tips', 'vs', 'compare', 'list'];
    foreach ($topic_words as $tw) {
      $in_source = false;
      $in_target = false;
      foreach ($source_keywords as $sk) {
        if (strpos($sk, $tw) !== false) { $in_source = true; break; }
      }
      foreach ($target_keywords as $tk) {
        if (strpos($tk, $tw) !== false) { $in_target = true; break; }
      }
      if ($in_source && $in_target) {
        $score += 5;
      }
    }

    // Cap at 100
    return min(100, $score);
  }

  /**
   * Extract significant words from a keyword phrase
   * Removes common stop words and returns meaningful terms
   */
  private static function extract_significant_words($phrase) {
    $phrase = strtolower(trim($phrase));
    if ($phrase === '') return [];

    // Split into words
    $words = preg_split('/[\s\-\_\.\,\:\;\(\)\[\]\/\|]+/', $phrase);
    $words = array_filter(array_map('trim', $words));

    // Stop words to exclude
    $stop_words = [
      'a', 'an', 'the', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for',
      'of', 'with', 'by', 'from', 'as', 'is', 'are', 'was', 'were', 'be',
      'been', 'being', 'have', 'has', 'had', 'do', 'does', 'did', 'will',
      'would', 'could', 'should', 'may', 'might', 'must', 'shall', 'can',
      'this', 'that', 'these', 'those', 'it', 'its', 'i', 'you', 'we', 'they',
      'he', 'she', 'him', 'her', 'his', 'your', 'our', 'their', 'my',
      'what', 'which', 'who', 'whom', 'whose', 'when', 'where', 'why', 'how',
      'all', 'each', 'every', 'both', 'few', 'more', 'most', 'other', 'some',
      'such', 'no', 'nor', 'not', 'only', 'own', 'same', 'so', 'than', 'too',
      'very', 'just', 'also', 'now', 'here', 'there', 'then', 'once',
    ];

    $significant = [];
    foreach ($words as $w) {
      if (strlen($w) < 3) continue; // Skip very short words
      if (in_array($w, $stop_words, true)) continue;
      $significant[] = $w;
    }

    return $significant;
  }

  private static function anchor_to_pattern($anchor) {
    $anchor = trim((string)$anchor);
    if ($anchor === '') return null;

    $anchor = preg_replace('/\s+/u', ' ', $anchor);
    $words = preg_split('/\s+/u', $anchor);
    $words = array_values(array_filter(array_map('trim', $words)));

    if (empty($words)) return null;

    if (count($words) === 1) {
      $w = preg_quote($words[0], '/');
      return '/(?<![\p{L}\p{N}])' . $w . '(?![\p{L}\p{N}])/iu';
    }

    $escaped = array_map(function($w){ return preg_quote($w, '/'); }, $words);
    $sep = '[\s\-\–\—\.\,\:\;\(\)\[\]\/\|]+';
    $mid = implode($sep, $escaped);

    return '/(?<![\p{L}\p{N}])' . $mid . '(?![\p{L}\p{N}])/iu';
  }

  private static function brand_sub_phrases($primary) {
    $words = preg_split('/\s+/u', trim($primary));
    $words = array_values(array_filter(array_map('trim', $words)));
    if (count($words) < 3) return [];  // Need 3+ words to generate sub-phrases

    // Find the brand using strict detection (not the capitalization fallback)
    $brand_idx = self::find_brand_word_in_phrase($words);
    if ($brand_idx < 0) return [];  // No brand detected, skip sub-phrases

    $brand_word = $words[$brand_idx];
    $after_brand = array_slice($words, $brand_idx + 1);
    if (empty($after_brand)) return [];

    $phrases = [];

    // Brand + progressively fewer trailing words (keep brand + at least 1 other word)
    for ($len = count($after_brand) - 1; $len >= 1; $len--) {
      $phrases[] = $brand_word . ' ' . implode(' ', array_slice($after_brand, 0, $len));
    }

    // Also try brand + last word if not already included
    $brand_last = $brand_word . ' ' . end($after_brand);
    if (!in_array(strtolower($brand_last), array_map('strtolower', $phrases))) {
      $phrases[] = $brand_last;
    }

    return $phrases;
  }

  /**
   * Expand an anchor with 1-2 context words from the surrounding sentence.
   * Uses AI to validate if expansion makes sense from SEO perspective.
   *
   * @param string $anchor The matched anchor text
   * @param string $sentence The full sentence containing the anchor
   * @return string Expanded anchor (or original if expansion not advisable)
   */
  private static function expand_anchor_with_context($anchor, $sentence) {
    $anchor = trim($anchor);
    if ($anchor === '' || $sentence === '') return $anchor;

    // Find anchor position in sentence (case-insensitive)
    $anchor_lower = strtolower($anchor);
    $sentence_lower = strtolower($sentence);
    $pos = strpos($sentence_lower, $anchor_lower);
    if ($pos === false) return $anchor;

    // Get actual anchor from sentence (preserve original case)
    $actual_anchor = substr($sentence, $pos, strlen($anchor));

    // Extract 1-2 words before the anchor
    $before_text = trim(substr($sentence, 0, $pos));
    $words_before = preg_split('/\s+/', $before_text);
    $words_before = array_filter($words_before, fn($w) => preg_match('/^[\p{L}\p{N}\-]+$/u', $w));
    $words_before = array_values($words_before);
    $candidate_before = array_slice($words_before, -2); // Last 2 words

    // Extract 1-2 words after the anchor
    $after_text = trim(substr($sentence, $pos + strlen($anchor)));
    $words_after = preg_split('/\s+/', $after_text);
    $words_after = array_filter($words_after, fn($w) => preg_match('/^[\p{L}\p{N}\-]+$/u', $w));
    $words_after = array_values($words_after);
    $candidate_after = array_slice($words_after, 0, 2); // First 2 words

    // If no candidate words, return original
    if (empty($candidate_before) && empty($candidate_after)) {
      return $actual_anchor;
    }

    // Build possible expansions to evaluate
    $expansions = [];

    // Try adding 1 word before
    if (!empty($candidate_before)) {
      $word = end($candidate_before);
      $expansions[] = $word . ' ' . $actual_anchor;
    }

    // Try adding 1 word after
    if (!empty($candidate_after)) {
      $word = reset($candidate_after);
      $expansions[] = $actual_anchor . ' ' . $word;
    }

    // Try adding 1 word before + 1 after
    if (!empty($candidate_before) && !empty($candidate_after)) {
      $expansions[] = end($candidate_before) . ' ' . $actual_anchor . ' ' . reset($candidate_after);
    }

    // Filter expansions to max 7 words
    $expansions = array_filter($expansions, fn($e) => str_word_count($e) <= 7);

    if (empty($expansions)) {
      return $actual_anchor;
    }

    // Ask AI to pick the best anchor (original or expanded)
    if (class_exists('InternalLinksTool_OpenAI')) {
      $best = InternalLinksTool_OpenAI::pick_best_anchor($actual_anchor, $expansions, $sentence);
      if ($best !== null && $best !== '') {
        return $best;
      }
    }

    return $actual_anchor;
  }

  /**
   * Strict brand detection for sub-phrase generation.
   * Keywords are in Title Case so capitalization alone is unreliable.
   * 1) Check known brand map
   * 2) Look for words that are NOT common English words (likely brand/entity names)
   * Returns word index or -1.
   */
  private static function find_brand_word_in_phrase($words) {
    // 1. Known brand list (same as detect_brand_in_keyword)
    static $brand_map = null;
    if ($brand_map === null) {
      $brand_map = array_flip([
        'amazon','walmart','costco','target','ikea','wayfair','chewy','ebay','etsy','alibaba',
        'samsung','apple','google','microsoft','sony','lg','dell','hp','lenovo','asus','acer',
        'nvidia','amd','intel','qualcomm','bose','jbl','beats','sennheiser','sonos',
        'canon','nikon','fujifilm','gopro','dyson','roomba','irobot','ninja','kitchenaid',
        'cuisinart','keurig','breville','dewalt','makita','bosch','milwaukee','ryobi','craftsman',
        'husqvarna','toyota','honda','ford','bmw','audi','tesla','mercedes','hyundai','kia',
        'chevrolet','subaru','mazda','volvo','lexus','nike','adidas','puma','reebok','yeti',
        'columbia','patagonia','netflix','spotify','disney','hulu','lego','nerf','barbie',
        'hasbro','mattel','nintendo','playstation','xbox','gucci','prada','chanel','zara',
        'philips','braun','gillette','olay','dove','casper','purple','tempur','serta',
        'beautyrest','purina','pedigree','kong',
      ]);
    }
    foreach ($words as $i => $w) {
      if (isset($brand_map[strtolower($w)])) return $i;
    }

    // 2. Common English words — anything NOT in this set is likely a brand/entity
    static $common = null;
    if ($common === null) {
      $common = array_flip([
        // Function words
        'a','an','the','and','or','but','for','to','in','on','at','by','with','from','of',
        'about','as','is','are','was','were','be','been','has','have','had','do','does','did',
        'will','would','can','could','should','may','might','must','shall','not','no','if',
        'then','than','that','this','these','those','it','its','he','she','they','we','you',
        'me','him','her','us','them','my','your','his','our','their','who','what','where',
        'when','how','why','which','whom','whose','all','each','every','both','either',
        'neither','any','some','many','much','more','most','few','less','least','other',
        'another','such','same','only','just','even','also','too','very','quite','really',
        'so','up','down','out','off','over','under','into','onto','upon','vs','versus','per',
        // Common adjectives
        'best','top','good','great','new','old','big','small','large','mini','full','half',
        'long','short','high','low','fast','slow','quick','easy','hard','soft','hot','cold',
        'warm','cool','light','heavy','dark','bright','clean','dirty','fresh','raw','dry',
        'wet','flat','deep','wide','narrow','thin','thick','round','fine','smooth','sharp',
        'loud','quiet','rich','poor','cheap','free','safe','secure','strong','weak','tough',
        'gentle','pure','real','true','false','right','wrong','simple','basic','plain','fancy',
        'modern','classic','vintage','digital','electric','electronic','manual','automatic',
        'smart','wireless','portable','mobile','indoor','outdoor','natural','organic','premium',
        'luxury','standard','regular','normal','average','typical','common','popular',
        'professional','commercial','industrial','residential','personal','private','public',
        'open','closed','single','double','triple','extra','super','ultra','advanced',
        'essential','ultimate','complete','comprehensive','total','overall','general',
        'specific','particular','certain','special','unique','custom','original','traditional',
        'alternative','different','similar','various','main','primary','secondary','key',
        'major','minor','critical','important','significant','reliable','effective','efficient',
        'affordable','expensive','durable','lightweight','waterproof','adjustable','comfortable',
        'functional','practical','ideal','perfect','excellent','superior','improved','updated',
        'certified','approved','tested','proven','recommended','rated','trusted','leading',
        'latest','newest','first','second','third','last','next','final','sized','based',
        'powered','inspired','friendly','proof','resistant','ready','worth',
        // Common nouns
        'human','humans','dog','dogs','cat','cats','pet','pets','animal','animals','baby',
        'kid','kids','child','children','adult','adults','man','woman','men','women','boy',
        'girl','people','person','family','home','house','room','kitchen','bathroom','bedroom',
        'living','office','garden','yard','car','truck','vehicle','bike','food','water','air',
        'fire','earth','wood','metal','plastic','glass','paper','stone','rubber','leather',
        'cotton','fabric','material','size','color','shape','form','style','type','model',
        'kind','sort','way','method','step','process','system','plan','idea','tip','tips',
        'trick','tricks','guide','review','reviews','list','table','chart','set','kit','pack',
        'pair','piece','part','section','unit','item','thing','product','products','tool',
        'tools','device','machine','equipment','gear','supply','supplies','accessory',
        'accessories','feature','features','option','options','choice','benefit','benefits',
        'advantage','use','uses','purpose','function','role','value','quality','level','rate',
        'range','limit','point','line','edge','side','end','center','middle','surface','base',
        'frame','cover','case','bag','box','container','holder','rack','stand','mount',
        'band','belt','strap','chain','cord','wire','cable','plug','switch','button','handle',
        'wheel','door','window','wall','floor','ceiling','roof','shelf','panel','board',
        'screen','display','monitor','speaker','camera','sensor','filter','fan','pump','motor',
        'engine','battery','charger','adapter','power','energy','heat','noise','sound','space',
        'area','zone','spot','place','position','location','distance','weight','height',
        'length','width','depth','capacity','volume','speed','time','hour','minute','day',
        'week','month','year','season','summer','winter','spring','fall','morning','night',
        'price','cost','budget','deal','deals','sale','discount','offer','order','payment',
        'delivery','shipping','return','warranty','service','support','care','maintenance',
        'cleaning','storage','installation','setup','repair','replacement','protection',
        'safety','security','performance','efficiency','comfort','design','look','finish',
        'pattern','texture','bed','beds','chair','chairs','desk','sofa','couch','bench',
        'seat','pillow','cushion','blanket','sheet','mattress','foam','gel','memory','layer',
        'furniture','decor','appliance','appliances','clothing','shoes','wear','health',
        'fitness','beauty','skin','hair','body','eye','teeth','foot','feet','hand','hands',
        'back','neck','head','face','arm','leg','knee','joint','muscle','bone','pain',
        'relief','treatment','therapy','medicine','vitamin','supplement','diet','nutrition',
        'exercise','workout','training','sport','sports','game','games','play','travel',
        'trip','tour','vacation','holiday','camp','camping','hiking','fishing','hunting',
        'garden','gardening','lawn','plant','plants','flower','flowers','tree','seed','soil',
        'pot','pots','fence','gate','outdoor','patio','deck','pool','grill','stove','oven',
        'fridge','freezer','washer','dryer','heater','cooler','humidifier','purifier',
        'vacuum','mop','broom','trash','bin','bag','rack','hook','hanger','basket','tray',
        'mat','rug','carpet','curtain','blind','lamp','bulb','candle','clock','mirror',
        'sign','tag','label','tape','glue','paint','brush','roller','nail','screw','bolt',
        'drill','saw','wrench','hammer','pliers','knife','blade','scissors','tool',
        'computer','laptop','tablet','phone','printer','router','server','software','app',
        'program','code','data','file','network','internet','website','page','post','blog',
        'media','video','audio','music','photo','image','picture','book','magazine',
        'newspaper','report','article','story','content','text','word','words','number',
        'math','science','art','history','education','school','class','course','lesson',
        'degree','job','work','career','business','company','industry','market','trade',
        'shop','store','mall','brand','brands','customer','client','user','buyer','seller',
        'owner','manager','worker','employee','team','group','community','member','partner',
        'country','state','city','town','street','road','park','beach','mountain','river',
        'lake','island','forest','weather','rain','snow','wind','sun','moon','star',
        // Common verbs
        'buy','sell','find','get','use','make','build','create','choose','pick','select',
        'compare','check','test','try','learn','know','see','look','read','watch','work',
        'help','need','want','like','love','save','pay','shop','cook','clean','wash','fix',
        'keep','hold','put','take','give','show','add','change','improve','reduce','increase',
        'start','stop','run','turn','fit','cut','fold','roll','pull','push','lift','drop',
        'hang','place','move','carry','wear','sleep','sit','stand','walk','drive','grow',
        'feed','train','teach','protect','prevent','avoid','handle','manage','control',
        'maintain','remove','replace','install','assemble','adjust','measure','organize',
        'store','convert','connect','attach','go','come','set','bring','send','call','tell',
        'ask','say','think','feel','believe','consider','expect','mean','seem','become',
        'remain','stay','leave','enter','pass','follow','lead','include','offer','provide',
        'require','allow','enable','cause','affect','apply','cover','fill','open','close',
        // Common SEO suffixes / modifiers
        'ideas','pros','cons','comparison','alternatives','near','online','nearby','local',
      ]);
    }

    foreach ($words as $i => $w) {
      $lower = strtolower($w);
      if (mb_strlen($lower) >= 3 && !isset($common[$lower])) return $i;
    }

    return -1;
  }

  /**
   * Calculate specificity score for a keyword/anchor.
   * Higher score = more specific (contains brand/unique terms).
   * Lower score = more generic (common terms only).
   *
   * @param string $text The keyword or anchor text
   * @return int Specificity score: 0 = generic, 1+ = specific (has brand/unique terms)
   */
  private static function calculate_specificity($text) {
    $text = trim($text);
    if ($text === '') return 0;

    $words = preg_split('/\s+/u', $text);
    $words = array_values(array_filter(array_map('trim', $words)));
    if (empty($words)) return 0;

    // Check if any word is a brand/unique term (not common English)
    $brand_idx = self::find_brand_word_in_phrase($words);

    if ($brand_idx >= 0) {
      // Found a brand/unique word - this is specific
      return 1;
    }

    // All words are common - this is generic
    return 0;
  }

  /**
   * Check if anchor-target specificity is aligned.
   * Returns true if the match is appropriate, false if misaligned.
   *
   * Rules:
   * - Generic anchor + Generic target = OK (aligned)
   * - Specific anchor + Specific target = OK (aligned)
   * - Specific anchor + Generic target = OK (specific anchors are always good)
   * - Generic anchor + Specific target = BAD (wasting specific page on generic anchor)
   *
   * @param string $anchor The anchor text
   * @param string $target_primary The target's primary keyword
   * @return bool True if aligned, false if misaligned
   */
  private static function is_specificity_aligned($anchor, $target_primary) {
    $anchor_spec = self::calculate_specificity($anchor);
    $target_spec = self::calculate_specificity($target_primary);

    // Generic anchor (0) + Specific target (1) = misaligned
    if ($anchor_spec === 0 && $target_spec > 0) {
      return false;
    }

    // All other combinations are OK
    return true;
  }

  private static function flexible_anchor_pattern($anchor) {
    $anchor = trim((string)$anchor);
    if ($anchor === '') return null;

    $words = preg_split('/\s+/u', $anchor);
    $words = array_values(array_filter(array_map('trim', $words)));
    if (count($words) < 2 || count($words) > 4) return null;

    $escaped = array_map(function($w){ return preg_quote($w, '/'); }, $words);

    // Separator: whitespace/punctuation, optionally with a filler word
    $filler = '(?:of|from|for|and|the|a|an|in|on|with|by|to|at|or|is|are)';
    $sep = '[\s\-\x{2013}\x{2014}]+(?:' . $filler . '[\s\-\x{2013}\x{2014}]+)?';

    // Generate all permutations of the keyword words
    $perms = self::word_permutations($escaped);
    $alternatives = [];
    foreach ($perms as $perm) {
      $alternatives[] = implode($sep, $perm);
    }

    $core = '(?:' . implode('|', $alternatives) . ')';

    // Context expansion: allow 0-1 words before and 0-1 words after
    $word_pat = '[\p{L}\p{N}\'\-]+';
    $ws = '[\s\-\x{2013}\x{2014}]+';
    $before = '(?:' . $word_pat . $ws . ')?';
    $after = '(?:' . $ws . $word_pat . ')?';

    return '/(?<![\p{L}\p{N}])(' . $before . $core . $after . ')(?![\p{L}\p{N}])/iu';
  }

  private static function word_permutations($arr) {
    if (count($arr) <= 1) return [$arr];
    $result = [];
    foreach ($arr as $i => $item) {
      $rest = array_values(array_diff_key($arr, [$i => '']));
      foreach (self::word_permutations($rest) as $perm) {
        array_unshift($perm, $item);
        $result[] = $perm;
      }
    }
    return $result;
  }

  private static function insert_links_into_content($html, $source_url, $source_post_id, $targets, $settings, $source_cats = [], $dry_run = true, $source_meta = [], $source_keywords = [], &$batch_target_counts = [], $tentative = false, $batch_used_anchors = []) {
    $max_links_page = (int)($settings['max_links_per_page'] ?? 3);
    $max_per_target = (int)($settings['max_links_per_target'] ?? 1);
    $skip_sentences = (int)($settings['skip_sentences'] ?? 2);
    $spread_mode    = (string)($settings['spread_mode'] ?? 'spread');
    $same_cat_only  = !empty($settings['same_category_only']);

    // Get Strategy settings
    $strategy = class_exists('InternalLinksTool_Strategy')
      ? InternalLinksTool_Strategy::get_settings()
      : [];

    // Strategy guardrails (override if set)
    $max_links_per_run = (int)($strategy['max_links_per_run'] ?? $max_links_page);
    if ($max_links_per_run > 0 && $max_links_per_run < $max_links_page) {
      $max_links_page = $max_links_per_run;
    }

    // Minimum relevance score (0-100, 0 = no filtering)
    $min_relevance = (int)($strategy['min_relevance_score'] ?? 20);

    // Tier budgets (percentages)
    $tier_budgets = [
      'homepage' => (int)($strategy['budget_homepage'] ?? 10),
      'tier1'    => (int)($strategy['budget_tier1'] ?? 50),
      'tier2'    => (int)($strategy['budget_tier2'] ?? 30),
      'tier3'    => (int)($strategy['budget_tier3'] ?? 10),
    ];

    // Anchor type mix (percentages)
    $anchor_mix = [
      'exact'       => (int)($strategy['anchor_exact'] ?? 80),
      'partial'     => (int)($strategy['anchor_partial'] ?? 10),
      'descriptive' => (int)($strategy['anchor_descriptive'] ?? 5),
      'contextual'  => (int)($strategy['anchor_contextual'] ?? 5),
      'generic'     => (int)($strategy['anchor_generic'] ?? 0),
    ];

    // Parse source keywords — used ONLY for relevance scoring (NOT as anchor text).
    // Anchor text comes from TARGET page keywords, not the source page.
    $source_primary = trim($source_keywords['primary'] ?? '');
    $source_secondary_raw = trim($source_keywords['secondary'] ?? '');

    // Build list of source keywords for relevance scoring only
    $source_anchor_candidates = [];
    if ($source_primary !== '') {
      $source_anchor_candidates[] = $source_primary;
    }
    if ($source_secondary_raw !== '') {
      $secondary_arr = array_filter(array_map('trim', explode(',', $source_secondary_raw)));
      foreach ($secondary_arr as $sk) {
        if ($sk !== '' && $sk !== $source_primary) {
          $source_anchor_candidates[] = $sk;
        }
      }
    }
    // Sort by length (longer first for better matching)
    usort($source_anchor_candidates, function($a, $b) {
      return strlen($b) <=> strlen($a);
    });

    // Also build lowercase version for relevance scoring
    $source_all_keywords = [];
    foreach ($source_anchor_candidates as $kw) {
      $source_all_keywords[] = strtolower($kw);
      $source_all_keywords = array_merge($source_all_keywords, self::extract_significant_words($kw));
    }
    $source_all_keywords = array_unique(array_filter($source_all_keywords));

    // Normalize source URL for self-link comparison (strip trailing slash, protocol, www)
    $source_url_norm = self::normalize_url_for_compare($source_url);

    // Filter eligible targets and calculate relevance scores
    $eligible_targets = [];
    foreach ($targets as $t) {
      // Block self-links by BOTH post_id AND URL (belt + suspenders)
      if ((int)$t['post_id'] === (int)$source_post_id) continue;
      $target_url_norm = self::normalize_url_for_compare((string)$t['url']);
      if ($target_url_norm !== '' && $target_url_norm === $source_url_norm) continue;
      if ($same_cat_only) {
        if (empty($source_cats) || empty($t['cats'])) continue;
        if (empty(array_intersect($source_cats, $t['cats']))) continue;
      }

      // Brand-in-source requirement: if target primary contains a brand,
      // that brand must exist in the source content
      if (!empty($t['primary_secondary'][0])) {
        $target_primary = $t['primary_secondary'][0];
        $target_brand = self::detect_brand_in_keyword($target_primary);
        if ($target_brand !== '') {
          // Brand detected in target — check if it exists in source content
          if (stripos($html, $target_brand) === false) {
            // Brand not found in source content — skip this target
            continue;
          }
        }
      }

      // Calculate topic relevance score
      $relevance_score = self::calculate_relevance_score($source_all_keywords, $t, $source_cats);
      $t['relevance_score'] = $relevance_score;

      // Filter by minimum relevance if set
      if ($min_relevance > 0 && $relevance_score < $min_relevance) {
        continue;
      }

      $eligible_targets[] = $t;
    }
    if (empty($eligible_targets)) return ['changed'=>false,'html'=>$html,'inserted'=>0,'preview'=>[]];

    // Sort by relevance score (highest first)
    usort($eligible_targets, function($a, $b) {
      return ($b['relevance_score'] ?? 0) <=> ($a['relevance_score'] ?? 0);
    });

    // Build ordered target list — either by tier budgets or purely by relevance
    $ignore_tiering = !empty($strategy['ignore_tiering']);

    if ($ignore_tiering) {
      // No tier logic — just use relevance-sorted targets directly
      $ordered_targets = array_slice($eligible_targets, 0, $max_links_page * 3);
    } else {
      // Group targets by tier for budget-based selection
      $targets_by_tier = ['homepage' => [], 'tier1' => [], 'tier2' => [], 'tier3' => []];
      foreach ($eligible_targets as $t) {
        $tier = $t['tier'] ?? 'tier3';
        if (!isset($targets_by_tier[$tier])) $tier = 'tier3';
        $targets_by_tier[$tier][] = $t;
      }

      // Within each tier: sort by relevance score (highest first).
      foreach ($targets_by_tier as $tier => &$arr) {
        usort($arr, function($a, $b) {
          return ($b['relevance_score'] ?? 0) <=> ($a['relevance_score'] ?? 0);
        });
      }
      unset($arr); // break reference

      // Build ordered target list based on tier budgets
      $ordered_targets = self::order_targets_by_budget($targets_by_tier, $tier_budgets, $max_links_page * 3);
    }

    $doc = new DOMDocument();
    libxml_use_internal_errors(true);
    $wrapped = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>' . $html . '</body></html>';
    $ok = $doc->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();
    if (!$ok) return ['changed'=>false,'html'=>$html,'inserted'=>0,'preview'=>[]];

    $body = $doc->getElementsByTagName('body')->item(0);
    if (!$body) return ['changed'=>false,'html'=>$html,'inserted'=>0,'preview'=>[]];

    $blocks = self::collect_blocks($body);
    if (empty($blocks)) return ['changed'=>false,'html'=>$html,'inserted'=>0,'preview'=>[]];

    $block_count = count($blocks);
    $range = self::spread_ranges($spread_mode, $block_count);
    $block_order = self::order_blocks_by_spread($block_count, $range);

    $per_target_counts = [];
    $used_anchors = []; // Track used anchor texts to prevent duplicates
    $inserted = 0;
    $global_sentence_index = 0;
    $preview = [];
    $exact_match_count = 0;

    $source_title = get_the_title($source_post_id);

    // Tracking for anchor type distribution
    $anchor_type_counts = ['exact' => 0, 'partial' => 0, 'descriptive' => 0, 'contextual' => 0, 'generic' => 0];

    // Tracking for primary/secondary keyword distribution (80/20 rule)
    $primary_anchor_count = 0;
    $secondary_anchor_count = 0;
    // Calculate slots: 80% primary, 20% secondary
    $primary_slots = (int)round($max_links_page * 0.8);
    $secondary_slots = $max_links_page - $primary_slots;
    if ($primary_slots < 1 && $max_links_page > 0) $primary_slots = 1;  // At least 1 primary if we're inserting links

    // Pre-compute exact slot counts per type based on max_links_page and anchor mix.
    // E.g. max=5, 80% exact, 20% partial → exact=4, partial=1
    $anchor_mix_total = array_sum($anchor_mix);
    if ($anchor_mix_total <= 0) $anchor_mix_total = 100;
    $type_slots = [];
    $slots_assigned = 0;
    // Sort by percentage descending so rounding favours the highest-% type
    arsort($anchor_mix);
    foreach ($anchor_mix as $atype => $pct) {
      if ($pct <= 0) {
        $type_slots[$atype] = 0;
        continue;
      }
      $slot = (int)round($max_links_page * $pct / $anchor_mix_total);
      $type_slots[$atype] = $slot;
      $slots_assigned += $slot;
    }
    // Adjust for rounding: add/remove from the highest-% type
    if ($slots_assigned !== $max_links_page && !empty($type_slots)) {
      reset($anchor_mix);
      $top_type = key($anchor_mix);
      $type_slots[$top_type] += ($max_links_page - $slots_assigned);
    }

    foreach ($block_order as $bIndex) {
      if ($inserted >= $max_links_page) break;

      $block = $blocks[$bIndex];

      if (self::block_has_link($block) || self::block_has_image($block)) {
        $global_sentence_index += self::estimate_sentence_count($block->textContent);
        continue;
      }

      // Determine which anchor type is needed next based on Strategy mix settings.
      // Uses slot-based approach: e.g. max=5, 80% exact → 4 exact slots, 1 partial slot.
      $desired_type = self::select_anchor_type_by_mix($anchor_mix, $anchor_type_counts, $inserted, $type_slots);

      // Determine primary/secondary preference based on 80/20 distribution
      // Prefer primary if we haven't filled primary slots yet, prefer secondary if primary is filled
      $prefer_primary = null; // null = no preference
      if ($primary_anchor_count < $primary_slots) {
        $prefer_primary = true;  // Still need more primary anchors
      } elseif ($secondary_anchor_count < $secondary_slots) {
        $prefer_primary = false; // Primary filled, need secondary
      }
      // If both are filled, prefer_primary stays null (use any available)

      // CORRECT APPROACH: Search for TARGET keywords in SOURCE content
      // Anchor text should describe the TARGET (destination), not the source
      // Only candidates matching $desired_type are considered — if none found, skip this block.
      $res = self::try_insert_link_with_target_keywords(
        $doc,
        $block,
        $ordered_targets,
        $per_target_counts,
        $max_per_target,
        $skip_sentences,
        $global_sentence_index,
        $dry_run,
        $settings,
        $used_anchors,
        $desired_type,
        $batch_used_anchors,
        $tentative,
        $prefer_primary
      );

      $global_sentence_index += (int)($res['sentences_seen'] ?? 0);

      if (!empty($res['inserted'])) {
        // ── SELF-LINK SAFETY NET ──
        // Final check: reject if target is the same page as source (should never happen
        // but catches edge-cases like mismatched post_ids or URL variants).
        $target_pid_check = (int)($res['preview_row']['target_post_id'] ?? 0);
        $target_url_check = self::normalize_url_for_compare((string)($res['preview_row']['target_url'] ?? ''));
        if ($target_pid_check === (int)$source_post_id) continue;
        if ($target_url_check !== '' && $target_url_check === $source_url_norm) continue;

        $inserted += (int)$res['inserted'];

        // Track used anchor to prevent duplicates
        if (!empty($res['preview_row']['anchor'])) {
          $used_anchors[strtolower($res['preview_row']['anchor'])] = true;
        }

        // Track anchor type used
        $used_type = $res['anchor_type'] ?? 'exact';
        if (isset($anchor_type_counts[$used_type])) {
          $anchor_type_counts[$used_type]++;
        }
        if ($used_type === 'exact') {
          $exact_match_count++;
        }

        // Track primary/secondary distribution (80/20 rule)
        if (!empty($res['from_primary'])) {
          $primary_anchor_count++;
        } else {
          $secondary_anchor_count++;
        }

        if (!empty($res['preview_row'])) {
          $preview[] = [
            'source_post_id' => (int)$source_post_id,
            'source_title' => (string)$source_title,
            'source_url' => (string)$source_url,
            'source_meta_title' => (string)($source_meta['meta_title'] ?? ''),
            'source_meta_desc' => (string)($source_meta['meta_desc'] ?? ''),
            'source_yoast_fkw' => (string)($source_meta['yoast_focus_kw'] ?? ''),
            'target_post_id' => (int)($res['preview_row']['target_post_id'] ?? 0),
            'target_url' => (string)$res['preview_row']['target_url'],
            'target_meta_title' => (string)($res['preview_row']['target_meta_title'] ?? ''),
            'target_meta_desc' => (string)($res['preview_row']['target_meta_desc'] ?? ''),
            'target_keywords' => isset($res['preview_row']['target_keywords']) ? $res['preview_row']['target_keywords'] : [],
            'target_anchor_bank' => isset($res['preview_row']['target_anchor_bank']) ? $res['preview_row']['target_anchor_bank'] : [],
            'anchor' => (string)$res['preview_row']['anchor'],
            'anchor_type' => (string)($res['preview_row']['anchor_type'] ?? 'exact'),
            'anchor_source' => (string)($res['preview_row']['anchor_source'] ?? 'keyword'),
            'from_primary' => !empty($res['preview_row']['from_primary']),  // 80/20 tracking
            'target_tier' => (string)($res['preview_row']['target_tier'] ?? ''),
            'relevance_score' => (int)($res['preview_row']['relevance_score'] ?? 0),
            'rewritten_anchor' => (string)($res['preview_row']['rewritten_anchor'] ?? $res['preview_row']['anchor']),
            'rewrite_changed' => !empty($res['preview_row']['rewrite_changed']),
            'rewrite_reason' => (string)($res['preview_row']['rewrite_reason'] ?? ''),
            'rewrite_error' => $res['preview_row']['rewrite_error'] ?? null,
            'excerpt' => (string)$res['preview_row']['excerpt'],
          ];
        }
      }
    }

    $new_html = self::inner_html($body);
    $changed = ($new_html !== $html);

    return ['changed'=>$changed,'html'=>$new_html,'inserted'=>$inserted,'preview'=>$preview];
  }

  /**
   * Order targets based on tier budgets
   */
  private static function order_targets_by_budget($targets_by_tier, $tier_budgets, $max_count) {
    $ordered = [];
    $total_budget = array_sum($tier_budgets);
    if ($total_budget <= 0) $total_budget = 100;

    // Calculate how many from each tier based on budget
    $tier_slots = [];
    foreach ($tier_budgets as $tier => $pct) {
      $tier_slots[$tier] = max(1, (int)round(($pct / $total_budget) * $max_count));
    }

    // Round-robin through tiers based on their weight
    $tier_indices = ['homepage' => 0, 'tier1' => 0, 'tier2' => 0, 'tier3' => 0];
    $added = 0;
    $rounds = 0;
    $max_rounds = $max_count * 2;

    while ($added < $max_count && $rounds < $max_rounds) {
      $rounds++;
      foreach (['tier1', 'tier2', 'tier3', 'homepage'] as $tier) {
        if ($added >= $max_count) break;

        $idx = $tier_indices[$tier];
        if ($idx < count($targets_by_tier[$tier]) && $idx < $tier_slots[$tier]) {
          $ordered[] = $targets_by_tier[$tier][$idx];
          $tier_indices[$tier]++;
          $added++;
        }
      }

      // Check if we've exhausted all tiers
      $all_done = true;
      foreach ($tier_indices as $tier => $idx) {
        if ($idx < count($targets_by_tier[$tier])) {
          $all_done = false;
          break;
        }
      }
      if ($all_done) break;
    }

    return $ordered;
  }

  /**
   * Select anchor type based on Strategy mix percentages (slot-based).
   *
   * Uses pre-computed slot counts so the mix works correctly at small numbers.
   * E.g. max 5 links + 80% exact → 4 exact slots, 1 partial slot.
   * Returns the type with the most remaining slots.
   */
  private static function select_anchor_type_by_mix($anchor_mix, $anchor_type_counts, $total_inserted, $type_slots = []) {
    // If pre-computed slots provided, use remaining-slot approach (most accurate)
    if (!empty($type_slots)) {
      $best_type = 'exact';
      $best_remaining = -1;
      foreach ($type_slots as $type => $target_count) {
        $used = (int)($anchor_type_counts[$type] ?? 0);
        $remaining = $target_count - $used;
        if ($remaining > $best_remaining) {
          $best_remaining = $remaining;
          $best_type = $type;
        }
      }
      // If all slots filled, return the type with highest target (allow overflow)
      if ($best_remaining <= 0) {
        arsort($type_slots);
        return key($type_slots);
      }
      return $best_type;
    }

    // Fallback: percentage-based (used when slots not available)
    $best_type = 'exact';
    $best_deficit = -9999;
    $total = max(1, array_sum($anchor_type_counts));

    foreach ($anchor_mix as $type => $target_pct) {
      if ($target_pct <= 0) continue;
      $current_pct = ($total_inserted > 0) ? (($anchor_type_counts[$type] ?? 0) / $total) * 100 : 0;
      $deficit = $target_pct - $current_pct;
      if ($deficit > $best_deficit) {
        $best_deficit = $deficit;
        $best_type = $type;
      }
    }

    return $best_type;
  }

  /**
   * Search for TARGET keywords in SOURCE content.
   * Anchor text describes the TARGET (destination page).
   *
   * Loop order: target (tier-budget) → candidate (primary KW first) → sentence.
   * This ensures Strategy tier ordering is respected AND the best anchor
   * (closest to the Keywords page) is chosen across all sentences.
   *
   * $desired_type: The anchor type the Strategy mix needs next (e.g. 'exact').
   * Only candidates matching this type are tried. If none match, no link is placed.
   */
  private static function try_insert_link_with_target_keywords(
    DOMDocument $doc,
    DOMNode $block,
    $targets,
    &$per_target_counts,
    $max_per_target,
    $skip_sentences,
    $global_sentence_index,
    $dry_run,
    $settings = [],
    $used_anchors = [],
    $desired_type = '',
    $batch_used_anchors = [],
    $tentative = false,
    $prefer_primary = null  // null = no preference, true = prefer primary, false = prefer secondary
  ) {
    $text = trim((string)$block->textContent);
    if ($text === '') return ['inserted'=>0,'sentences_seen'=>0];

    $sentences = self::split_sentences($text);
    if (empty($sentences)) return ['inserted'=>0,'sentences_seen'=>0];

    $sentences_seen = count($sentences);

    if (empty($targets)) {
      return ['inserted'=>0,'sentences_seen'=>$sentences_seen];
    }

    // For each target (already ordered by tier budget + relevance):
    //   Build candidate anchors for THIS target (primary → bank exact → variations → secondary → bank others)
    //   Filter to only candidates matching $desired_type (Strategy mix enforcement)
    //   For each candidate: try ALL eligible sentences
    //   First match wins → link inserted, return immediately
    foreach ($targets as $t) {
      $url = (string)$t['url'];
      $post_id = (int)$t['post_id'];
      $kkey = md5(strtolower($url));

      $count = (int)($per_target_counts[$kkey] ?? 0);
      if ($count >= $max_per_target) continue;

      // Build ordered candidates for this target only
      $candidates = self::build_target_anchor_candidates($t, $used_anchors);

      // Filter candidates to only the desired anchor type (enforce Strategy mix)
      if ($desired_type !== '') {
        $candidates = array_values(array_filter($candidates, function($c) use ($desired_type) {
          return $c['anchor_type'] === $desired_type;
        }));
      }

      // Filter by primary/secondary preference (80/20 distribution)
      // If prefer_primary is true, try primary candidates first
      // If prefer_primary is false, try secondary candidates first
      // If prefer_primary is null, no preference (use all candidates in order)
      if ($prefer_primary === true) {
        // First try only primary candidates
        $primary_candidates = array_values(array_filter($candidates, function($c) {
          return !empty($c['from_primary']);
        }));
        // If we have primary candidates, use only those
        if (!empty($primary_candidates)) {
          $candidates = $primary_candidates;
        }
        // Otherwise fall through to use all candidates
      } elseif ($prefer_primary === false) {
        // First try only secondary candidates
        $secondary_candidates = array_values(array_filter($candidates, function($c) {
          return empty($c['from_primary']);
        }));
        // If we have secondary candidates, use only those
        if (!empty($secondary_candidates)) {
          $candidates = $secondary_candidates;
        }
        // Otherwise fall through to use all candidates
      }

      foreach ($candidates as $c) {
        $anchor = $c['anchor'];
        $anchor_type = $c['anchor_type'];

        if (!empty($c['flexible'])) {
          $pattern = self::flexible_anchor_pattern($anchor);
        } else {
          $pattern = self::anchor_to_pattern($anchor);
        }
        if (!$pattern) continue;

        // Try this candidate in ALL eligible sentences
        for ($si = 0; $si < count($sentences); $si++) {
          $current_global_idx = $global_sentence_index + $si;
          if ($current_global_idx < $skip_sentences) continue;

          $sentence = trim($sentences[$si]);
          if ($sentence === '') continue;

          if (!preg_match($pattern, $sentence, $m)) continue;

          // For flexible matches, use the actual matched text as anchor
          if (!empty($c['flexible']) && !empty($m[1])) {
            $anchor = trim($m[1]);
          }

          // Try to expand anchor with 1-2 context words from the sentence (makes anchor more specific)
          $anchor = self::expand_anchor_with_context($anchor, $sentence);

          // Specificity alignment check: don't waste specific targets on generic anchors
          // Generic anchor + Specific target = skip (prefer finding a specific anchor or a generic target)
          // This ensures branded pages get branded anchors, and generic anchors go to generic pages
          $target_primary = isset($t['primary_secondary'][0]) ? $t['primary_secondary'][0] : '';
          if ($target_primary !== '' && !self::is_specificity_aligned($anchor, $target_primary)) {
            // Anchor is generic but target is specific - skip this match
            // Exception: if this is from secondary keywords (20% quota), allow it
            if (!empty($c['from_primary'])) {
              continue; // Primary match must be specificity-aligned
            }
            // Secondary keywords are allowed to be generic (they're part of the 20% quota)
          }

          // Reject anchors ending in dangling prepositions or conjunctions (fragments like "dog bed for", "scissors and")
          $anchor_words = preg_split('/\s+/', strtolower(trim($anchor)));
          if (count($anchor_words) >= 2) {
            static $bad_endings = ['for', 'to', 'in', 'on', 'at', 'with', 'of', 'by', 'from', 'as', 'into', 'onto', 'upon', 'and', 'or', 'but', 'nor', 'yet', 'so'];
            if (in_array(end($anchor_words), $bad_endings, true)) continue;
          }

          // MATCH FOUND — target keyword (or close variation) exists in source content
          // Safety: double-check this anchor isn't already used on this source page
          $norm_check = strtolower(trim($anchor));
          if (isset($used_anchors[$norm_check])) continue;

          if ($dry_run) {
            $per_target_counts[$kkey] = $count + 1;

            // Anchor rewriting (AI-powered, dry run only)
            $rewritten_anchor = $anchor;
            $rewrite_changed = false;
            $rewrite_reason = '';
            $rewrite_error = null;

            if (!$tentative && !empty($settings['anchor_rewriting']) && class_exists('InternalLinksTool_OpenAI')) {
              $rewrite_result = InternalLinksTool_OpenAI::rewrite_anchor($anchor, $sentence, '');
              if (is_array($rewrite_result)) {
                $rewritten_anchor = (string)($rewrite_result['rewritten'] ?? $anchor);
                $rewrite_changed = !empty($rewrite_result['changed']);
                $rewrite_reason = (string)($rewrite_result['reason'] ?? '');
                $rewrite_error = $rewrite_result['error'] ?? null;
              }
            }

            return [
              'inserted' => 1,
              'sentences_seen' => $sentences_seen,
              'anchor_type' => $anchor_type,
              'from_primary' => !empty($c['from_primary']),  // For 80/20 tracking
              'preview_row' => [
                'target_post_id' => $post_id,
                'target_url' => $url,
                'target_meta_title' => (string)($t['meta_title'] ?? ''),
                'target_meta_desc' => (string)($t['meta_desc'] ?? ''),
                'target_keywords' => isset($t['primary_secondary']) ? $t['primary_secondary'] : [],
                'target_anchor_bank' => isset($t['anchor_bank']) ? $t['anchor_bank'] : [],
                'anchor' => $anchor,
                'anchor_type' => $anchor_type,
                'anchor_source' => !empty($c['from_bank']) ? 'bank' : 'keyword',
                'from_primary' => !empty($c['from_primary']),  // For display
                'target_tier' => (string)($t['tier'] ?? 'tier3'),
                'relevance_score' => (int)($t['relevance_score'] ?? 0),
                'rewritten_anchor' => $rewritten_anchor,
                'rewrite_changed' => $rewrite_changed,
                'rewrite_reason' => $rewrite_reason,
                'rewrite_error' => $rewrite_error,
                'excerpt' => mb_substr($sentence, 0, 220),
              ],
            ];
          }

          // Actually insert the link
          $link_title = !empty($settings['link_title_attr']) ? $anchor : '';
          $ok = self::replace_first_keyword_in_block($doc, $block, $pattern, $url, $link_title);
          if ($ok) {
            $per_target_counts[$kkey] = $count + 1;

            // Update anchor bank used_count
            if (!empty($t['doc_id']) && !empty($c['from_bank']) && class_exists('InternalLinksTool_DB')) {
              global $wpdb;
              $banks = InternalLinksTool_DB::table('anchor_banks');
              $wpdb->query($wpdb->prepare(
                "UPDATE {$banks} SET used_count = used_count + 1 WHERE document_id = %d AND anchor_type = %s AND anchor_text = %s",
                (int)$t['doc_id'], $anchor_type, $anchor
              ));
            }

            return [
              'inserted' => 1,
              'sentences_seen' => $sentences_seen,
              'anchor_type' => $anchor_type,
              'from_primary' => !empty($c['from_primary']),  // For 80/20 tracking
              'preview_row' => [
                'target_post_id' => $post_id,
                'target_url' => $url,
                'target_meta_title' => (string)($t['meta_title'] ?? ''),
                'target_meta_desc' => (string)($t['meta_desc'] ?? ''),
                'target_keywords' => isset($t['primary_secondary']) ? $t['primary_secondary'] : [],
                'target_anchor_bank' => isset($t['anchor_bank']) ? $t['anchor_bank'] : [],
                'anchor' => $anchor,
                'anchor_type' => $anchor_type,
                'anchor_source' => !empty($c['from_bank']) ? 'bank' : 'keyword',
                'from_primary' => !empty($c['from_primary']),  // For display
                'target_tier' => (string)($t['tier'] ?? 'tier3'),
                'relevance_score' => (int)($t['relevance_score'] ?? 0),
                'excerpt' => mb_substr($sentence, 0, 220),
              ],
            ];
          }
        }
      }
    }

    return ['inserted'=>0,'sentences_seen'=>$sentences_seen,'from_primary'=>null];
  }

  /**
   * Build ordered anchor candidates for a single target.
   *
   * Priority order (interleaves anchor bank with keywords for balanced selection):
   *   1. Primary keywords (exact) — highest priority (now supports multiple primaries)
   *   1b. Primary plural/singular (exact)
   *   1c. Primary flexible (word reorder + context expansion, partial)
   *   1d. Primary brand sub-phrases
   *   2. Anchor bank "exact" entries (AI-optimized exact-match anchors)
   *   3. Long secondary keywords 3+ words (partial) — specific enough to be useful
   *   4. Anchor bank "partial" entries
   *   5. Short secondary keywords 1-2 words (partial) — generic, lower priority
   *   6. Anchor bank "descriptive" entries
   *   7. Anchor bank "contextual" entries
   *   8. Anchor bank "generic" entries
   *
   * Candidates include 'from_primary' flag for 80/20 distribution tracking.
   * Minimum anchor length: single-word anchors under 5 chars are skipped (too generic).
   */
  private static function build_target_anchor_candidates($t, $used_anchors = []) {
    $candidates = [];
    $seen = []; // deduplicate anchors (case-insensitive)

    // How many keywords are primaries (rest are secondaries)
    $primary_count = isset($t['primary_count']) ? (int)$t['primary_count'] : 1;

    // Minimum quality filter: skip very short single-word anchors (e.g. "bed", "dog")
    $add = function($anchor, $type, $from_bank = false, $from_primary = true) use (&$candidates, &$seen, $used_anchors) {
      $anchor = trim($anchor);
      if ($anchor === '') return;
      $lower = strtolower($anchor);
      if (isset($seen[$lower])) return;
      if (isset($used_anchors[$lower])) return;

      // Skip single-word anchors under 5 characters (too generic to be useful)
      $word_count = str_word_count($anchor);
      if ($word_count <= 1 && mb_strlen($anchor) < 5) return;

      // Skip anchors ending in dangling prepositions or conjunctions (fragments like "dog bed for", "scissors and")
      static $bad_endings = ['for', 'to', 'in', 'on', 'at', 'with', 'of', 'by', 'from', 'as', 'into', 'onto', 'upon', 'and', 'or', 'but', 'nor', 'yet', 'so'];
      if ($word_count >= 2) {
        $words = preg_split('/\s+/', $lower);
        $last_word = end($words);
        if (in_array($last_word, $bad_endings, true)) return;
      }

      $seen[$lower] = true;
      $candidates[] = ['anchor' => $anchor, 'anchor_type' => $type, 'from_bank' => $from_bank, 'from_primary' => $from_primary];
    };

    $add_flexible = function($anchor, $type, $from_primary = true) use (&$candidates, &$seen, $used_anchors) {
      $anchor = trim($anchor);
      if ($anchor === '') return;
      $key = 'flexible:' . strtolower($anchor);
      if (isset($seen[$key])) return;
      $seen[$key] = true;
      $candidates[] = ['anchor' => $anchor, 'anchor_type' => $type, 'from_bank' => false, 'flexible' => true, 'from_primary' => $from_primary];
    };

    // ── 1. Primary keywords — full form (exact, highest priority) ──
    // Now supports multiple primaries (indices 0 to primary_count-1)
    $brand = ''; // Detected brand name for partial anchor prioritization
    for ($pi = 0; $pi < $primary_count && $pi < count($t['primary_secondary'] ?? []); $pi++) {
      $primary = trim($t['primary_secondary'][$pi] ?? '');
      if ($primary === '') continue;

      // Detect brand from first primary
      if ($pi === 0) {
        $brand = self::detect_brand_in_keyword($primary);
      }

      $add($primary, 'exact', false, true);

      // Plural/singular variation of primary (e.g. "dog bed" ↔ "dog beds")
      $pv = self::plural_singular_variation($primary);
      if ($pv) $add($pv, 'exact', false, true);

      // Primary keyword flexible match (reordered words + context)
      if (str_word_count($primary) >= 2 && str_word_count($primary) <= 4) {
        $add_flexible($primary, 'exact', true);
      }

      // Primary keyword brand-preserving sub-phrases (only for first primary to avoid duplicates)
      if ($pi === 0 && str_word_count($primary) >= 3) {
        $sub_phrases = self::brand_sub_phrases($primary);
        foreach ($sub_phrases as $sp) {
          $add($sp, 'exact', false, true);
        }
      }
    }

    // ── 2. Anchor bank "exact" entries (AI-generated exact-match anchors) ──
    // Treat anchor bank entries as primary-derived (they're based on primary keyword)
    if (!empty($t['anchor_bank']['exact'])) {
      foreach ($t['anchor_bank']['exact'] as $bank_anchor) {
        $add($bank_anchor, 'exact', true, true);
      }
    }

    // ── 3. Secondary keywords + their plural/singular ──
    // Long secondaries (3+ words) first, then short (1-2 words)
    $short_secondaries = [];
    if (!empty($t['primary_secondary'])) {
      for ($i = $primary_count; $i < count($t['primary_secondary']); $i++) {
        $secondary = trim($t['primary_secondary'][$i]);
        if ($secondary === '') continue;

        $sec_words = preg_split('/\s+/', $secondary);
        if (count($sec_words) >= 3) {
          $add($secondary, 'partial', false, false);  // from_primary = false
          // Plural/singular of secondary
          $sv = self::plural_singular_variation($secondary);
          if ($sv) $add($sv, 'exact', false, false);
        } else {
          $short_secondaries[] = $secondary;
        }
      }
    }

    // ── 4. Anchor bank "partial" entries ──
    // ── 4. Anchor bank "partial" entries (derived from primary, so from_primary=true) ──
    if (!empty($t['anchor_bank']['partial'])) {
      foreach ($t['anchor_bank']['partial'] as $bank_anchor) {
        $add($bank_anchor, 'partial', true, true);
      }
    }

    // ── 5. Short secondary keywords 1-2 words (partial — generic, lower priority) ──
    foreach ($short_secondaries as $secondary) {
      $add($secondary, 'partial', false, false);  // from_primary = false
      // Plural/singular of short secondary
      $sv = self::plural_singular_variation($secondary);
      if ($sv) $add($sv, 'exact', false, false);
    }

    // ── 6. Anchor bank "descriptive" entries (less specific, from_primary=false) ──
    if (!empty($t['anchor_bank']['descriptive'])) {
      foreach ($t['anchor_bank']['descriptive'] as $bank_anchor) {
        $add($bank_anchor, 'descriptive', true, false);
      }
    }

    // ── 7. Anchor bank "contextual" entries (less specific, from_primary=false) ──
    if (!empty($t['anchor_bank']['contextual'])) {
      foreach ($t['anchor_bank']['contextual'] as $bank_anchor) {
        $add($bank_anchor, 'contextual', true, false);
      }
    }

    // ── 8. Anchor bank "generic" entries (least specific, from_primary=false) ──
    if (!empty($t['anchor_bank']['generic'])) {
      foreach ($t['anchor_bank']['generic'] as $bank_anchor) {
        $add($bank_anchor, 'generic', true, false);
      }
    }

    // Brand prioritization: within 'partial' candidates, move brand-containing ones first
    if ($brand !== '') {
      $brand_lower = strtolower($brand);
      $result = [];
      $brand_partials = [];
      $other_partials = [];
      $partials_inserted = false;

      foreach ($candidates as $c) {
        if ($c['anchor_type'] !== 'partial') {
          if (!$partials_inserted && (!empty($brand_partials) || !empty($other_partials))) {
            foreach ($brand_partials as $bp) $result[] = $bp;
            foreach ($other_partials as $op) $result[] = $op;
            $partials_inserted = true;
          }
          $result[] = $c;
        } elseif (stripos($c['anchor'], $brand_lower) !== false) {
          $brand_partials[] = $c;
        } else {
          $other_partials[] = $c;
        }
      }
      // Flush remaining partials (if candidates end with partials)
      if (!$partials_inserted) {
        foreach ($brand_partials as $bp) $result[] = $bp;
        foreach ($other_partials as $op) $result[] = $op;
      }

      $candidates = $result;
    }

    return $candidates;
  }

  /**
   * Detect a brand name within a keyword phrase.
   * Checks against a known brand list, then falls back to capitalization
   * (AI is instructed to only capitalize proper nouns/brands).
   * Returns lowercase brand name or empty string.
   */
  private static function detect_brand_in_keyword($keyword) {
    $keyword = trim($keyword);
    if ($keyword === '') return '';

    $words = preg_split('/\s+/', $keyword);
    if (count($words) < 2) return ''; // Single-word keyword = no separable brand

    static $brand_map = null;
    if ($brand_map === null) {
      $brand_map = array_flip([
        // E-commerce & retail
        'amazon', 'walmart', 'costco', 'target', 'ikea', 'wayfair', 'chewy', 'ebay', 'etsy', 'alibaba',
        // Tech & electronics
        'samsung', 'apple', 'google', 'microsoft', 'sony', 'lg', 'dell', 'hp', 'lenovo', 'asus', 'acer',
        'nvidia', 'amd', 'intel', 'qualcomm',
        // Audio
        'bose', 'jbl', 'beats', 'sennheiser', 'sonos',
        // Camera
        'canon', 'nikon', 'fujifilm', 'gopro',
        // Home & kitchen
        'dyson', 'roomba', 'irobot', 'ninja', 'kitchenaid', 'cuisinart', 'keurig', 'breville',
        // Tools
        'dewalt', 'makita', 'bosch', 'milwaukee', 'ryobi', 'craftsman', 'husqvarna',
        // Automotive
        'toyota', 'honda', 'ford', 'bmw', 'audi', 'tesla', 'mercedes', 'hyundai', 'kia', 'chevrolet', 'subaru', 'mazda', 'volvo', 'lexus',
        // Sports & outdoor
        'nike', 'adidas', 'puma', 'reebok', 'yeti', 'columbia', 'patagonia',
        // Streaming
        'netflix', 'spotify', 'disney', 'hulu',
        // Toys & games
        'lego', 'nerf', 'barbie', 'hasbro', 'mattel', 'nintendo', 'playstation', 'xbox',
        // Fashion
        'gucci', 'prada', 'chanel', 'zara',
        // Personal care
        'philips', 'braun', 'gillette', 'olay', 'dove',
        // Furniture & mattress
        'casper', 'purple', 'tempur', 'serta', 'beautyrest',
        // Pet
        'purina', 'pedigree', 'kong',
      ]);
    }

    // Check each word against brand list (case-insensitive)
    foreach ($words as $w) {
      if (isset($brand_map[strtolower($w)])) return strtolower($w);
    }

    // Fallback: AI only capitalizes proper nouns/brands,
    // so a capitalized word in the keyword is likely a brand
    foreach ($words as $w) {
      if (preg_match('/^[A-Z]/', $w) && mb_strlen($w) >= 2) return strtolower($w);
    }

    return '';
  }

  /**
   * Generate plural ↔ singular variation of a phrase (changes last word only).
   * Returns null if no variation could be generated.
   */
  private static function plural_singular_variation($phrase) {
    $words = preg_split('/\s+/', trim($phrase));
    if (empty($words)) return null;

    $last = $words[count($words) - 1];
    $last_lower = strtolower($last);
    $new_last = null;

    // Plural → singular
    if (strlen($last_lower) > 4 && preg_match('/ies$/i', $last_lower)) {
      $new_last = substr($last, 0, -3) . 'y';        // berries → berry
    } elseif (strlen($last_lower) > 4 && preg_match('/ves$/i', $last_lower)) {
      $new_last = substr($last, 0, -3) . 'fe';       // knives → knife
    } elseif (strlen($last_lower) > 4 && preg_match('/ses$/i', $last_lower)) {
      $new_last = substr($last, 0, -2);               // cases → cas — handled below
    } elseif (strlen($last_lower) > 3 && preg_match('/[^s]s$/i', $last_lower)) {
      $new_last = substr($last, 0, -1);               // beds → bed
    }
    // Singular → plural
    elseif (preg_match('/[^s]$/i', $last_lower)) {
      if (preg_match('/[^aeiou]y$/i', $last_lower)) {
        $new_last = substr($last, 0, -1) . 'ies';    // berry → berries
      } elseif (preg_match('/(s|sh|ch|x|z)$/i', $last_lower)) {
        $new_last = $last . 'es';                     // brush → brushes
      } else {
        $new_last = $last . 's';                      // bed → beds
      }
    }

    if ($new_last === null || strtolower($new_last) === $last_lower) return null;

    $words[count($words) - 1] = $new_last;
    return implode(' ', $words);
  }

  private static function collect_blocks(DOMNode $root) {
    $blocks = [];
    $walker = function($node) use (&$blocks, &$walker) {
      if (!$node) return;

      if ($node->nodeType === XML_ELEMENT_NODE) {
        $tag = strtolower($node->nodeName);

        if (in_array($tag, ['h1','h2','h3','h4','h5','h6','script','style'], true)) return;

        if (in_array($tag, ['p','li','blockquote','td','th'], true)) {
          $blocks[] = $node;
          return;
        }

        if ($node->hasChildNodes()) {
          foreach ($node->childNodes as $ch) $walker($ch);
        }
      }
    };
    $walker($root);
    return $blocks;
  }

  private static function block_has_link(DOMNode $block) {
    $xpath = new DOMXPath($block->ownerDocument);
    $nodes = $xpath->query('.//a', $block);
    return ($nodes && $nodes->length > 0);
  }

  private static function block_has_image(DOMNode $block) {
    $xpath = new DOMXPath($block->ownerDocument);
    $nodes = $xpath->query('.//img', $block);
    return ($nodes && $nodes->length > 0);
  }

  /**
   * Try to insert one link using anchor banks (v2)
   */
  private static function try_insert_one_link_in_block_v2(
    DOMDocument $doc,
    DOMNode $block,
    $targets,
    &$per_target_counts,
    $max_per_target,
    $skip_sentences,
    $global_sentence_index,
    $dry_run,
    $settings = [],
    $preferred_anchor_type = 'partial',
    $exact_match_count = 0
  ) {
    $text = trim((string)$block->textContent);
    if ($text === '') return ['inserted'=>0,'sentences_seen'=>0];

    $sentences = self::split_sentences($text);
    if (empty($sentences)) return ['inserted'=>0,'sentences_seen'=>0];

    $sentences_seen = count($sentences);

    // Track available anchors per target for debugging
    $target_anchor_summary = [];

    // Build candidates from anchor banks or fallback anchors
    $candidates = [];
    foreach ($targets as $t) {
      $url = (string)$t['url'];
      $post_id = (int)$t['post_id'];
      $doc_id = (int)($t['doc_id'] ?? 0);
      $tier = (string)($t['tier'] ?? 'tier3');
      $meta_title = (string)($t['meta_title'] ?? '');
      $meta_desc = (string)($t['meta_desc'] ?? '');
      $relevance_score = (int)($t['relevance_score'] ?? 0);

      // Check if this target has anchor banks
      $anchor_bank = isset($t['anchor_bank']) && is_array($t['anchor_bank']) ? $t['anchor_bank'] : [];

      // Track anchor counts per type for this target
      $anchor_counts = ['exact' => 0, 'partial' => 0, 'descriptive' => 0, 'contextual' => 0, 'generic' => 0];

      if (!empty($anchor_bank)) {
        // Count anchors by type
        foreach ($anchor_bank as $atype => $anchors) {
          if (isset($anchor_counts[$atype])) {
            $anchor_counts[$atype] = count($anchors);
          }
        }

        // Use anchor bank - prioritize preferred type
        $type_order = self::get_anchor_type_order($preferred_anchor_type, false);

        foreach ($type_order as $atype) {
          if (!isset($anchor_bank[$atype])) continue;
          foreach ($anchor_bank[$atype] as $anchor) {
            $anchor = trim((string)$anchor);
            if ($anchor === '') continue;
            $candidates[] = [
              'post_id'     => $post_id,
              'doc_id'      => $doc_id,
              'url'         => $url,
              'meta_title'  => $meta_title,
              'meta_desc'   => $meta_desc,
              'anchor'      => $anchor,
              'anchor_type' => $atype,
              'tier'        => $tier,
              'relevance_score' => $relevance_score,
              'anchor_counts' => $anchor_counts,
            ];
          }
        }
      } else {
        // Fallback to keywords (treat as exact/partial)
        $fallback = isset($t['primary_secondary']) && is_array($t['primary_secondary']) ? $t['primary_secondary'] : [];
        $anchor_counts['exact'] = count($fallback) > 0 ? 1 : 0;
        $anchor_counts['partial'] = max(0, count($fallback) - 1);

        foreach ($fallback as $idx => $anchor) {
          $anchor = trim((string)$anchor);
          if ($anchor === '') continue;
          // First is exact, rest are partial
          $atype = ($idx === 0) ? 'exact' : 'partial';

          $candidates[] = [
            'post_id'     => $post_id,
            'doc_id'      => $doc_id,
            'url'         => $url,
            'meta_title'  => $meta_title,
            'meta_desc'   => $meta_desc,
            'anchor'      => $anchor,
            'anchor_type' => $atype,
            'tier'        => $tier,
            'relevance_score' => $relevance_score,
            'anchor_counts' => $anchor_counts,
          ];
        }
      }
    }

    // Sort by anchor length (longer first for better matching)
    usort($candidates, function($x, $y){
      return strlen((string)$y['anchor']) <=> strlen((string)$x['anchor']);
    });

    for ($i = 0; $i < count($sentences); $i++) {
      $current_global_idx = $global_sentence_index + $i;
      if ($current_global_idx < $skip_sentences) continue;

      $sentence = trim($sentences[$i]);
      if ($sentence === '') continue;

      foreach ($candidates as $c) {
        $url = (string)$c['url'];
        $anchor = (string)$c['anchor'];
        $anchor_type = (string)($c['anchor_type'] ?? 'exact');
        if ($url === '' || $anchor === '') continue;

        $kkey = md5(strtolower($url));
        $count = (int)($per_target_counts[$kkey] ?? 0);
        if ($count >= $max_per_target) continue;

        $pattern = self::anchor_to_pattern($anchor);
        if (!$pattern || !preg_match($pattern, $sentence)) continue;

        if ($dry_run) {
          $per_target_counts[$kkey] = $count + 1;

          // Anchor rewriting (AI-powered, dry run only)
          $rewritten_anchor = $anchor;
          $rewrite_changed = false;
          $rewrite_reason = '';
          $rewrite_error = null;

          if (!empty($settings['anchor_rewriting']) && class_exists('InternalLinksTool_OpenAI')) {
            $rewrite_result = InternalLinksTool_OpenAI::rewrite_anchor($anchor, $sentence, '');
            if (is_array($rewrite_result)) {
              $rewritten_anchor = (string)($rewrite_result['rewritten'] ?? $anchor);
              $rewrite_changed = !empty($rewrite_result['changed']);
              $rewrite_reason = (string)($rewrite_result['reason'] ?? '');
              $rewrite_error = $rewrite_result['error'] ?? null;
            }
          }

          // Get anchor counts for this target
          $anchor_counts = isset($c['anchor_counts']) ? $c['anchor_counts'] : [];

          return [
            'inserted' => 1,
            'sentences_seen' => $sentences_seen,
            'anchor_type' => $anchor_type,
            'preview_row' => [
              'target_post_id' => (int)$c['post_id'],
              'target_url' => $url,
              'target_meta_title' => (string)$c['meta_title'],
              'target_meta_desc' => (string)$c['meta_desc'],
              'anchor' => $anchor,
              'anchor_type' => $anchor_type,
              'anchor_source' => !empty($c['from_bank']) ? 'bank' : 'keyword',
              'target_tier' => (string)$c['tier'],
              'relevance_score' => (int)$c['relevance_score'],
              'anchor_counts' => $anchor_counts,
              'rewritten_anchor' => $rewritten_anchor,
              'rewrite_changed' => $rewrite_changed,
              'rewrite_reason' => $rewrite_reason,
              'rewrite_error' => $rewrite_error,
              'excerpt' => mb_substr($sentence, 0, 220),
            ],
          ];
        }

        $link_title = !empty($settings['link_title_attr']) ? $anchor : '';
        $ok = self::replace_first_keyword_in_block($doc, $block, $pattern, $url, $link_title);
        if ($ok) {
          $per_target_counts[$kkey] = $count + 1;

          // Update anchor bank used_count if we have doc_id
          if (!empty($c['doc_id']) && class_exists('InternalLinksTool_DB')) {
            global $wpdb;
            $banks = InternalLinksTool_DB::table('anchor_banks');
            $wpdb->query($wpdb->prepare(
              "UPDATE {$banks} SET used_count = used_count + 1 WHERE document_id = %d AND anchor_type = %s AND anchor_text = %s",
              $c['doc_id'], $anchor_type, $anchor
            ));
          }

          return [
            'inserted' => 1,
            'sentences_seen' => $sentences_seen,
            'anchor_type' => $anchor_type,
            'preview_row' => [
              'target_post_id' => (int)$c['post_id'],
              'target_url' => $url,
              'target_meta_title' => (string)$c['meta_title'],
              'target_meta_desc' => (string)$c['meta_desc'],
              'anchor' => $anchor,
              'anchor_type' => $anchor_type,
              'anchor_source' => !empty($c['from_bank']) ? 'bank' : 'keyword',
              'target_tier' => (string)$c['tier'],
              'excerpt' => mb_substr($sentence, 0, 220),
            ],
          ];
        }
      }
    }

    return ['inserted'=>0,'sentences_seen'=>$sentences_seen];
  }

  /**
   * Get anchor type order based on preference
   */
  private static function get_anchor_type_order($preferred, $skip_exact = false) {
    $all_types = ['exact', 'partial', 'descriptive', 'contextual', 'generic'];

    // Remove exact if we need to skip it
    if ($skip_exact) {
      $all_types = array_diff($all_types, ['exact']);
    }

    // Put preferred first
    $order = [$preferred];
    foreach ($all_types as $type) {
      if ($type !== $preferred) {
        $order[] = $type;
      }
    }

    return array_values(array_unique($order));
  }

  /**
   * Legacy method - kept for backwards compatibility
   */
  private static function try_insert_one_link_in_block(
    DOMDocument $doc,
    DOMNode $block,
    $targets,
    &$per_target_counts,
    $max_per_target,
    $skip_sentences,
    $global_sentence_index,
    $dry_run,
    $settings = []
  ) {
    // Redirect to v2 with defaults
    return self::try_insert_one_link_in_block_v2(
      $doc, $block, $targets, $per_target_counts, $max_per_target,
      $skip_sentences, $global_sentence_index, $dry_run, $settings,
      'partial', 1, 0
    );
  }

  private static function replace_first_keyword_in_block(DOMDocument $doc, DOMNode $block, $pattern, $url, $title = '') {
    $textNodes = self::get_text_nodes($block);

    foreach ($textNodes as $tn) {
      $val = (string)$tn->nodeValue;
      if ($val === '') continue;

      if (!preg_match($pattern, $val, $m, PREG_OFFSET_CAPTURE)) continue;

      $match = $m[0][0];
      $pos = (int)$m[0][1];

      $before = substr($val, 0, $pos);
      $after  = substr($val, $pos + strlen($match));

      $frag = $doc->createDocumentFragment();
      if ($before !== '') $frag->appendChild($doc->createTextNode($before));

      $a = $doc->createElement('a', $match);
      $a->setAttribute('href', esc_url($url));
      if ($title !== '') $a->setAttribute('title', $title);
      $a->setAttribute('rel', 'noopener');
      $frag->appendChild($a);

      if ($after !== '') $frag->appendChild($doc->createTextNode($after));

      $tn->parentNode->replaceChild($frag, $tn);
      return true;
    }

    return false;
  }

  private static function get_text_nodes(DOMNode $node) {
    $out = [];
    $walker = function($n) use (&$out, &$walker) {
      if ($n->nodeType === XML_TEXT_NODE) {
        $out[] = $n;
        return;
      }
      if ($n->nodeType === XML_ELEMENT_NODE) {
        $tag = strtolower($n->nodeName);
        if (in_array($tag, ['h1','h2','h3','h4','h5','h6','a','script','style','img'], true)) return;
        if ($n->hasChildNodes()) foreach ($n->childNodes as $ch) $walker($ch);
      }
    };
    $walker($node);
    return $out;
  }

  private static function split_sentences($text) {
    $text = trim((string)$text);
    if ($text === '') return [];
    $parts = preg_split('/(?<=[\.\!\?])\s+|\n+/', $text);
    return array_values(array_filter(array_map('trim', $parts), fn($x) => $x !== ''));
  }

  private static function estimate_sentence_count($text) {
    return count(self::split_sentences((string)$text));
  }

  private static function spread_ranges($mode, $count) {
    if ($count <= 0) return [0, -1];
    if ($mode === 'start')  return [0, max(0, (int)floor($count * 0.40))];
    if ($mode === 'middle') return [max(0, (int)floor($count * 0.30)), max(0, (int)floor($count * 0.70))];
    if ($mode === 'end')    return [max(0, (int)floor($count * 0.60)), $count - 1];
    return [0, $count - 1];
  }

  private static function order_blocks_by_spread($count, $range) {
    $order = [];
    $start = (int)$range[0];
    $end   = (int)$range[1];
    if ($end < $start) { $start = 0; $end = $count - 1; }

    for ($i = $start; $i <= $end; $i++) $order[] = $i;
    for ($i = 0; $i < $start; $i++) $order[] = $i;
    for ($i = $end + 1; $i < $count; $i++) $order[] = $i;

    return array_values(array_unique($order));
  }

  private static function inner_html(DOMNode $node) {
    $html = '';
    foreach ($node->childNodes as $child) $html .= $node->ownerDocument->saveHTML($child);
    return $html;
  }

  private static function get_progress_counts($settings) {
    global $wpdb;

    $types    = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);
    [$types_ph, $types_params] = self::in_placeholders($types);
    [$st_ph, $st_params]       = self::in_placeholders($statuses);

    $tables = self::get_tables();
    $docs = $tables['docs'];
    $kws  = $tables['kws'];

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    $sql = "
      SELECT COUNT(*)
      FROM {$docs} d
      INNER JOIN {$kws} k ON k.document_id = d.id
      WHERE d.type IN {$types_ph}
        AND d.status IN {$st_ph}
        {$robots_sql}
        AND k.primary_keyword IS NOT NULL AND k.primary_keyword <> ''
    ";

    $params = array_merge($types_params, $st_params);
    $total = (int)$wpdb->get_var($wpdb->prepare($sql, $params));

    return ['total'=>$total];
  }

  /**
   * Deduplicate anchor texts across sources in a batch.
   * Groups proposals by normalized anchor text. Single-entry groups pass through.
   * Multi-entry groups: gap > 10 → highest relevance wins; gap ≤ 10 → AI tiebreaker.
   *
   * @param array $proposals Tentative proposals from Phase 1
   * @return array ['surviving' => [...], 'removed' => [...], 'dedup_count' => int]
   */
  private static function deduplicate_anchors($proposals) {
    if (empty($proposals)) {
      return ['surviving' => [], 'removed' => [], 'dedup_count' => 0];
    }

    // Group proposals by normalized anchor text
    $groups = [];
    foreach ($proposals as $p) {
      $anchor = (string)($p['anchor'] ?? '');
      $norm = strtolower(trim(preg_replace('/\s+/', ' ', $anchor)));
      if ($norm === '') continue;
      $groups[$norm][] = $p;
    }

    $surviving = [];
    $removed = [];
    $dedup_count = 0;

    foreach ($groups as $norm_anchor => $entries) {
      if (count($entries) <= 1) {
        // No conflict — pass through
        foreach ($entries as $e) $surviving[] = $e;
        continue;
      }

      // Multiple sources want the same anchor text — resolve conflict
      // Sort by relevance_score descending
      usort($entries, function($a, $b) {
        return ((int)($b['relevance_score'] ?? 0)) <=> ((int)($a['relevance_score'] ?? 0));
      });

      $top_score = (int)($entries[0]['relevance_score'] ?? 0);
      $second_score = (int)($entries[1]['relevance_score'] ?? 0);
      $gap = $top_score - $second_score;

      $winner_idx = 0; // default: highest relevance wins

      if ($gap <= 10) {
        // Close call — use AI tiebreaker
        $ai_idx = self::ai_rank_anchor_relevance($norm_anchor, $entries);
        if ($ai_idx >= 0 && $ai_idx < count($entries)) {
          $winner_idx = $ai_idx;
        }
      }

      // Winner survives, rest are removed
      foreach ($entries as $idx => $entry) {
        if ($idx === $winner_idx) {
          $surviving[] = $entry;
        } else {
          $removed[] = $entry;
          $dedup_count++;
        }
      }
    }

    return [
      'surviving' => $surviving,
      'removed' => $removed,
      'dedup_count' => $dedup_count,
    ];
  }

  /**
   * AI-based anchor relevance ranking for deduplication tiebreaker.
   * Calls OpenAI to pick the best source-target pair for a contested anchor.
   * Falls back to index 0 (highest relevance) on error or missing API key.
   *
   * @param string $anchor_text The contested anchor text
   * @param array $candidates Array of proposal entries
   * @return int Winner index
   */
  private static function ai_rank_anchor_relevance($anchor_text, $candidates) {
    if (!class_exists('InternalLinksTool_OpenAI')) {
      return 0;
    }

    $key = InternalLinksTool_OpenAI::api_key();
    if ($key === '') {
      return 0;
    }

    // Build candidate info for the AI call
    $ai_candidates = [];
    foreach ($candidates as $c) {
      $ai_candidates[] = [
        'source_title' => (string)($c['source_title'] ?? ''),
        'source_url'   => (string)($c['source_url'] ?? ''),
        'target_title' => (string)($c['target_meta_title'] ?? ''),
        'target_url'   => (string)($c['target_url'] ?? ''),
        'relevance_score' => (int)($c['relevance_score'] ?? 0),
      ];
    }

    $result = InternalLinksTool_OpenAI::rank_anchor_relevance($anchor_text, $ai_candidates);

    if (is_array($result) && isset($result['winner_index']) && empty($result['error'])) {
      $idx = (int)$result['winner_index'];
      if ($idx >= 0 && $idx < count($candidates)) {
        return $idx;
      }
    }

    // Fallback: highest relevance (index 0, already sorted)
    return 0;
  }

  /**
   * AJAX handler for redo anchor text
   */
  public static function ajax_redo_anchor() {
    // Check permissions
    if (!current_user_can('manage_options')) {
      wp_send_json_error('No permission');
    }

    // Verify nonce
    if (!check_ajax_referer('internallinkstool_redo_anchor', '_ajax_nonce', false)) {
      wp_send_json_error('Nonce verification failed');
    }

    $anchor = isset($_POST['anchor']) ? sanitize_text_field($_POST['anchor']) : '';
    $sentence = isset($_POST['sentence']) ? sanitize_text_field($_POST['sentence']) : '';
    $target_title = isset($_POST['target_title']) ? sanitize_text_field($_POST['target_title']) : '';
    $target_desc = isset($_POST['target_desc']) ? sanitize_text_field($_POST['target_desc']) : '';

    if (empty($target_title) && empty($anchor)) {
      wp_send_json_error('Missing target information');
    }

    // Check if OpenAI class is available
    if (!class_exists('InternalLinksTool_OpenAI')) {
      wp_send_json_error('OpenAI class not available');
    }

    try {
      // Use the new suggest_anchor method for generating fresh suggestions
      if (method_exists('InternalLinksTool_OpenAI', 'suggest_anchor')) {
        $result = InternalLinksTool_OpenAI::suggest_anchor($anchor, $sentence, $target_title, $target_desc);

        if (is_array($result) && !empty($result['error'])) {
          wp_send_json_error($result['error']);
        }

        if (is_array($result) && isset($result['suggestion'])) {
          $alternatives = isset($result['alternatives']) ? $result['alternatives'] : [];

          wp_send_json_success([
            'rewritten' => $result['suggestion'],
            'alternatives' => $alternatives,
            'reason' => $result['reason'] ?? '',
            'changed' => ($result['suggestion'] !== $anchor),
          ]);
        } else {
          wp_send_json_error('Invalid response from AI');
        }
      } else {
        wp_send_json_error('suggest_anchor method not available');
      }
    } catch (Exception $e) {
      wp_send_json_error('AI error: ' . $e->getMessage());
    }
  }
}
