<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_Keywords {

  public static function init() {
    add_action('admin_post_internallinkstool_run_keywords', [__CLASS__, 'handle_run_keywords']);
    add_action('admin_post_internallinkstool_reset_keywords', [__CLASS__, 'handle_reset_keywords']);

    // AJAX handler for redo keywords
    add_action('wp_ajax_internallinkstool_redo_keywords', [__CLASS__, 'ajax_redo_keywords']);

    // AJAX handler for single-page keyword processing (progress bar)
    add_action('wp_ajax_internallinkstool_process_single_keyword', [__CLASS__, 'ajax_process_single_keyword']);
  }

  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : '';
    $err = isset($_GET['err']) ? sanitize_text_field($_GET['err']) : '';

    $progress = self::get_progress_counts();

    // Get settings for display
    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];
    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    echo '<div class="wrap">';
    echo '<h1>Keywords</h1>';

    echo '<div style="background:#fef8e7;border-left:4px solid #f0c33c;padding:12px 16px;margin:10px 0 16px;">';
    echo '<p style="margin:0;font-size:13px;"><strong>You don\'t need to run this manually.</strong> The <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-linker')) . '">Linker</a> page\'s "Run All" handles scanning, keyword extraction, and anchor bank generation automatically. This page (along with Scanner and AI Anchor Banks) is available for SEO experts who want more granular control &mdash; to review which keywords are generated, edit them, or re-run extraction for specific pages.</p>';
    echo '</div>';

    echo '<p><strong>Run the <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-scanner')) . '">Scanner</a> first</strong> to populate the pages database.</p>';

    echo '<div style="background:#f0f6fc;border:1px solid #c3c4c7;border-left:4px solid #2271b1;padding:12px 16px;margin:15px 0;">';
    echo '<p style="margin:0;"><strong>Keyword Extraction</strong> &mdash; Analyzes each scanned page and determines its primary keyword (the main topic) and secondary keywords (related terms). Uses 1 AI call per page.</p>';
    echo '</div>';
    echo '<p>To generate anchor text variations, visit the <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-anchor-banks')) . '">AI Anchor Banks</a> page.</p>';

    // Build display labels
    $types_list = [];
    if (in_array('post', $types)) $types_list[] = 'Posts';
    if (in_array('page', $types)) $types_list[] = 'Pages';
    if (empty($types_list)) $types_list = ['Posts', 'Pages'];

    $statuses_list = [];
    if (in_array('publish', $statuses)) $statuses_list[] = 'Published';
    if (in_array('draft', $statuses)) $statuses_list[] = 'Draft';
    if (empty($statuses_list)) $statuses_list = ['Published'];

    $respect_robots = !empty($settings['respect_robots']) ? 'Yes (excluding noindex/nofollow)' : 'No';

    // Current settings summary
    echo '<div class="notice notice-warning" style="border-left-color:#2271b1;background:#f0f6fc;">';
    echo '<p><strong>Current Keyword Extraction Settings:</strong></p>';
    echo '<ul style="margin:5px 0 5px 20px;">';
    echo '<li><strong>Post Types:</strong> ' . esc_html(implode(', ', $types_list)) . '</li>';
    echo '<li><strong>Statuses:</strong> ' . esc_html(implode(', ', $statuses_list)) . '</li>';
    echo '<li><strong>Respect Robots:</strong> ' . esc_html($respect_robots) . '</li>';
    echo '</ul>';
    echo '<p class="description" style="margin-top:5px;">Change these in <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-settings')) . '">Link Settings</a>.</p>';
    echo '</div>';

    if ($err) echo '<div class="notice notice-error"><p>' . esc_html($err) . '</p></div>';
    if ($msg) echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';

    echo '<p><strong>Keyword Progress:</strong> ';
    echo '<code>' . (int)$progress['done'] . '</code> done out of <code>' . (int)$progress['total'] . '</code>. ';
    echo 'Remaining: <code>' . (int)$progress['remaining'] . '</code>.';
    echo '</p>';

    // AJAX-based keyword extraction with progress bar
    $nonce = wp_create_nonce('internallinkstool_batch_keywords');
    ?>
    <div style="margin:15px 0;">
      <label style="display:inline-flex;align-items:center;gap:6px;margin-bottom:10px;cursor:pointer;">
        <input type="checkbox" id="ilt-also-anchor-banks" checked />
        <strong>Also generate Anchor Banks</strong>
        <span class="description">(generates anchor text variations for each page after keywords are extracted)</span>
      </label>
    </div>

    <div style="margin:10px 0;">
      <button type="button" id="ilt-kw-start" class="button button-primary button-hero">Run Keyword Extraction</button>
      <button type="button" id="ilt-kw-stop" class="button button-secondary" style="display:none;margin-left:8px;">Stop</button>
    </div>

    <div id="ilt-kw-progress-wrap" style="display:none;margin:15px 0 20px;">
      <div style="background:#e0e0e0;border-radius:4px;height:28px;width:100%;max-width:600px;overflow:hidden;">
        <div id="ilt-kw-bar" style="background:#2271b1;height:100%;width:0%;transition:width 0.3s;display:flex;align-items:center;justify-content:center;color:#fff;font-size:13px;font-weight:600;min-width:40px;">0%</div>
      </div>
      <p id="ilt-kw-stats" style="margin:6px 0;font-size:13px;color:#555;">Processed: 0 of 0 | Remaining: 0 | Errors: 0 | Elapsed: 0s</p>
    </div>

    <div id="ilt-kw-log-wrap" style="display:none;margin:10px 0;">
      <div id="ilt-kw-log" style="max-height:300px;overflow-y:auto;background:#1d2327;color:#c3c4c7;font-family:monospace;font-size:12px;padding:10px;border-radius:4px;line-height:1.6;"></div>
    </div>

    <div id="ilt-kw-summary" style="display:none;margin:15px 0;padding:12px 16px;background:#e7f5e7;border:1px solid #46b450;border-radius:4px;">
      <strong>Keyword extraction complete!</strong>
      <span id="ilt-kw-summary-text"></span>
      &mdash; <a href="<?php echo esc_url(admin_url('admin.php?page=internallinkstool-keywords')); ?>">Refresh Page</a>
    </div>

    <script type="text/javascript">
    (function(){
      var running = false;
      var startBtn = document.getElementById('ilt-kw-start');
      var stopBtn = document.getElementById('ilt-kw-stop');
      var progressWrap = document.getElementById('ilt-kw-progress-wrap');
      var bar = document.getElementById('ilt-kw-bar');
      var stats = document.getElementById('ilt-kw-stats');
      var logWrap = document.getElementById('ilt-kw-log-wrap');
      var log = document.getElementById('ilt-kw-log');
      var summary = document.getElementById('ilt-kw-summary');
      var summaryText = document.getElementById('ilt-kw-summary-text');
      var checkbox = document.getElementById('ilt-also-anchor-banks');
      var nonce = '<?php echo esc_js($nonce); ?>';

      var processed = 0, errorCount = 0, startTime = 0;

      startBtn.addEventListener('click', function(){
        if (running) return;
        running = true;
        processed = 0;
        errorCount = 0;
        startTime = Date.now();

        startBtn.style.display = 'none';
        stopBtn.style.display = '';
        progressWrap.style.display = '';
        logWrap.style.display = '';
        summary.style.display = 'none';
        log.innerHTML = '';
        bar.style.width = '0%';
        bar.textContent = '0%';

        appendLog('Starting keyword extraction...');
        processNext();
      });

      stopBtn.addEventListener('click', function(){
        running = false;
        stopBtn.disabled = true;
        stopBtn.textContent = 'Stopping...';
        appendLog('Stop requested — finishing current page...');
      });

      function processNext() {
        if (!running) {
          finish('Stopped by user.');
          return;
        }

        var formData = new FormData();
        formData.append('action', 'internallinkstool_process_single_keyword');
        formData.append('_ajax_nonce', nonce);
        formData.append('also_anchor_banks', checkbox.checked ? '1' : '0');

        fetch(ajaxurl, { method: 'POST', body: formData })
          .then(function(r){ return r.json(); })
          .then(function(resp){
            if (!resp.success) {
              appendLog('ERROR: ' + (resp.data || 'Unknown error'), true);
              errorCount++;
              if (running) { setTimeout(processNext, 2000); }
              return;
            }

            var d = resp.data;

            if (d.done) {
              updateBar(d.progress);
              appendLog('All pages processed.');
              running = false;
              finish('All done!');
              return;
            }

            processed++;
            updateBar(d.progress);

            var line = '#' + d.doc_id + ' ' + d.url + ' — KW: "' + d.primary + '" (' + d.source + ')';
            if (!d.kw_success) {
              line += ' [KW ERROR: ' + d.kw_error + ']';
              errorCount++;
            }
            if (d.ab_result) {
              if (d.ab_result.success) {
                line += ' | AB: ' + d.ab_result.anchors_generated + ' anchors';
              } else {
                line += ' | AB ERROR';
              }
            }
            appendLog(line, !d.kw_success);

            if (running) { processNext(); }
            else { finish('Stopped by user.'); }
          })
          .catch(function(err){
            appendLog('Network error: ' + err.message + ' — retrying...', true);
            errorCount++;
            if (running) { setTimeout(processNext, 2000); }
            else { finish('Stopped after error.'); }
          });
      }

      function updateBar(progress) {
        if (!progress) return;
        var pct = progress.total > 0 ? Math.round((progress.done / progress.total) * 100) : 0;
        bar.style.width = pct + '%';
        bar.textContent = pct + '%';
        var elapsed = Math.round((Date.now() - startTime) / 1000);
        stats.textContent = 'Processed: ' + progress.done + ' of ' + progress.total +
          ' | Remaining: ' + progress.remaining + ' | Errors: ' + errorCount + ' | Elapsed: ' + elapsed + 's';
      }

      function appendLog(text, isError) {
        var span = document.createElement('div');
        span.textContent = text;
        if (isError) span.style.color = '#e65054';
        log.appendChild(span);
        log.scrollTop = log.scrollHeight;
      }

      function finish(msg) {
        stopBtn.style.display = 'none';
        stopBtn.disabled = false;
        stopBtn.textContent = 'Stop';
        startBtn.style.display = '';
        var elapsed = Math.round((Date.now() - startTime) / 1000);
        summaryText.textContent = ' Processed ' + processed + ' pages, ' + errorCount + ' errors, ' + elapsed + 's elapsed.';
        summary.style.display = '';
        appendLog(msg + ' (' + processed + ' pages, ' + errorCount + ' errors, ' + elapsed + 's)');
      }
    })();
    </script>
    <?php

    // Reset Keywords button
    echo '<hr>';
    echo '<h2>Reset Keywords</h2>';
    echo '<p class="description">Clear all extracted keywords from the database. This will allow you to re-run keyword extraction from scratch.</p>';
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'Are you sure you want to reset ALL keywords? This cannot be undone.\');">';
    echo '<input type="hidden" name="action" value="internallinkstool_reset_keywords" />';
    wp_nonce_field('internallinkstool_reset_keywords');
    submit_button('Reset All Keywords', 'delete');
    echo '</form>';

    // Keywords table display
    echo '<hr>';
    echo '<h2>Current Keywords</h2>';

    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');
    $keywords_table = InternalLinksTool_DB::table('keywords');

    // Build filters for display
    $type_placeholders = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));
    $robots_sql_display = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql_display = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Pagination
    $per_page = 100;
    $current_page = isset($_GET['paged']) ? max(1, (int)$_GET['paged']) : 1;
    $offset = ($current_page - 1) * $per_page;

    // Count total keywords
    $sql_count = "SELECT COUNT(*)
                  FROM {$keywords_table} k
                  INNER JOIN {$docs} d ON d.id = k.document_id
                  WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword <> ''
                    AND d.type IN ({$type_placeholders})
                    AND d.status IN ({$status_placeholders})
                    {$robots_sql_display}";
    $params_count = array_merge($types, $statuses);
    $total_keywords = (int)$wpdb->get_var($wpdb->prepare($sql_count, $params_count));
    $total_pages = ceil($total_keywords / $per_page);

    $sql_keywords = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, k.primary_keyword, k.secondary_keywords, k.source
                     FROM {$keywords_table} k
                     INNER JOIN {$docs} d ON d.id = k.document_id
                     WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword <> ''
                       AND d.type IN ({$type_placeholders})
                       AND d.status IN ({$status_placeholders})
                       {$robots_sql_display}
                     ORDER BY k.id DESC
                     LIMIT %d OFFSET %d";
    $params_keywords = array_merge($types, $statuses, [$per_page, $offset]);
    $keyword_rows = $wpdb->get_results($wpdb->prepare($sql_keywords, $params_keywords), ARRAY_A);

    // Batch-fetch Yoast focus keywords for all post IDs
    $post_ids = array_unique(array_filter(array_map(function($r) {
      return (int)($r['post_id'] ?? 0);
    }, $keyword_rows)));
    $yoast_map = [];
    if (!empty($post_ids)) {
      $ids_placeholder = implode(',', array_map('intval', $post_ids));
      $yoast_rows = $wpdb->get_results(
        "SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE post_id IN ({$ids_placeholder}) AND meta_key = '_yoast_wpseo_focuskw'",
        ARRAY_A
      );
      if ($yoast_rows) {
        foreach ($yoast_rows as $yr) {
          $yoast_map[(int)$yr['post_id']] = (string)$yr['meta_value'];
        }
      }
    }

    if (empty($keyword_rows)) {
      echo '<p><em>No keywords extracted yet.</em></p>';
    } else {
      // Pagination info
      $start_num = $offset + 1;
      $end_num = min($offset + $per_page, $total_keywords);
      echo '<p class="description">Showing ' . $start_num . '-' . $end_num . ' of ' . $total_keywords . ' pages with keywords. Click "Redo" to regenerate keywords for a specific page.</p>';

      // Pagination links
      if ($total_pages > 1) {
        echo '<div class="tablenav"><div class="tablenav-pages">';
        echo '<span class="displaying-num">' . $total_keywords . ' items</span>';
        echo '<span class="pagination-links">';

        $base_url = admin_url('admin.php?page=internallinkstool-keywords');
        if ($current_page > 1) {
          echo '<a class="button" href="' . esc_url($base_url . '&paged=1') . '">&laquo; First</a> ';
          echo '<a class="button" href="' . esc_url($base_url . '&paged=' . ($current_page - 1)) . '">&lsaquo; Prev</a> ';
        }
        echo '<span class="paging-input">' . $current_page . ' of ' . $total_pages . '</span>';
        if ($current_page < $total_pages) {
          echo ' <a class="button" href="' . esc_url($base_url . '&paged=' . ($current_page + 1)) . '">Next &rsaquo;</a>';
          echo ' <a class="button" href="' . esc_url($base_url . '&paged=' . $total_pages) . '">Last &raquo;</a>';
        }
        echo '</span></div></div>';
      }

      echo '<table class="widefat fixed striped" id="keywords-table">';
      echo '<thead><tr>';
      echo '<th style="width:4%;">ID</th>';
      echo '<th style="width:22%;">URL</th>';
      echo '<th style="width:12%;">Meta Title</th>';
      echo '<th style="width:8%;">Yoast Focus KW</th>';
      echo '<th style="width:14%;">Primary KW</th>';
      echo '<th style="width:22%;">Secondary KW</th>';
      echo '<th style="width:5%;">Source</th>';
      echo '<th style="width:5%;">Redo</th>';
      echo '</tr></thead><tbody>';

      foreach ($keyword_rows as $kr) {
        $doc_id = (int)($kr['id'] ?? 0);
        $pid = (int)($kr['post_id'] ?? 0);
        $url = (string)($kr['url'] ?? '');
        $meta_title = (string)($kr['meta_title'] ?? '');
        $meta_desc = (string)($kr['meta_desc'] ?? '');
        $yoast_fkw = $yoast_map[$pid] ?? '';
        $pk = (string)($kr['primary_keyword'] ?? '');
        $sk = (string)($kr['secondary_keywords'] ?? '');
        $source = (string)($kr['source'] ?? '');

        $row_id = 'kw-row-' . $doc_id;

        echo '<tr id="' . esc_attr($row_id) . '">';
        echo '<td><code>' . $pid . '</code></td>';
        echo '<td style="word-break:break-all;font-size:12px;"><a href="' . esc_url($url) . '" target="_blank" rel="noopener">' . esc_html($url) . '</a></td>';
        echo '<td style="font-size:12px;">' . esc_html($meta_title) . '</td>';
        echo '<td><code style="font-size:11px;">' . esc_html($yoast_fkw) . '</code></td>';
        echo '<td id="' . esc_attr($row_id) . '-pk"><code style="background:#e3f2fd;padding:2px 4px;">' . esc_html($pk) . '</code></td>';
        echo '<td id="' . esc_attr($row_id) . '-sk" style="font-size:12px;">' . esc_html($sk) . '</td>';
        echo '<td id="' . esc_attr($row_id) . '-src"><code style="font-size:10px;">' . esc_html($source) . '</code></td>';
        echo '<td>';
        echo '<button type="button" class="button button-small redo-keywords-btn" ';
        echo 'data-doc-id="' . esc_attr($doc_id) . '" ';
        echo 'data-row-id="' . esc_attr($row_id) . '" ';
        echo 'data-meta-title="' . esc_attr($meta_title) . '" ';
        echo 'data-meta-desc="' . esc_attr($meta_desc) . '" ';
        echo 'data-url="' . esc_attr($url) . '">';
        echo 'Redo</button>';
        echo '</td>';
        echo '</tr>';
      }

      echo '</tbody></table>';

      // Pagination links at bottom
      if ($total_pages > 1) {
        echo '<div class="tablenav bottom"><div class="tablenav-pages">';
        echo '<span class="displaying-num">' . $total_keywords . ' items</span>';
        echo '<span class="pagination-links">';

        $base_url = admin_url('admin.php?page=internallinkstool-keywords');
        if ($current_page > 1) {
          echo '<a class="button" href="' . esc_url($base_url . '&paged=1') . '">&laquo; First</a> ';
          echo '<a class="button" href="' . esc_url($base_url . '&paged=' . ($current_page - 1)) . '">&lsaquo; Prev</a> ';
        }
        echo '<span class="paging-input">' . $current_page . ' of ' . $total_pages . '</span>';
        if ($current_page < $total_pages) {
          echo ' <a class="button" href="' . esc_url($base_url . '&paged=' . ($current_page + 1)) . '">Next &rsaquo;</a>';
          echo ' <a class="button" href="' . esc_url($base_url . '&paged=' . $total_pages) . '">Last &raquo;</a>';
        }
        echo '</span></div></div>';
      }

      // JavaScript for Redo Keywords AJAX
      ?>
      <script type="text/javascript">
      (function() {
        var buttons = document.querySelectorAll('.redo-keywords-btn');
        buttons.forEach(function(btn) {
          btn.addEventListener('click', function() {
            var docId = this.getAttribute('data-doc-id');
            var rowId = this.getAttribute('data-row-id');
            var metaTitle = this.getAttribute('data-meta-title');
            var metaDesc = this.getAttribute('data-meta-desc');
            var url = this.getAttribute('data-url');
            var button = this;

            var pkCell = document.getElementById(rowId + '-pk');
            var skCell = document.getElementById(rowId + '-sk');
            var srcCell = document.getElementById(rowId + '-src');

            // Disable button and show loading
            button.disabled = true;
            button.textContent = 'Loading...';
            pkCell.innerHTML = '<span style="color:#666;">...</span>';
            skCell.innerHTML = '<span style="color:#666;">Generating...</span>';

            // AJAX request
            var formData = new FormData();
            formData.append('action', 'internallinkstool_redo_keywords');
            formData.append('doc_id', docId);
            formData.append('meta_title', metaTitle);
            formData.append('meta_desc', metaDesc);
            formData.append('url', url);
            formData.append('_ajax_nonce', '<?php echo wp_create_nonce("internallinkstool_redo_keywords"); ?>');

            fetch(ajaxurl, {
              method: 'POST',
              body: formData
            })
            .then(function(response) { return response.json(); })
            .then(function(data) {
              button.disabled = false;
              button.textContent = 'Redo';

              if (data.success) {
                pkCell.innerHTML = '<code style="background:#e7f5e7;color:#2e7d32;">' + escapeHtml(data.data.primary) + '</code>';
                skCell.innerHTML = '<span style="background:#e7f5e7;color:#2e7d32;">' + escapeHtml(data.data.secondary) + '</span>';
                srcCell.innerHTML = '<code>ai-redo</code>';
              } else {
                pkCell.innerHTML = '<span style="color:#d63638;">Error</span>';
                skCell.innerHTML = '<span style="color:#d63638;">' + escapeHtml(data.data || 'Unknown error') + '</span>';
              }
            })
            .catch(function(err) {
              button.disabled = false;
              button.textContent = 'Redo';
              pkCell.innerHTML = '<span style="color:#d63638;">Failed</span>';
              skCell.innerHTML = '<span style="color:#d63638;">Request failed</span>';
            });
          });
        });

        function escapeHtml(text) {
          var div = document.createElement('div');
          div.appendChild(document.createTextNode(text));
          return div.innerHTML;
        }
      })();
      </script>
      <?php
    }

    echo '</div>';
  }

  public static function handle_run_keywords() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_run_keywords')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-keywords&err=' . rawurlencode('Nonce failed. Refresh and try again.')));
      exit;
    }

    $batch_size = isset($_POST['batch_size']) ? (int)$_POST['batch_size'] : 20;
    $batch_size = max(1, min(100, $batch_size));

    try {
      $res = self::run_batch($batch_size);
      $msg = 'Keyword run complete. Processed: ' . (int)$res['processed'] . ', Saved: ' . (int)$res['saved'];
      if ($res['errors']) {
        $msg .= ', Errors: ' . (int)$res['errors'];
      }
      $msg .= '.';
      wp_redirect(admin_url('admin.php?page=internallinkstool-keywords&batch_size=' . $batch_size . '&msg=' . rawurlencode($msg)));
      exit;
    } catch (Exception $e) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-keywords&batch_size=' . $batch_size . '&err=' . rawurlencode('Keywords error: ' . $e->getMessage())));
      exit;
    }
  }

  public static function handle_reset_keywords() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_reset_keywords')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-keywords&err=' . rawurlencode('Nonce failed. Refresh and try again.')));
      exit;
    }

    try {
      if (!class_exists('InternalLinksTool_DB')) throw new Exception('DB class not loaded.');
      global $wpdb;

      $keywords_table = InternalLinksTool_DB::table('keywords');

      // Clear all keywords from the keywords table
      $updated = $wpdb->query("TRUNCATE TABLE {$keywords_table}");

      $msg = 'Keywords reset complete. ' . (int)$updated . ' documents cleared.';
      wp_redirect(admin_url('admin.php?page=internallinkstool-keywords&msg=' . rawurlencode($msg)));
      exit;

    } catch (Exception $e) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-keywords&err=' . rawurlencode('Reset error: ' . $e->getMessage())));
      exit;
    }
  }

  public static function run_batch($batch_size = 20) {
    if (!class_exists('InternalLinksTool_DB')) throw new Exception('DB class not loaded.');
    global $wpdb;

    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    // Get settings
    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];

    // Build filters based on settings
    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    $type_placeholders = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Find documents that don't have keywords yet (LEFT JOIN) with filters
    $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, d.h1
            FROM {$docs} d
            LEFT JOIN {$keywords} k ON k.document_id = d.id
            WHERE (k.id IS NULL OR k.primary_keyword IS NULL OR k.primary_keyword = '')
              AND d.type IN ({$type_placeholders})
              AND d.status IN ({$status_placeholders})
              {$robots_sql}
            ORDER BY d.id ASC
            LIMIT %d";

    $params = array_merge($types, $statuses, [(int)$batch_size]);
    $rows = $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);

    if (!is_array($rows)) $rows = [];

    $processed = 0; $saved = 0; $errors = 0;

    foreach ($rows as $r) {
      $processed++;

      $doc_id  = (int)($r['id'] ?? 0);
      $post_id = (int)($r['post_id'] ?? 0);
      $url     = trim((string)($r['url'] ?? ''));

      $meta_title = trim((string)($r['meta_title'] ?? ''));
      $meta_desc  = trim((string)($r['meta_desc'] ?? ''));
      if ($meta_desc === '') {
        $meta_desc = trim((string)($r['meta_description'] ?? ''));
      }
      $h1 = trim((string)($r['h1'] ?? ''));

      try {
        // CSV override priority
        $csv_kw = self::get_csv_override_for_url($url);
        if ($csv_kw) {
          self::save_keywords($docs, $doc_id, $csv_kw['primary'], $csv_kw['secondary'], 'csv');
          $saved++;
          continue;
        }

        // Build robust context even if meta is missing
        $post_title = '';
        $snippet = '';
        $yoast_keyword = '';
        $content_paragraph = '';
        if ($post_id > 0) {
          $p = get_post($post_id);
          if ($p) {
            $post_title = is_string($p->post_title) ? trim($p->post_title) : '';
            if (is_string($p->post_content) && trim($p->post_content) !== '') {
              $snippet = self::content_snippet($p->post_content, 260);
              $content_paragraph = self::first_paragraph($p->post_content, 600);
            }
          }
          // Fetch Yoast focus keyword — strongest signal for primary keyword
          $yoast_keyword = trim((string)get_post_meta($post_id, '_yoast_wpseo_focuskw', true));
        }

        if ($meta_title === '' && $post_title !== '') $meta_title = $post_title;
        if ($meta_desc === '' && $snippet !== '') $meta_desc = $snippet;

        // Still allow extraction even if some fields empty
        $kw = self::extract_keywords_ai([
          'url' => $url,
          'meta_title' => $meta_title,
          'meta_description' => $meta_desc,
          'h1' => $h1,
          'post_title' => $post_title,
          'snippet' => $snippet,
          'yoast_keyword' => $yoast_keyword,
          'content_paragraph' => $content_paragraph,
        ]);

        if (!$kw || empty($kw['primary_keyword'])) {
          // Deterministic fallback if model returns empty
          $fallback = self::fallback_keywords($meta_title, $h1, $url);
          if (empty($fallback['primary'])) throw new Exception('Empty keyword result.');
          self::save_keywords($docs, $doc_id, $fallback['primary'], $fallback['secondary'], 'fallback');
          $saved++;
          continue;
        }

        // Use primary_keywords array if available (new format), else fall back to primary_keyword string
        $primaries = $kw['primary_keywords'] ?? [];
        if (empty($primaries)) {
          $primaries = [trim((string)$kw['primary_keyword'])];
        }
        $primaries = array_values(array_filter(array_map('trim', $primaries)));

        $secondary = $kw['secondary_keywords'] ?? [];
        if (!is_array($secondary)) $secondary = [];

        $secondary = array_values(array_unique(array_filter(array_map(function($x){
          $x = trim((string)$x);
          return $x !== '' ? $x : null;
        }, $secondary))));

        // Keep it tight
        $secondary = array_slice($secondary, 0, 8);

        self::save_keywords($docs, $doc_id, $primaries, $secondary, 'ai');
        $saved++;

      } catch (Exception $e) {
        $errors++;
        // Save fallback keywords so this page doesn't block the batch forever
        $fb = self::fallback_keywords($meta_title, $h1, $url);
        if (!empty($fb['primary'])) {
          self::save_keywords($docs, $doc_id, $fb['primary'], $fb['secondary'], 'fallback');
        }
      }
    }

    return ['processed'=>$processed,'saved'=>$saved,'errors'=>$errors];
  }

  /**
   * Process a single unprocessed page for keyword extraction.
   * Returns result array with success/error info, or 'none_remaining' error when done.
   */
  public static function process_single_page() {
    if (!class_exists('InternalLinksTool_DB')) throw new Exception('DB class not loaded.');
    global $wpdb;

    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];
    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    $type_placeholders = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, d.h1
            FROM {$docs} d
            LEFT JOIN {$keywords} k ON k.document_id = d.id
            WHERE (k.id IS NULL OR k.primary_keyword IS NULL OR k.primary_keyword = '')
              AND d.type IN ({$type_placeholders})
              AND d.status IN ({$status_placeholders})
              {$robots_sql}
            ORDER BY d.id ASC
            LIMIT 1";

    $params = array_merge($types, $statuses);
    $r = $wpdb->get_row($wpdb->prepare($sql, $params), ARRAY_A);

    if (!$r) {
      return ['success' => true, 'doc_id' => 0, 'url' => '', 'primary' => '', 'source' => '', 'error' => 'none_remaining'];
    }

    $doc_id  = (int)($r['id'] ?? 0);
    $post_id = (int)($r['post_id'] ?? 0);
    $url     = trim((string)($r['url'] ?? ''));
    $meta_title = trim((string)($r['meta_title'] ?? ''));
    $meta_desc  = trim((string)($r['meta_desc'] ?? ''));
    if ($meta_desc === '') {
      $meta_desc = trim((string)($r['meta_description'] ?? ''));
    }
    $h1 = trim((string)($r['h1'] ?? ''));

    try {
      // CSV override priority
      $csv_kw = self::get_csv_override_for_url($url);
      if ($csv_kw) {
        self::save_keywords($docs, $doc_id, $csv_kw['primary'], $csv_kw['secondary'], 'csv');
        return ['success' => true, 'doc_id' => $doc_id, 'url' => $url, 'primary' => $csv_kw['primary'], 'source' => 'csv', 'error' => ''];
      }

      // Build robust context
      $post_title = '';
      $snippet = '';
      $yoast_keyword = '';
      $content_paragraph = '';
      if ($post_id > 0) {
        $p = get_post($post_id);
        if ($p) {
          $post_title = is_string($p->post_title) ? trim($p->post_title) : '';
          if (is_string($p->post_content) && trim($p->post_content) !== '') {
            $snippet = self::content_snippet($p->post_content, 260);
            $content_paragraph = self::first_paragraph($p->post_content, 600);
          }
        }
        // Fetch Yoast focus keyword — strongest signal for primary keyword
        $yoast_keyword = trim((string)get_post_meta($post_id, '_yoast_wpseo_focuskw', true));
      }

      if ($meta_title === '' && $post_title !== '') $meta_title = $post_title;
      if ($meta_desc === '' && $snippet !== '') $meta_desc = $snippet;

      $kw = self::extract_keywords_ai([
        'url' => $url,
        'meta_title' => $meta_title,
        'meta_description' => $meta_desc,
        'h1' => $h1,
        'post_title' => $post_title,
        'snippet' => $snippet,
        'yoast_keyword' => $yoast_keyword,
        'content_paragraph' => $content_paragraph,
      ]);

      if (!$kw || empty($kw['primary_keyword'])) {
        $fallback = self::fallback_keywords($meta_title, $h1, $url);
        if (empty($fallback['primary'])) throw new Exception('Empty keyword result.');
        self::save_keywords($docs, $doc_id, $fallback['primary'], $fallback['secondary'], 'fallback');
        return ['success' => true, 'doc_id' => $doc_id, 'url' => $url, 'primary' => $fallback['primary'], 'source' => 'fallback', 'error' => ''];
      }

      // Use primary_keywords array if available (new format), else fall back to primary_keyword string
      $primaries = $kw['primary_keywords'] ?? [];
      if (empty($primaries)) {
        $primaries = [trim((string)$kw['primary_keyword'])];
      }
      $primaries = array_values(array_filter(array_map('trim', $primaries)));

      $secondary = $kw['secondary_keywords'] ?? [];
      if (!is_array($secondary)) $secondary = [];
      $secondary = array_values(array_unique(array_filter(array_map(function($x){
        $x = trim((string)$x);
        return $x !== '' ? $x : null;
      }, $secondary))));
      $secondary = array_slice($secondary, 0, 8);

      self::save_keywords($docs, $doc_id, $primaries, $secondary, 'ai');
      $primary = !empty($primaries) ? $primaries[0] : '';
      return ['success' => true, 'doc_id' => $doc_id, 'url' => $url, 'primary' => $primary, 'source' => 'ai', 'error' => ''];

    } catch (Exception $e) {
      return ['success' => false, 'doc_id' => $doc_id, 'url' => $url, 'primary' => '', 'source' => '', 'error' => $e->getMessage()];
    }
  }

  /**
   * AJAX handler for single-page keyword processing (progress bar).
   */
  public static function ajax_process_single_keyword() {
    if (!current_user_can('manage_options')) {
      wp_send_json_error('No permission');
    }

    if (!check_ajax_referer('internallinkstool_batch_keywords', '_ajax_nonce', false)) {
      wp_send_json_error('Nonce verification failed');
    }

    $also_anchor_banks = !empty($_POST['also_anchor_banks']) && $_POST['also_anchor_banks'] !== '0';

    // Process one page for keywords
    $result = self::process_single_page();

    // If no pages remaining, return done
    if ($result['error'] === 'none_remaining') {
      $progress = self::get_progress_counts();
      wp_send_json_success([
        'done' => true,
        'progress' => $progress,
      ]);
    }

    // Anchor bank generation if requested and keywords succeeded
    $ab_result = null;
    if ($also_anchor_banks && $result['success'] && $result['doc_id'] > 0) {
      try {
        if (class_exists('InternalLinksTool_Strategy') && method_exists('InternalLinksTool_Strategy', 'generate_anchor_bank_for_page')) {
          global $wpdb;
          $keywords_table = InternalLinksTool_DB::table('keywords');
          $docs_table = InternalLinksTool_DB::table('documents');
          $kw_row = $wpdb->get_row($wpdb->prepare(
            "SELECT k.primary_keyword, k.secondary_keywords, d.meta_title, d.meta_desc
             FROM {$keywords_table} k
             INNER JOIN {$docs_table} d ON d.id = k.document_id
             WHERE k.document_id = %d LIMIT 1",
            $result['doc_id']
          ), ARRAY_A);

          if ($kw_row && !empty($kw_row['primary_keyword'])) {
            $count = InternalLinksTool_Strategy::generate_anchor_bank_for_page(
              $result['doc_id'],
              $kw_row['primary_keyword'],
              $kw_row['secondary_keywords'] ?? '',
              $kw_row['meta_title'] ?? '',
              $kw_row['meta_desc'] ?? ''
            );
            $ab_result = ['success' => true, 'anchors_generated' => (int)$count];
          }
        }
      } catch (Exception $e) {
        $ab_result = ['success' => false, 'anchors_generated' => 0, 'error' => $e->getMessage()];
      }
    }

    $progress = self::get_progress_counts();

    wp_send_json_success([
      'done' => false,
      'doc_id' => $result['doc_id'],
      'url' => $result['url'],
      'primary' => $result['primary'],
      'source' => $result['source'],
      'kw_success' => $result['success'],
      'kw_error' => $result['error'],
      'ab_result' => $ab_result,
      'progress' => $progress,
    ]);
  }

  private static function save_keywords($docs_table, $doc_id, $primary, $secondary_arr, $source) {
    global $wpdb;

    $keywords_table = InternalLinksTool_DB::table('keywords');

    // Handle primary as array (multiple primaries) or string (single/backwards compatible)
    if (is_array($primary)) {
      $primary = array_values(array_unique(array_filter(array_map('trim', $primary))));
      $primary_csv = implode(', ', $primary);
    } else {
      $primary_csv = trim((string)$primary);
    }

    // Store SECONDARY as comma-separated (to match your main design)
    $secondary_arr = is_array($secondary_arr) ? $secondary_arr : [];
    $secondary_arr = array_values(array_unique(array_filter(array_map('trim', $secondary_arr))));
    $secondary_csv = implode(', ', $secondary_arr);

    $data = [
      'document_id' => (int)$doc_id,
      'primary_keyword' => $primary_csv,
      'secondary_keywords' => $secondary_csv,
      'source' => (string)$source,
      'updated_at' => current_time('mysql'),
    ];

    // Check if keyword row exists for this document
    $exists = $wpdb->get_var($wpdb->prepare(
      "SELECT id FROM {$keywords_table} WHERE document_id = %d LIMIT 1",
      (int)$doc_id
    ));

    if ($exists) {
      unset($data['document_id']);
      $wpdb->update($keywords_table, $data, ['document_id' => (int)$doc_id]);
    } else {
      $wpdb->insert($keywords_table, $data);
    }
  }

  public static function get_progress_counts() {
    if (!class_exists('InternalLinksTool_DB')) return ['total'=>0,'done'=>0,'remaining'=>0];
    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');

    // Get settings
    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];

    // Build filters based on settings
    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    $type_placeholders = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Count total eligible documents
    $sql_total = "SELECT COUNT(*) FROM {$docs} d
                  WHERE d.type IN ({$type_placeholders})
                    AND d.status IN ({$status_placeholders})
                    {$robots_sql}";
    $params_total = array_merge($types, $statuses);
    $total = (int)$wpdb->get_var($wpdb->prepare($sql_total, $params_total));

    // Count done (documents with keywords that match filters)
    $sql_done = "SELECT COUNT(*) FROM {$keywords} k
                 INNER JOIN {$docs} d ON d.id = k.document_id
                 WHERE (k.primary_keyword IS NOT NULL AND k.primary_keyword <> '')
                   AND d.type IN ({$type_placeholders})
                   AND d.status IN ({$status_placeholders})
                   {$robots_sql}";
    $params_done = array_merge($types, $statuses);
    $done = (int)$wpdb->get_var($wpdb->prepare($sql_done, $params_done));

    return ['total'=>$total,'done'=>$done,'remaining'=>max(0,$total-$done)];
  }

  private static function get_allowed_types($settings) {
    $types = [];
    if (!empty($settings['include_posts'])) $types[] = 'post';
    if (!empty($settings['include_pages'])) $types[] = 'page';
    if (empty($types)) $types = ['post', 'page'];
    return $types;
  }

  private static function get_allowed_statuses($settings) {
    if (!empty($settings['scan_statuses']) && is_array($settings['scan_statuses'])) {
      $statuses = array_values(array_unique(array_map('sanitize_key', $settings['scan_statuses'])));
      $statuses = array_values(array_intersect($statuses, ['publish', 'draft']));
      if (!empty($statuses)) return $statuses;
    }
    return ['publish'];
  }

  private static function get_csv_override_for_url($url) {
    $url = trim((string)$url);
    if ($url === '') return null;

    if (!class_exists('InternalLinksTool_CSV')) return null;
    if (!method_exists('InternalLinksTool_CSV', 'get_keyword_for_url')) return null;

    $row = InternalLinksTool_CSV::get_keyword_for_url($url);
    if (!$row || empty($row['keyword'])) return null;

    $primary = trim((string)$row['keyword']);
    if ($primary === '') return null;

    $secondary = [];
    if (!empty($row['secondary'])) {
      $secondary = self::parse_secondary_keywords((string)$row['secondary']);
    }

    return ['primary'=>$primary,'secondary'=>$secondary];
  }

  private static function extract_keywords_ai($ctx) {
    if (!class_exists('InternalLinksTool_OpenAI') || !method_exists('InternalLinksTool_OpenAI', 'extract_keywords')) {
      throw new Exception('OpenAI class not available.');
    }

    $meta_title        = trim((string)($ctx['meta_title'] ?? ''));
    $meta_desc         = trim((string)($ctx['meta_description'] ?? ''));
    $h1                = trim((string)($ctx['h1'] ?? ''));
    $url               = trim((string)($ctx['url'] ?? ''));
    $yoast_keyword     = trim((string)($ctx['yoast_keyword'] ?? ''));
    $content_paragraph = trim((string)($ctx['content_paragraph'] ?? ''));

    $result = InternalLinksTool_OpenAI::extract_keywords($meta_title, $meta_desc, $h1, $url, $yoast_keyword, $content_paragraph);

    if (!empty($result['error'])) {
      throw new Exception('OpenAI error: ' . $result['error']);
    }

    return [
      'primary_keyword' => $result['primary_keyword'] ?? '',
      'secondary_keywords' => $result['secondary_keywords'] ?? [],
    ];
  }

  private static function parse_model_output($raw) {
    $raw = trim((string)$raw);
    if ($raw === '') return null;

    $j = json_decode($raw, true);
    if (is_array($j) && isset($j['primary_keyword'], $j['secondary_keywords']) && is_array($j['secondary_keywords'])) {
      return $j;
    }

    $start = strpos($raw, '{');
    $end   = strrpos($raw, '}');
    if ($start !== false && $end !== false && $end > $start) {
      $sub = substr($raw, $start, $end - $start + 1);
      $j2 = json_decode($sub, true);
      if (is_array($j2) && isset($j2['primary_keyword'], $j2['secondary_keywords']) && is_array($j2['secondary_keywords'])) {
        return $j2;
      }
    }

    return null;
  }

  private static function fallback_keywords($meta_title, $h1, $url) {
    $base = $h1 !== '' ? $h1 : $meta_title;
    $base = trim((string)$base);

    // If still empty, use slug tokens
    if ($base === '') {
      $path = (string)wp_parse_url($url, PHP_URL_PATH);
      $path = trim($path, '/');
      $path = str_replace(['-','_'], ' ', $path);
      $base = trim($path);
    }

    // Primary = first 1-4 words
    $words = preg_split('/\s+/u', strtolower($base));
    $words = array_values(array_filter(array_map('trim', $words)));
    $primary = implode(' ', array_slice($words, 0, 4));

    // Simple secondary variants
    $secondary = [];
    if ($primary !== '') {
      $secondary[] = $primary . ' guide';
      $secondary[] = $primary . ' overview';
      $secondary[] = $primary . ' comparison';
    }

    $secondary = array_slice(array_values(array_unique($secondary)), 0, 4);

    return ['primary'=>$primary, 'secondary'=>$secondary];
  }

  private static function content_snippet($html, $max_len = 260) {
    $txt = wp_strip_all_tags((string)$html, true);
    $txt = preg_replace('/\s+/u', ' ', $txt);
    $txt = trim((string)$txt);
    if ($txt === '') return '';
    if (mb_strlen($txt) <= $max_len) return $txt;
    return mb_substr($txt, 0, $max_len) . '…';
  }

  private static function first_paragraph($html, $max_len = 600) {
    $text = wp_strip_all_tags((string)$html, true);
    $text = preg_replace('/\s+/u', ' ', trim($text));
    if ($text === '') return '';
    // Split on double newlines or period+space to find first paragraph
    $parts = preg_split('/\n\s*\n|(?<=\.)\s{2,}/', $text, 2);
    $para = trim($parts[0] ?? '');
    if ($para === '') return '';
    if (mb_strlen($para) > $max_len) return mb_substr($para, 0, $max_len) . '...';
    return $para;
  }

  private static function parse_secondary_keywords($raw) {
    $raw = trim((string)$raw);
    if ($raw === '') return [];
    $parts = preg_split('/\r\n|\r|\n|,/', $raw);
    $parts = array_values(array_filter(array_map('trim', $parts)));
    return array_slice($parts, 0, 12);
  }

  private static function column_exists($table, $column) {
    global $wpdb;
    $col = $wpdb->get_var($wpdb->prepare("SHOW COLUMNS FROM {$table} LIKE %s", $column));
    return !empty($col);
  }

  /**
   * AJAX handler for redo keywords extraction
   */
  public static function ajax_redo_keywords() {
    // Check permissions
    if (!current_user_can('manage_options')) {
      wp_send_json_error('No permission');
    }

    // Verify nonce
    if (!check_ajax_referer('internallinkstool_redo_keywords', '_ajax_nonce', false)) {
      wp_send_json_error('Nonce verification failed');
    }

    $doc_id = isset($_POST['doc_id']) ? (int)$_POST['doc_id'] : 0;
    $meta_title = isset($_POST['meta_title']) ? sanitize_text_field($_POST['meta_title']) : '';
    $meta_desc = isset($_POST['meta_desc']) ? sanitize_text_field($_POST['meta_desc']) : '';
    $url = isset($_POST['url']) ? esc_url_raw($_POST['url']) : '';

    if ($doc_id <= 0) {
      wp_send_json_error('Invalid document ID');
    }

    // Check if OpenAI class is available
    if (!class_exists('InternalLinksTool_OpenAI') || !method_exists('InternalLinksTool_OpenAI', 'extract_keywords')) {
      wp_send_json_error('OpenAI class not available');
    }

    try {
      // Try to get H1 from the document record
      global $wpdb;
      $docs_table = InternalLinksTool_DB::table('documents');
      $doc = $wpdb->get_row($wpdb->prepare(
        "SELECT post_id, h1 FROM {$docs_table} WHERE id = %d",
        $doc_id
      ), ARRAY_A);

      $h1 = '';
      $post_id = 0;
      if ($doc) {
        $h1 = (string)($doc['h1'] ?? '');
        $post_id = (int)($doc['post_id'] ?? 0);
      }

      // Fetch Yoast focus keyword
      $yoast_keyword = '';
      $content_paragraph = '';
      if ($post_id > 0) {
        $yoast_keyword = trim((string)get_post_meta($post_id, '_yoast_wpseo_focuskw', true));
      }

      // If meta is empty, try to get from post; also extract first paragraph
      if ($post_id > 0) {
        $post = get_post($post_id);
        if ($post) {
          if ($meta_title === '') $meta_title = $post->post_title;
          if ($meta_desc === '') {
            $meta_desc = self::content_snippet($post->post_content, 260);
          }
          if (is_string($post->post_content) && trim($post->post_content) !== '') {
            $content_paragraph = self::first_paragraph($post->post_content, 600);
          }
        }
      }

      // Call OpenAI redo_keywords (same logic as extract_keywords but with slight temperature variation)
      if (method_exists('InternalLinksTool_OpenAI', 'redo_keywords')) {
        $result = InternalLinksTool_OpenAI::redo_keywords($meta_title, $meta_desc, $h1, $url, $yoast_keyword, $content_paragraph);
      } else {
        // Fallback to regular extract_keywords
        $result = InternalLinksTool_OpenAI::extract_keywords($meta_title, $meta_desc, $h1, $url, $yoast_keyword, $content_paragraph);
      }

      if (!empty($result['error'])) {
        wp_send_json_error('AI error: ' . $result['error']);
      }

      $primary = trim((string)($result['primary_keyword'] ?? ''));
      $secondary_arr = $result['secondary_keywords'] ?? [];
      if (!is_array($secondary_arr)) $secondary_arr = [];

      $secondary_arr = array_values(array_unique(array_filter(array_map('trim', $secondary_arr))));
      $secondary_arr = array_slice($secondary_arr, 0, 8);
      $secondary_csv = implode(', ', $secondary_arr);

      if ($primary === '') {
        // Fallback
        $fallback = self::fallback_keywords($meta_title, $h1, $url);
        $primary = $fallback['primary'];
        $secondary_csv = implode(', ', $fallback['secondary']);
      }

      // Save to database
      $keywords_table = InternalLinksTool_DB::table('keywords');
      $exists = $wpdb->get_var($wpdb->prepare(
        "SELECT id FROM {$keywords_table} WHERE document_id = %d LIMIT 1",
        $doc_id
      ));

      $data = [
        'primary_keyword' => $primary,
        'secondary_keywords' => $secondary_csv,
        'source' => 'ai-redo',
        'updated_at' => current_time('mysql'),
      ];

      if ($exists) {
        $wpdb->update($keywords_table, $data, ['document_id' => $doc_id]);
      } else {
        $data['document_id'] = $doc_id;
        $wpdb->insert($keywords_table, $data);
      }

      wp_send_json_success([
        'primary' => $primary,
        'secondary' => $secondary_csv,
      ]);

    } catch (Exception $e) {
      wp_send_json_error('Error: ' . $e->getMessage());
    }
  }

  /**
   * Find primary keywords shared by 2+ pages (excluding CSV overrides).
   */
  public static function get_conflict_groups() {
    global $wpdb;
    $kw_table = InternalLinksTool_DB::table('keywords');

    // Find primary keywords used by more than one document (case-insensitive)
    $groups = $wpdb->get_results(
      "SELECT LOWER(TRIM(primary_keyword)) AS pk_lower, COUNT(*) AS cnt
       FROM {$kw_table}
       WHERE source != 'csv'
         AND primary_keyword IS NOT NULL
         AND TRIM(primary_keyword) != ''
       GROUP BY pk_lower
       HAVING cnt >= 2
       ORDER BY cnt DESC",
      ARRAY_A
    );

    if (empty($groups)) return [];

    $result = [];
    foreach ($groups as $g) {
      $pk = $g['pk_lower'];
      // Get all pages in this conflict group
      $pages = $wpdb->get_results($wpdb->prepare(
        "SELECT k.document_id, k.primary_keyword, k.secondary_keywords, k.source,
                d.post_id, d.url, d.h1, d.meta_title, d.meta_desc
         FROM {$kw_table} k
         JOIN " . InternalLinksTool_DB::table('documents') . " d ON d.id = k.document_id
         WHERE LOWER(TRIM(k.primary_keyword)) = %s
           AND k.source != 'csv'",
        $pk
      ), ARRAY_A);

      if (count($pages) >= 2) {
        $result[] = [
          'keyword' => $pk,
          'pages'   => $pages,
        ];
      }
    }
    return $result;
  }

  /**
   * Main orchestrator for keyword planning phase.
   * Processes up to $batch_size conflict groups per cron tick.
   */
  public static function plan_unique_keywords($batch_size = 3) {
    $groups = self::get_conflict_groups();

    if (empty($groups)) {
      return ['status' => 'done', 'resolved' => 0, 'total_conflicts' => 0];
    }

    $total = count($groups);
    $resolved = 0;

    // Process up to $batch_size groups per tick
    $to_process = array_slice($groups, 0, $batch_size);

    foreach ($to_process as $group) {
      $pages = $group['pages'];

      // Build page summaries for AI
      $page_summaries = [];
      foreach ($pages as $p) {
        $post_id = (int)($p['post_id'] ?? 0);
        $content_excerpt = '';
        if ($post_id > 0) {
          $post = get_post($post_id);
          if ($post && !empty($post->post_content)) {
            $content_excerpt = self::first_paragraph($post->post_content, 400);
          }
        }
        $page_summaries[] = [
          'document_id'       => $p['document_id'],
          'url'               => $p['url'] ?? '',
          'h1'                => $p['h1'] ?? '',
          'meta_title'        => $p['meta_title'] ?? '',
          'meta_desc'         => $p['meta_desc'] ?? '',
          'current_primary'   => $p['primary_keyword'] ?? '',
          'current_secondary' => $p['secondary_keywords'] ?? '',
          'content_excerpt'   => $content_excerpt,
        ];
      }

      // Large group handling: split into sub-batches of 8
      if (count($page_summaries) > 8) {
        $chunks = array_chunk($page_summaries, 8);
        $chunk_error = false;
        foreach ($chunks as $chunk) {
          $ai_result = InternalLinksTool_OpenAI::resolve_keyword_conflicts($chunk);
          if (!empty($ai_result['error'])) {
            $chunk_error = true;
            continue;
          }
          $assignments = $ai_result['assignments'] ?? [];
          foreach ($assignments as $a) {
            $doc_id = (int)($a['document_id'] ?? 0);
            if ($doc_id <= 0) continue;

            $new_primary   = trim($a['primary_keyword'] ?? '');
            $new_secondary = $a['secondary_keywords'] ?? [];
            if (!is_array($new_secondary)) $new_secondary = [$new_secondary];

            if ($new_primary === '') continue;

            self::save_keywords(
              InternalLinksTool_DB::table('documents'),
              $doc_id,
              $new_primary,
              $new_secondary,
              'ai-planned'
            );

            // Delete existing anchor bank for this document (force regeneration)
            global $wpdb;
            $ab_table = InternalLinksTool_DB::table('anchor_banks');
            $wpdb->delete($ab_table, ['document_id' => $doc_id]);
          }
        }
        if (!$chunk_error) $resolved++;
        continue;
      }

      // Call AI to resolve
      $ai_result = InternalLinksTool_OpenAI::resolve_keyword_conflicts($page_summaries);

      if (!empty($ai_result['error'])) {
        continue; // Skip this group on error, retry next tick
      }

      // Save resolved keywords
      $assignments = $ai_result['assignments'] ?? [];
      foreach ($assignments as $a) {
        $doc_id = (int)($a['document_id'] ?? 0);
        if ($doc_id <= 0) continue;

        $new_primary   = trim($a['primary_keyword'] ?? '');
        $new_secondary = $a['secondary_keywords'] ?? [];
        if (!is_array($new_secondary)) $new_secondary = [$new_secondary];

        if ($new_primary === '') continue;

        // Save with source 'ai-planned'
        self::save_keywords(
          InternalLinksTool_DB::table('documents'),
          $doc_id,
          $new_primary,
          $new_secondary,
          'ai-planned'
        );

        // Delete existing anchor bank for this document (force regeneration)
        global $wpdb;
        $ab_table = InternalLinksTool_DB::table('anchor_banks');
        $wpdb->delete($ab_table, ['document_id' => $doc_id]);
      }

      $resolved++;
    }

    // Check if all groups are resolved
    $remaining = self::get_conflict_groups();
    if (empty($remaining)) {
      return ['status' => 'done', 'resolved' => $resolved, 'total_conflicts' => $total];
    }

    return ['status' => 'in_progress', 'resolved' => $resolved, 'total_conflicts' => $total];
  }
}
