<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_ExistingLinks {

  public static function init() {
    // Auto-update when posts are saved
    add_action('save_post', [__CLASS__, 'on_post_save'], 20, 2);
    add_action('delete_post', [__CLASS__, 'on_post_delete']);

    // CSV export handler
    add_action('admin_post_internallinkstool_export_existing_links', [__CLASS__, 'handle_export_csv']);

    // Remove links handler
    add_action('admin_post_internallinkstool_remove_links', [__CLASS__, 'handle_remove_links']);

    // Scan and clear handlers
    add_action('admin_post_internallinkstool_scan_existing_links', [__CLASS__, 'handle_scan']);
    add_action('admin_post_internallinkstool_clear_existing_links', [__CLASS__, 'handle_clear']);
  }

  /**
   * Render the admin page
   */
  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : '';
    $err = isset($_GET['err']) ? sanitize_text_field($_GET['err']) : '';

    // Get filter values
    $filter_source = isset($_GET['filter_source']) ? (int)$_GET['filter_source'] : 0;
    $filter_target = isset($_GET['filter_target']) ? (int)$_GET['filter_target'] : 0;
    $filter_anchor = isset($_GET['filter_anchor']) ? sanitize_text_field($_GET['filter_anchor']) : '';

    // Pagination
    $per_page = 50;
    $current_page = isset($_GET['paged']) ? max(1, (int)$_GET['paged']) : 1;
    $offset = ($current_page - 1) * $per_page;

    // Get stats
    $stats = self::get_stats();

    // Get links with filters
    $links = self::get_links($filter_source, $filter_target, $filter_anchor, $per_page, $offset);
    $total_filtered = self::count_links($filter_source, $filter_target, $filter_anchor);
    $total_pages = ceil($total_filtered / $per_page);

    // Get all posts for filter dropdowns
    $all_sources = self::get_all_source_posts();
    $all_targets = self::get_all_target_posts();

    echo '<div class="wrap"><h1>Existing Internal Links</h1>';

    if ($err) echo '<div class="notice notice-error"><p>' . esc_html($err) . '</p></div>';
    if ($msg) echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';

    // Stats
    echo '<div class="notice notice-info"><p>';
    echo '<strong>Total internal links found:</strong> <code>' . (int)$stats['total_links'] . '</code> | ';
    echo '<strong>Source pages:</strong> <code>' . (int)$stats['source_pages'] . '</code> | ';
    echo '<strong>Target pages:</strong> <code>' . (int)$stats['target_pages'] . '</code>';
    echo '</p></div>';

    // Scan and Clear buttons
    echo '<div style="margin-bottom:15px;">';
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;">';
    echo '<input type="hidden" name="action" value="internallinkstool_scan_existing_links" />';
    wp_nonce_field('internallinkstool_scan_existing_links');
    submit_button('Scan All Posts', 'primary', 'submit', false);
    echo '</form>';
    echo ' <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="display:inline-block;margin-left:10px;" onsubmit="return confirm(\'Clear all link data?\');">';
    echo '<input type="hidden" name="action" value="internallinkstool_clear_existing_links" />';
    wp_nonce_field('internallinkstool_clear_existing_links');
    submit_button('Clear All Data', 'secondary', 'submit', false);
    echo '</form>';
    echo '<p class="description">Scan discovers existing internal links in your content. Links auto-update when posts are saved.</p>';
    echo '</div>';

    if ($stats['total_links'] === 0) {
      echo '<div class="notice notice-warning"><p>No links in database. Click <strong>Scan All Posts</strong> above to discover existing internal links.</p></div>';
    }

    // Export to CSV button
    if ($stats['total_links'] > 0) {
      echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" style="margin:10px 0;">';
      echo '<input type="hidden" name="action" value="internallinkstool_export_existing_links" />';
      wp_nonce_field('internallinkstool_export_existing_links');
      // Pass current filters to export
      echo '<input type="hidden" name="filter_source" value="' . esc_attr($filter_source) . '" />';
      echo '<input type="hidden" name="filter_target" value="' . esc_attr($filter_target) . '" />';
      echo '<input type="hidden" name="filter_anchor" value="' . esc_attr($filter_anchor) . '" />';
      submit_button('Export to CSV', 'secondary', 'export_csv', false);
      echo ' <span class="description">Exports ' . ($filter_source || $filter_target || $filter_anchor ? 'filtered' : 'all') . ' links</span>';
      echo '</form>';
    }

    // Filter form
    echo '<hr>';
    echo '<h2>Filter Links</h2>';
    echo '<form method="get" action="">';
    echo '<input type="hidden" name="page" value="internallinkstool-existing-links" />';

    echo '<table class="form-table"><tr>';

    // Source filter
    echo '<th scope="row">Source Page</th><td>';
    echo '<select name="filter_source" style="min-width:250px;">';
    echo '<option value="0">-- All Sources --</option>';
    foreach ($all_sources as $src) {
      $selected = ($filter_source === (int)$src['post_id']) ? 'selected' : '';
      echo '<option value="' . (int)$src['post_id'] . '" ' . $selected . '>' . esc_html($src['title']) . ' (' . (int)$src['link_count'] . ' links)</option>';
    }
    echo '</select>';
    echo '</td>';

    // Target filter
    echo '<th scope="row">Target Page</th><td>';
    echo '<select name="filter_target" style="min-width:250px;">';
    echo '<option value="0">-- All Targets --</option>';
    foreach ($all_targets as $tgt) {
      $selected = ($filter_target === (int)$tgt['post_id']) ? 'selected' : '';
      echo '<option value="' . (int)$tgt['post_id'] . '" ' . $selected . '>' . esc_html($tgt['title']) . ' (' . (int)$tgt['link_count'] . ' links)</option>';
    }
    echo '</select>';
    echo '</td>';

    // Anchor search
    echo '<th scope="row">Anchor Text</th><td>';
    echo '<input type="text" name="filter_anchor" value="' . esc_attr($filter_anchor) . '" placeholder="Search anchor..." style="width:200px;" />';
    echo '</td>';

    echo '<td>';
    submit_button('Filter', 'secondary', 'submit', false);
    echo ' <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-existing-links')) . '" class="button">Reset</a>';
    echo '</td>';

    echo '</tr></table>';
    echo '</form>';

    // Results table
    echo '<hr>';
    echo '<h2>Internal Links' . ($total_filtered !== $stats['total_links'] ? ' (filtered: ' . $total_filtered . ')' : '') . '</h2>';

    if (!empty($links)) {
      // Wrap table in form for link removal
      echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" id="remove-links-form">';
      echo '<input type="hidden" name="action" value="internallinkstool_remove_links" />';
      wp_nonce_field('internallinkstool_remove_links');
      // Pass current filters for redirect back
      echo '<input type="hidden" name="filter_source" value="' . esc_attr($filter_source) . '" />';
      echo '<input type="hidden" name="filter_target" value="' . esc_attr($filter_target) . '" />';
      echo '<input type="hidden" name="filter_anchor" value="' . esc_attr($filter_anchor) . '" />';
      echo '<input type="hidden" name="paged" value="' . esc_attr($current_page) . '" />';

      // Remove button at top
      echo '<div style="margin-bottom:10px;">';
      echo '<button type="submit" class="button button-secondary" onclick="return confirm(\'Are you sure you want to remove the selected links? The anchor text will remain but the link will be removed.\');">Remove Selected Links</button>';
      echo ' <span class="description">Select links below, then click to remove them from content (text remains, link is removed)</span>';
      echo '</div>';

      echo '<table class="widefat fixed striped">';
      echo '<thead><tr>';
      echo '<th style="width:3%;"><input type="checkbox" id="select-all-links" /></th>';
      echo '<th style="width:25%;">Source Page URL</th>';
      echo '<th style="width:15%;">Source Primary Keyword</th>';
      echo '<th style="width:25%;">Target Page URL</th>';
      echo '<th style="width:15%;">Target Primary Keyword</th>';
      echo '<th style="width:17%;">Anchor Text</th>';
      echo '</tr></thead><tbody>';

      foreach ($links as $link) {
        $link_id = (int)$link['id'];
        $source_url = (string)($link['source_url'] ?? '');
        $target_url = (string)($link['target_url'] ?? '');
        $source_kw  = (string)($link['source_primary_kw'] ?? '');
        $target_kw  = (string)($link['target_primary_kw'] ?? '');

        echo '<tr>';

        // Checkbox
        echo '<td><input type="checkbox" name="link_ids[]" value="' . $link_id . '" class="link-checkbox" /></td>';

        // Source Page URL (full, no truncation)
        echo '<td style="word-break:break-all;font-size:12px;">';
        echo '<a href="' . esc_url($source_url) . '" target="_blank" rel="noopener">' . esc_html($source_url) . '</a>';
        echo '</td>';

        // Source Primary Keyword
        echo '<td>';
        if ($source_kw !== '') {
          echo '<code style="background:#e3f2fd;padding:2px 6px;border-radius:3px;font-size:11px;">' . esc_html($source_kw) . '</code>';
        } else {
          echo '<span style="color:#999;font-size:11px;">-</span>';
        }
        echo '</td>';

        // Target Page URL (full, no truncation)
        echo '<td style="word-break:break-all;font-size:12px;">';
        if ($target_url !== '') {
          echo '<a href="' . esc_url($target_url) . '" target="_blank" rel="noopener">' . esc_html($target_url) . '</a>';
        } else {
          echo '<span style="color:#999;">External/Unknown</span>';
        }
        echo '</td>';

        // Target Primary Keyword
        echo '<td>';
        if ($target_kw !== '') {
          echo '<code style="background:#e8f5e9;padding:2px 6px;border-radius:3px;font-size:11px;">' . esc_html($target_kw) . '</code>';
        } else {
          echo '<span style="color:#999;font-size:11px;">-</span>';
        }
        echo '</td>';

        // Anchor Text
        echo '<td><code style="background:#fff3cd;color:#856404;padding:3px 6px;font-size:12px;font-weight:bold;">' . esc_html($link['anchor_text'] ?: '(empty)') . '</code></td>';

        echo '</tr>';
      }

      echo '</tbody></table>';

      // JavaScript for select all checkbox
      echo '<script>
        document.getElementById("select-all-links").addEventListener("change", function() {
          var checkboxes = document.querySelectorAll(".link-checkbox");
          for (var i = 0; i < checkboxes.length; i++) {
            checkboxes[i].checked = this.checked;
          }
        });
      </script>';

      echo '</form>';

      // Pagination
      if ($total_pages > 1) {
        echo '<div class="tablenav bottom"><div class="tablenav-pages">';
        $base_url = admin_url('admin.php?page=internallinkstool-existing-links');
        if ($filter_source) $base_url .= '&filter_source=' . $filter_source;
        if ($filter_target) $base_url .= '&filter_target=' . $filter_target;
        if ($filter_anchor) $base_url .= '&filter_anchor=' . urlencode($filter_anchor);

        echo '<span class="displaying-num">' . $total_filtered . ' items</span>';
        echo '<span class="pagination-links">';

        if ($current_page > 1) {
          echo '<a class="button" href="' . esc_url($base_url . '&paged=1') . '">&laquo;</a> ';
          echo '<a class="button" href="' . esc_url($base_url . '&paged=' . ($current_page - 1)) . '">&lsaquo;</a> ';
        }

        echo '<span class="paging-input">' . $current_page . ' of ' . $total_pages . '</span>';

        if ($current_page < $total_pages) {
          echo ' <a class="button" href="' . esc_url($base_url . '&paged=' . ($current_page + 1)) . '">&rsaquo;</a>';
          echo ' <a class="button" href="' . esc_url($base_url . '&paged=' . $total_pages) . '">&raquo;</a>';
        }

        echo '</span></div></div>';
      }

    } else {
      echo '<p><em>No internal links found. Go to <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-scan-links')) . '">Scan Links</a> to scan your content.</em></p>';
    }

    echo '</div>';
  }

  /**
   * Extract internal links from content
   */
  private static function extract_internal_links($content, $source_post_id, $site_host) {
    $links = [];
    $source_url = get_permalink($source_post_id);

    if (empty($content)) return $links;

    // Parse HTML
    $doc = new DOMDocument();
    libxml_use_internal_errors(true);
    $wrapped = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>' . $content . '</body></html>';
    $doc->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();

    $xpath = new DOMXPath($doc);
    $anchors = $xpath->query('//a[@href]');

    $position = 0;
    foreach ($anchors as $a) {
      $href = $a->getAttribute('href');
      if (empty($href)) continue;

      // Skip anchors, javascript, mailto
      if (preg_match('/^(#|javascript:|mailto:|tel:)/i', $href)) continue;

      // Normalize URL
      $full_url = self::normalize_url($href, $source_url);
      if (!$full_url) continue;

      // Check if internal
      $link_host = wp_parse_url($full_url, PHP_URL_HOST);
      if (!$link_host || strtolower($link_host) !== strtolower($site_host)) continue;

      // Get anchor text
      $anchor_text = trim($a->textContent);

      // Try to find target post ID
      $target_post_id = url_to_postid($full_url);

      $position++;
      $links[] = [
        'source_post_id' => $source_post_id,
        'source_url' => $source_url,
        'target_url' => $full_url,
        'target_post_id' => $target_post_id ?: null,
        'anchor_text' => $anchor_text,
        'link_position' => $position,
        'updated_at' => current_time('mysql'),
      ];
    }

    return $links;
  }

  /**
   * Normalize relative URLs
   */
  private static function normalize_url($href, $base_url) {
    $href = trim($href);
    if (empty($href)) return '';

    // Already absolute
    if (preg_match('/^https?:\/\//i', $href)) {
      return $href;
    }

    // Protocol-relative
    if (strpos($href, '//') === 0) {
      return 'https:' . $href;
    }

    // Relative URL
    $base_parts = wp_parse_url($base_url);
    $scheme = $base_parts['scheme'] ?? 'https';
    $host = $base_parts['host'] ?? '';

    if (strpos($href, '/') === 0) {
      // Absolute path
      return $scheme . '://' . $host . $href;
    }

    // Relative path
    $base_path = $base_parts['path'] ?? '/';
    $base_dir = dirname($base_path);
    if ($base_dir === '\\' || $base_dir === '.') $base_dir = '/';

    return $scheme . '://' . $host . rtrim($base_dir, '/') . '/' . $href;
  }

  /**
   * Update links when a post is saved
   */
  public static function on_post_save($post_id, $post) {
    if (wp_is_post_revision($post_id)) return;
    if (wp_is_post_autosave($post_id)) return;
    if (!in_array($post->post_type, ['post', 'page'])) return;

    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');
    $site_host = wp_parse_url(get_site_url(), PHP_URL_HOST);

    // Delete existing links from this source
    $wpdb->delete($table, ['source_post_id' => $post_id]);

    // Re-extract if published/draft
    if (in_array($post->post_status, ['publish', 'draft', 'private'])) {
      $links = self::extract_internal_links($post->post_content, $post_id, $site_host);
      foreach ($links as $link) {
        $wpdb->insert($table, $link);
      }
    }
  }

  /**
   * Remove links when a post is deleted
   */
  public static function on_post_delete($post_id) {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');
    $wpdb->delete($table, ['source_post_id' => $post_id]);
  }

  /**
   * Clear all data
   */
  public static function clear_all() {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');
    $wpdb->query("TRUNCATE TABLE {$table}");
  }

  /**
   * Get stats
   */
  private static function get_stats() {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    $total = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    $sources = (int)$wpdb->get_var("SELECT COUNT(DISTINCT source_post_id) FROM {$table}");
    $targets = (int)$wpdb->get_var("SELECT COUNT(DISTINCT target_post_id) FROM {$table} WHERE target_post_id IS NOT NULL");

    return [
      'total_links' => $total,
      'source_pages' => $sources,
      'target_pages' => $targets,
    ];
  }

  /**
   * Get links with filters and pagination
   */
  private static function get_links($filter_source = 0, $filter_target = 0, $filter_anchor = '', $limit = 50, $offset = 0) {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    $where = '1=1';
    $params = [];

    if ($filter_source > 0) {
      $where .= ' AND e.source_post_id = %d';
      $params[] = $filter_source;
    }

    if ($filter_target > 0) {
      $where .= ' AND e.target_post_id = %d';
      $params[] = $filter_target;
    }

    if ($filter_anchor !== '') {
      $where .= ' AND e.anchor_text LIKE %s';
      $params[] = '%' . $wpdb->esc_like($filter_anchor) . '%';
    }

    $params[] = (int)$limit;
    $params[] = (int)$offset;

    $docs = InternalLinksTool_DB::table('documents');
    $kws  = InternalLinksTool_DB::table('keywords');

    $sql = "
      SELECT e.*,
             ps.post_title AS source_title,
             pt.post_title AS target_title,
             sk.primary_keyword AS source_primary_kw,
             tk.primary_keyword AS target_primary_kw
      FROM {$table} e
      LEFT JOIN {$wpdb->posts} ps ON ps.ID = e.source_post_id
      LEFT JOIN {$wpdb->posts} pt ON pt.ID = e.target_post_id
      LEFT JOIN {$docs} sd ON sd.post_id = e.source_post_id
      LEFT JOIN {$kws} sk ON sk.document_id = sd.id
      LEFT JOIN {$docs} td ON td.post_id = e.target_post_id
      LEFT JOIN {$kws} tk ON tk.document_id = td.id
      WHERE {$where}
      ORDER BY e.source_post_id ASC, e.link_position ASC
      LIMIT %d OFFSET %d
    ";

    return $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);
  }

  /**
   * Count links with filters
   */
  private static function count_links($filter_source = 0, $filter_target = 0, $filter_anchor = '') {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    $where = '1=1';
    $params = [];

    if ($filter_source > 0) {
      $where .= ' AND source_post_id = %d';
      $params[] = $filter_source;
    }

    if ($filter_target > 0) {
      $where .= ' AND target_post_id = %d';
      $params[] = $filter_target;
    }

    if ($filter_anchor !== '') {
      $where .= ' AND anchor_text LIKE %s';
      $params[] = '%' . $wpdb->esc_like($filter_anchor) . '%';
    }

    $sql = "SELECT COUNT(*) FROM {$table} WHERE {$where}";

    if (empty($params)) {
      return (int)$wpdb->get_var($sql);
    }

    return (int)$wpdb->get_var($wpdb->prepare($sql, $params));
  }

  /**
   * Get all source posts for filter dropdown
   */
  private static function get_all_source_posts() {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    return $wpdb->get_results(
      "SELECT e.source_post_id AS post_id, p.post_title AS title, COUNT(*) AS link_count
       FROM {$table} e
       LEFT JOIN {$wpdb->posts} p ON p.ID = e.source_post_id
       GROUP BY e.source_post_id, p.post_title
       ORDER BY p.post_title ASC",
      ARRAY_A
    );
  }

  /**
   * Get all target posts for filter dropdown
   */
  private static function get_all_target_posts() {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    return $wpdb->get_results(
      "SELECT e.target_post_id AS post_id, p.post_title AS title, COUNT(*) AS link_count
       FROM {$table} e
       LEFT JOIN {$wpdb->posts} p ON p.ID = e.target_post_id
       WHERE e.target_post_id IS NOT NULL
       GROUP BY e.target_post_id, p.post_title
       ORDER BY p.post_title ASC",
      ARRAY_A
    );
  }

  /**
   * Truncate URL for display
   */
  private static function truncate_url($url, $max = 40) {
    if (strlen($url) <= $max) return $url;
    return substr($url, 0, $max - 3) . '...';
  }

  /**
   * Handle CSV export
   */
  public static function handle_export_csv() {
    if (!current_user_can('manage_options')) {
      wp_die('No permission');
    }

    // Verify nonce
    $nonce = isset($_POST['_wpnonce']) ? $_POST['_wpnonce'] : '';
    if (!$nonce || !wp_verify_nonce($nonce, 'internallinkstool_export_existing_links')) {
      wp_die('Nonce verification failed. Please refresh the page and try again.');
    }

    // Get filter parameters
    $filter_source = isset($_POST['filter_source']) ? (int)$_POST['filter_source'] : 0;
    $filter_target = isset($_POST['filter_target']) ? (int)$_POST['filter_target'] : 0;
    $filter_anchor = isset($_POST['filter_anchor']) ? sanitize_text_field($_POST['filter_anchor']) : '';

    // Get all links (no pagination limit)
    $links = self::get_links_for_export($filter_source, $filter_target, $filter_anchor);

    // Generate filename
    $filename = 'internal-links-export-' . date('Y-m-d-His') . '.csv';

    // Set headers for CSV download
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');

    // Open output stream
    $output = fopen('php://output', 'w');

    // Add UTF-8 BOM for Excel compatibility
    fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

    // Write CSV header row
    fputcsv($output, [
      'Source Page',
      'Source URL',
      'Target Page',
      'Target URL',
      'Anchor Text',
      'Link Position',
    ]);

    // Write data rows
    foreach ($links as $link) {
      $source_title = $link['source_title'] ?: 'Post #' . $link['source_post_id'];
      $target_title = $link['target_title'] ?: ($link['target_post_id'] ? 'Post #' . $link['target_post_id'] : 'External/Unknown');

      fputcsv($output, [
        $source_title,
        $link['source_url'],
        $target_title,
        $link['target_url'],
        $link['anchor_text'],
        $link['link_position'],
      ]);
    }

    fclose($output);
    exit;
  }

  /**
   * Get all links for export (no limit)
   */
  private static function get_links_for_export($filter_source = 0, $filter_target = 0, $filter_anchor = '') {
    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    $where = '1=1';
    $params = [];

    if ($filter_source > 0) {
      $where .= ' AND e.source_post_id = %d';
      $params[] = $filter_source;
    }

    if ($filter_target > 0) {
      $where .= ' AND e.target_post_id = %d';
      $params[] = $filter_target;
    }

    if ($filter_anchor !== '') {
      $where .= ' AND e.anchor_text LIKE %s';
      $params[] = '%' . $wpdb->esc_like($filter_anchor) . '%';
    }

    $sql = "
      SELECT e.*,
             ps.post_title AS source_title,
             pt.post_title AS target_title
      FROM {$table} e
      LEFT JOIN {$wpdb->posts} ps ON ps.ID = e.source_post_id
      LEFT JOIN {$wpdb->posts} pt ON pt.ID = e.target_post_id
      WHERE {$where}
      ORDER BY e.source_post_id ASC, e.link_position ASC
    ";

    if (empty($params)) {
      return $wpdb->get_results($sql, ARRAY_A);
    }

    return $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);
  }

  /**
   * Handle removing selected links from content
   */
  public static function handle_remove_links() {
    if (!current_user_can('manage_options')) {
      wp_die('No permission');
    }

    // Verify nonce
    $nonce = isset($_POST['_wpnonce']) ? $_POST['_wpnonce'] : '';
    if (!$nonce || !wp_verify_nonce($nonce, 'internallinkstool_remove_links')) {
      wp_die('Nonce verification failed. Please refresh the page and try again.');
    }

    // Get selected link IDs
    $link_ids = isset($_POST['link_ids']) && is_array($_POST['link_ids']) ? array_map('intval', $_POST['link_ids']) : [];

    if (empty($link_ids)) {
      $redirect_url = self::build_redirect_url('No links selected.');
      wp_safe_redirect($redirect_url);
      exit;
    }

    global $wpdb;
    $table = InternalLinksTool_DB::table('existing_links');

    // Fetch the link details
    $placeholders = implode(',', array_fill(0, count($link_ids), '%d'));
    $links = $wpdb->get_results(
      $wpdb->prepare(
        "SELECT * FROM {$table} WHERE id IN ({$placeholders})",
        $link_ids
      ),
      ARRAY_A
    );

    if (empty($links)) {
      $redirect_url = self::build_redirect_url('Selected links not found.');
      wp_safe_redirect($redirect_url);
      exit;
    }

    // Group links by source post ID to minimize post updates
    $links_by_post = [];
    foreach ($links as $link) {
      $pid = (int)$link['source_post_id'];
      if (!isset($links_by_post[$pid])) {
        $links_by_post[$pid] = [];
      }
      $links_by_post[$pid][] = $link;
    }

    $removed_count = 0;
    $error_count = 0;

    foreach ($links_by_post as $post_id => $post_links) {
      $post = get_post($post_id);
      if (!$post) {
        $error_count += count($post_links);
        continue;
      }

      $content = $post->post_content;
      $original_content = $content;

      foreach ($post_links as $link) {
        $target_url = $link['target_url'];
        $anchor_text = $link['anchor_text'];

        // Remove the specific link from content
        $content = self::remove_link_from_content($content, $target_url, $anchor_text);
      }

      // Only update if content changed
      if ($content !== $original_content) {
        // Temporarily unhook our save_post action to prevent re-scanning during update
        remove_action('save_post', [__CLASS__, 'on_post_save'], 20);

        wp_update_post([
          'ID' => $post_id,
          'post_content' => $content,
        ]);

        // Re-hook our save_post action
        add_action('save_post', [__CLASS__, 'on_post_save'], 20, 2);

        $removed_count += count($post_links);

        // Delete the link records from our table
        foreach ($post_links as $link) {
          $wpdb->delete($table, ['id' => (int)$link['id']]);
        }
      } else {
        // Content didn't change - links may have already been removed or not found
        // Still delete from our tracking table
        foreach ($post_links as $link) {
          $wpdb->delete($table, ['id' => (int)$link['id']]);
          $removed_count++;
        }
      }
    }

    $msg = "Removed {$removed_count} link(s) from content.";
    if ($error_count > 0) {
      $msg .= " {$error_count} link(s) could not be processed.";
    }

    $redirect_url = self::build_redirect_url($msg);
    wp_safe_redirect($redirect_url);
    exit;
  }

  /**
   * Remove a specific link from HTML content, keeping the anchor text
   */
  private static function remove_link_from_content($content, $target_url, $anchor_text) {
    if (empty($content)) return $content;

    $doc = new DOMDocument();
    libxml_use_internal_errors(true);
    $wrapped = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>' . $content . '</body></html>';
    $doc->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();

    $xpath = new DOMXPath($doc);
    $anchors = $xpath->query('//a[@href]');

    $found = false;
    foreach ($anchors as $a) {
      $href = $a->getAttribute('href');
      $text = trim($a->textContent);

      // Normalize URLs for comparison
      $href_normalized = self::normalize_url_for_compare($href);
      $target_normalized = self::normalize_url_for_compare($target_url);

      // Match by URL and anchor text
      if ($href_normalized === $target_normalized && $text === $anchor_text) {
        // Replace the <a> element with its text content
        $text_node = $doc->createTextNode($a->textContent);
        $a->parentNode->replaceChild($text_node, $a);
        $found = true;
        break; // Only remove the first match
      }
    }

    if (!$found) {
      return $content;
    }

    // Extract body content
    $body = $doc->getElementsByTagName('body')->item(0);
    if (!$body) return $content;

    $new_content = '';
    foreach ($body->childNodes as $child) {
      $new_content .= $doc->saveHTML($child);
    }

    return $new_content;
  }

  /**
   * Normalize URL for comparison (remove trailing slashes, lowercase host)
   */
  private static function normalize_url_for_compare($url) {
    $url = trim($url);
    $url = rtrim($url, '/');
    $parts = wp_parse_url($url);

    if (isset($parts['host'])) {
      $parts['host'] = strtolower($parts['host']);
    }

    $normalized = '';
    if (isset($parts['scheme'])) $normalized .= $parts['scheme'] . '://';
    if (isset($parts['host'])) $normalized .= $parts['host'];
    if (isset($parts['port'])) $normalized .= ':' . $parts['port'];
    if (isset($parts['path'])) $normalized .= rtrim($parts['path'], '/');
    if (isset($parts['query'])) $normalized .= '?' . $parts['query'];

    return $normalized;
  }

  /**
   * Build redirect URL with filters preserved
   */
  private static function build_redirect_url($msg, $is_error = false) {
    $filter_source = isset($_POST['filter_source']) ? (int)$_POST['filter_source'] : 0;
    $filter_target = isset($_POST['filter_target']) ? (int)$_POST['filter_target'] : 0;
    $filter_anchor = isset($_POST['filter_anchor']) ? sanitize_text_field($_POST['filter_anchor']) : '';
    $paged = isset($_POST['paged']) ? (int)$_POST['paged'] : 1;

    $url = admin_url('admin.php?page=internallinkstool-existing-links');
    if ($filter_source) $url .= '&filter_source=' . $filter_source;
    if ($filter_target) $url .= '&filter_target=' . $filter_target;
    if ($filter_anchor) $url .= '&filter_anchor=' . urlencode($filter_anchor);
    if ($paged > 1) $url .= '&paged=' . $paged;

    $url .= '&' . ($is_error ? 'err' : 'msg') . '=' . rawurlencode($msg);

    return $url;
  }

  /**
   * Handle scan action
   */
  public static function handle_scan() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_scan_existing_links')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-existing-links&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    $result = self::scan_all_posts();

    $msg = 'Scan complete. Found ' . (int)$result['total_links'] . ' internal links across ' . (int)$result['posts_scanned'] . ' posts/pages.';
    wp_redirect(admin_url('admin.php?page=internallinkstool-existing-links&msg=' . rawurlencode($msg)));
    exit;
  }

  /**
   * Handle clear action
   */
  public static function handle_clear() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_clear_existing_links')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-existing-links&err=' . rawurlencode('Nonce failed.')));
      exit;
    }

    self::clear_all();

    wp_redirect(admin_url('admin.php?page=internallinkstool-existing-links&msg=' . rawurlencode('All link data cleared.')));
    exit;
  }

  /**
   * Scan all posts and pages for internal links
   */
  public static function scan_all_posts() {
    global $wpdb;

    $table = InternalLinksTool_DB::table('existing_links');

    // Clear existing data
    $wpdb->query("TRUNCATE TABLE {$table}");

    // Get all posts and pages
    $posts = $wpdb->get_results(
      "SELECT ID, post_content FROM {$wpdb->posts}
       WHERE post_type IN ('post', 'page')
       AND post_status IN ('publish', 'draft', 'private')
       AND post_content != ''",
      ARRAY_A
    );

    $site_url = trailingslashit(get_site_url());
    $site_host = wp_parse_url($site_url, PHP_URL_HOST);

    $total_links = 0;
    $posts_scanned = 0;

    foreach ($posts as $post) {
      $post_id = (int)$post['ID'];
      $content = $post['post_content'];

      $links_found = self::extract_internal_links($content, $post_id, $site_host);
      $total_links += count($links_found);
      $posts_scanned++;

      // Insert links
      foreach ($links_found as $link) {
        $wpdb->insert($table, $link);
      }
    }

    return [
      'posts_scanned' => $posts_scanned,
      'total_links' => $total_links,
    ];
  }
}
