<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_DB {

  // --- Back-compat: old code expects InternalLinksTool_DB::table('documents') ---
  public static function table($name) {
    $name = sanitize_key((string)$name);
    $map = [
      'documents'      => self::table_documents(),
      'keywords'       => self::table_keywords(),
      'changes'        => self::table_changes(),
      'existing_links' => self::table_existing_links(),
      'anchor_banks'   => self::table_anchor_banks(),
    ];
    if (isset($map[$name])) return $map[$name];

    global $wpdb;
    return $wpdb->prefix . 'internallinkstool_' . $name;
  }

  public static function table_documents() {
    global $wpdb;
    return $wpdb->prefix . 'internallinkstool_documents';
  }

  public static function table_keywords() {
    global $wpdb;
    return $wpdb->prefix . 'internallinkstool_keywords';
  }

  public static function table_changes() {
    global $wpdb;
    return $wpdb->prefix . 'internallinkstool_changes';
  }

  public static function table_existing_links() {
    global $wpdb;
    return $wpdb->prefix . 'internallinkstool_existing_links';
  }

  public static function table_anchor_banks() {
    global $wpdb;
    return $wpdb->prefix . 'internallinkstool_anchor_banks';
  }

  /**
   * Creates tables (called on plugin activation).
   * Safe to run multiple times.
   */
  public static function install() {
    global $wpdb;
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    $charset_collate = $wpdb->get_charset_collate();

    $docs           = self::table_documents();
    $keywords       = self::table_keywords();
    $changes        = self::table_changes();
    $existing_links = self::table_existing_links();
    $anchor_banks   = self::table_anchor_banks();

    $sql_docs = "CREATE TABLE {$docs} (
      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
      post_id BIGINT(20) UNSIGNED NOT NULL,
      url TEXT NULL,
      type VARCHAR(20) NULL,
      status VARCHAR(20) NULL,
      title TEXT NULL,
      h1 TEXT NULL,
      meta_title TEXT NULL,
      meta_desc TEXT NULL,
      meta_robots TEXT NULL,
      is_indexable TINYINT(1) NOT NULL DEFAULT 1,
      is_robots_blocked TINYINT(1) NOT NULL DEFAULT 0,
      content_hash VARCHAR(64) NULL,
      updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      UNIQUE KEY post_id (post_id)
    ) {$charset_collate};";

    $sql_keywords = "CREATE TABLE {$keywords} (
      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
      document_id BIGINT(20) UNSIGNED NOT NULL,
      primary_keyword TEXT NULL,
      secondary_keywords LONGTEXT NULL,
      source VARCHAR(50) NULL,
      confidence FLOAT NULL,
      updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      UNIQUE KEY document_id (document_id),
      KEY document_id_idx (document_id)
    ) {$charset_collate};";

    $sql_changes = "CREATE TABLE {$changes} (
      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
      run_id VARCHAR(64) NOT NULL,
      post_id BIGINT(20) UNSIGNED NOT NULL,
      before_content LONGTEXT NULL,
      after_content LONGTEXT NULL,
      created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      KEY run_id_idx (run_id),
      KEY post_id_idx (post_id)
    ) {$charset_collate};";

    $sql_existing_links = "CREATE TABLE {$existing_links} (
      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
      source_post_id BIGINT(20) UNSIGNED NOT NULL,
      source_url TEXT NULL,
      target_url TEXT NOT NULL,
      target_post_id BIGINT(20) UNSIGNED NULL,
      anchor_text TEXT NULL,
      link_position INT UNSIGNED NULL,
      updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      KEY source_post_id_idx (source_post_id),
      KEY target_post_id_idx (target_post_id)
    ) {$charset_collate};";

    $sql_anchor_banks = "CREATE TABLE {$anchor_banks} (
      id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
      document_id BIGINT(20) UNSIGNED NOT NULL,
      anchor_type VARCHAR(30) NOT NULL,
      anchor_text VARCHAR(255) NOT NULL,
      used_count INT UNSIGNED NOT NULL DEFAULT 0,
      created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (id),
      KEY document_id_idx (document_id),
      KEY anchor_type_idx (anchor_type),
      UNIQUE KEY doc_type_text (document_id, anchor_type, anchor_text(100))
    ) {$charset_collate};";

    dbDelta($sql_docs);
    dbDelta($sql_keywords);
    dbDelta($sql_changes);
    dbDelta($sql_existing_links);
    dbDelta($sql_anchor_banks);
  }

  /**
   * Upsert document row from WP post data.
   */
  public static function upsert_document_from_post($post_id) {
    global $wpdb;

    $post_id = (int)$post_id;
    if ($post_id <= 0) return false;

    $post = get_post($post_id);
    if (!$post) return false;

    $url = get_permalink($post_id);
    if (!is_string($url)) $url = '';
    $url = (string)$url;

    $type   = (string)$post->post_type;
    $status = (string)$post->post_status;

    $title   = (string)get_the_title($post_id);
    $content = (string)$post->post_content;

    // ✅ Pull meta via YOUR actual API: InternalLinksTool_Meta::extract()
    $meta_title  = '';
    $meta_desc   = '';
    $meta_robots = '';
    $h1          = '';

    if (class_exists('InternalLinksTool_Meta') && method_exists('InternalLinksTool_Meta', 'extract')) {
      $m = (array) InternalLinksTool_Meta::extract($post_id);

      $meta_title  = isset($m['meta_title']) ? (string)$m['meta_title'] : '';
      // your meta class uses meta_description key
      $meta_desc   = isset($m['meta_description']) ? (string)$m['meta_description'] : '';
      $meta_robots = isset($m['meta_robots']) ? (string)$m['meta_robots'] : '';
      $h1          = isset($m['h1']) ? (string)$m['h1'] : '';
    }

    // Indexable detection
    $is_indexable = 1;
    if ($meta_robots) {
      $robots_l = strtolower($meta_robots);
      if (strpos($robots_l, 'noindex') !== false) $is_indexable = 0;
    }

    // Robots blocked detection (supports both method names)
    $is_robots_blocked = 0;
    if ($url !== '' && class_exists('InternalLinksTool_Robots')) {
      if (method_exists('InternalLinksTool_Robots', 'is_url_blocked')) {
        $is_robots_blocked = InternalLinksTool_Robots::is_url_blocked($url) ? 1 : 0;
      } elseif (method_exists('InternalLinksTool_Robots', 'is_url_allowed')) {
        $is_robots_blocked = InternalLinksTool_Robots::is_url_allowed($url) ? 0 : 1;
      }
    }

    $content_hash = md5($content);

    $table = self::table_documents();

    $data = [
      'post_id'           => $post_id,
      'url'               => $url,
      'type'              => $type,
      'status'            => $status,
      'title'             => $title,
      'h1'                => $h1,
      'meta_title'        => $meta_title,
      'meta_desc'         => $meta_desc,
      'meta_robots'       => $meta_robots,
      'is_indexable'      => (int)$is_indexable,
      'is_robots_blocked' => (int)$is_robots_blocked,
      'content_hash'      => $content_hash,
      'updated_at'        => current_time('mysql'),
    ];

    $exists = $wpdb->get_var($wpdb->prepare(
      "SELECT id FROM {$table} WHERE post_id = %d LIMIT 1",
      $post_id
    ));

    if ($exists) {
      $wpdb->update($table, $data, ['post_id' => $post_id]);
      return (int)$exists;
    }

    $wpdb->insert($table, $data);
    return (int)$wpdb->insert_id;
  }

  /**
   * Return array of post IDs that are eligible but NOT yet mapped in documents table.
   */
  public static function get_unmapped_post_ids($types, $statuses, $limit = 50) {
    global $wpdb;

    $types    = is_array($types) ? array_values(array_filter($types)) : ['post','page'];
    $statuses = is_array($statuses) ? array_values(array_filter($statuses)) : ['publish'];
    $limit    = max(1, (int)$limit);

    $types    = array_map('sanitize_key', $types);
    $statuses = array_map('sanitize_key', $statuses);

    $docs  = self::table_documents();
    $posts = $wpdb->posts;

    $type_placeholders   = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $sql = "
      SELECT p.ID
      FROM {$posts} p
      LEFT JOIN {$docs} d ON d.post_id = p.ID
      WHERE d.post_id IS NULL
        AND p.post_type IN ({$type_placeholders})
        AND p.post_status IN ({$status_placeholders})
      ORDER BY p.ID ASC
      LIMIT %d
    ";

    $params = array_merge($types, $statuses, [$limit]);
    return array_map('intval', $wpdb->get_col($wpdb->prepare($sql, $params)));
  }

  public static function count_documents_total() {
    global $wpdb;
    $docs = self::table_documents();
    return (int)$wpdb->get_var("SELECT COUNT(*) FROM {$docs}");
  }

  public static function count_documents_by_types_and_statuses($types, $statuses) {
    global $wpdb;
    $docs = self::table_documents();

    $types    = is_array($types) ? array_values(array_filter($types)) : ['post','page'];
    $statuses = is_array($statuses) ? array_values(array_filter($statuses)) : ['publish'];

    $types    = array_map('sanitize_key', $types);
    $statuses = array_map('sanitize_key', $statuses);

    $type_placeholders   = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $sql = "
      SELECT COUNT(*)
      FROM {$docs}
      WHERE type IN ({$type_placeholders})
        AND status IN ({$status_placeholders})
    ";

    $params = array_merge($types, $statuses);
    return (int)$wpdb->get_var($wpdb->prepare($sql, $params));
  }
}
