<?php


if (!defined('ABSPATH')) exit;





class InternalLinksTool_CSV {


    


    /**


     * Expected CSV:


     * keyword,url


     * human dog beds,https://example.com/human-dog-beds


     *


     * This importer stores rows into: wp_internallinkstool_csv_targets


     *


     * IMPORTANT:


     * - CSV rows are PRIORITY targets (but they do NOT delete DB targets).


     * - This file assumes the CSV table exists (we’ll add it in DB.php).


     */


    public static function import_csv($filepath) {


        global $wpdb;


        


        $table = self::table();


        


        if (!file_exists($filepath) || !is_readable($filepath)) {


            throw new Exception('CSV file not readable');


        }


        


        $handle = fopen($filepath, 'r');


        if (!$handle) {


            throw new Exception('Failed to open CSV file');


        }


        


        // Read header


        $header = fgetcsv($handle);


        if (!is_array($header)) {


            fclose($handle);


            throw new Exception('CSV missing header row');


        }


        


        $header_map = self::normalize_header($header);


        if (!isset($header_map['keyword']) || !isset($header_map['url'])) {


            fclose($handle);


            throw new Exception('CSV header must include: keyword,url');


        }


        


        $inserted = 0;


        $updated = 0;


        $skipped = 0;


        


        $now = current_time('mysql');


        


        // Use transaction when possible


        $wpdb->query('START TRANSACTION');


        


        try {


            while (($row = fgetcsv($handle)) !== false) {


                if (!is_array($row) || count($row) < 2) {


                    $skipped++;


                    continue;


                }


                


                $keyword = trim((string)($row[$header_map['keyword']] ?? ''));


                $url     = trim((string)($row[$header_map['url']] ?? ''));


                


                if ($keyword === '' || $url === '') {


                    $skipped++;


                    continue;


                }


                


                $keyword_norm = self::normalize_keyword($keyword);


                $url_norm = self::normalize_url($url);


                


                // Basic URL validation


                if (!filter_var($url, FILTER_VALIDATE_URL)) {


                    $skipped++;


                    continue;


                }


                


                // Upsert on keyword_norm (unique)


                $existing_id = (int) $wpdb->get_var(


                    $wpdb->prepare("SELECT id FROM $table WHERE keyword_norm = %s", $keyword_norm)


                    );


                


                $data = [


                    'keyword' => $keyword,


                    'keyword_norm' => $keyword_norm,


                    'url' => $url,


                    'url_norm' => $url_norm,


                    'updated_at' => $now,


                ];


                


                if ($existing_id > 0) {


                    $ok = $wpdb->update($table, $data, ['id' => $existing_id]);


                    if ($ok !== false) $updated++;


                    else $skipped++;


                } else {


                    $data['created_at'] = $now;


                    $ok = $wpdb->insert($table, $data);


                    if ($ok) $inserted++;


                    else $skipped++;


                }


            }


            


            fclose($handle);


            $wpdb->query('COMMIT');


        } catch (Exception $e) {


            fclose($handle);


            $wpdb->query('ROLLBACK');


            throw $e;


        }


        


        return [


            'inserted' => $inserted,


            'updated' => $updated,


            'skipped' => $skipped,


        ];


    }


    


    /**


     * Returns all CSV targets as a normalized pool for the linker:


     * [


     *   [


     *     'keyword' => 'Human dog beds',


     *     'keyword_norm' => 'human dog beds',


     *     'url' => 'https://...',


     *     'url_norm' => 'https://...'


     *   ],


     *   ...


     * ]


     */


    public static function get_targets($limit = 5000) {


        global $wpdb;


        $table = self::table();


        


        $limit = max(1, min(20000, (int)$limit));


        


        $rows = $wpdb->get_results(


            $wpdb->prepare("SELECT keyword, keyword_norm, url, url_norm FROM $table ORDER BY updated_at DESC LIMIT %d", $limit),


            ARRAY_A


            );


        


        if (!is_array($rows)) return [];


        return $rows;


    }


    


    public static function count_targets() {


        global $wpdb;


        $table = self::table();


        return (int) $wpdb->get_var("SELECT COUNT(*) FROM $table");


    }


    


    public static function clear_all() {


        global $wpdb;


        $table = self::table();


        $wpdb->query("TRUNCATE TABLE $table");


    }


    


    // -----------------------


    // Internals


    // -----------------------


    


    private static function table() {


        return InternalLinksTool_DB::table('csv_targets');


    }


    


    private static function normalize_header($header) {


        // returns map: ['keyword'=>index, 'url'=>index]


        $map = [];


        foreach ($header as $i => $col) {


            $c = trim(strtolower((string)$col));


            $c = str_replace([' ', "\t", "\r", "\n"], '', $c);


            


            if ($c === 'keyword' || $c === 'kw' || $c === 'primarykeyword') {


                $map['keyword'] = $i;


            }


            if ($c === 'url' || $c === 'page' || $c === 'targeturl') {


                $map['url'] = $i;


            }


        }


        return $map;


    }


    


    private static function normalize_keyword($keyword) {


        $k = wp_strip_all_tags((string)$keyword);


        $k = trim($k);


        $k = preg_replace('/\s+/u', ' ', $k);


        // lowercase for matching


        if (function_exists('mb_strtolower')) $k = mb_strtolower($k);


        else $k = strtolower($k);


        return $k;


    }


    


    private static function normalize_url($url) {


        $u = trim((string)$url);


        // remove trailing slashes


        $u = preg_replace('~/+$~', '', $u);


        return $u;


    }


}


