<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_AnchorBanks {

  public static function init() {
    add_action('admin_post_internallinkstool_run_anchor_banks', [__CLASS__, 'handle_run_anchor_banks']);
    add_action('admin_post_internallinkstool_reset_anchor_banks', [__CLASS__, 'handle_reset_anchor_banks']);

    // AJAX handler for single-page anchor bank processing (progress bar)
    add_action('wp_ajax_internallinkstool_process_single_anchor_bank', [__CLASS__, 'ajax_process_single_anchor_bank']);
  }

  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : '';
    $err = isset($_GET['err']) ? sanitize_text_field($_GET['err']) : '';

    echo '<div class="wrap">';
    echo '<h1>AI Anchor Banks</h1>';

    echo '<div style="background:#fef8e7;border-left:4px solid #f0c33c;padding:12px 16px;margin:10px 0 16px;">';
    echo '<p style="margin:0;font-size:13px;"><strong>You don\'t need to run this manually.</strong> The <a href="' . esc_url(admin_url('admin.php?page=internallinkstool-linker')) . '">Linker</a> page\'s "Run All" handles scanning, keyword extraction, and anchor bank generation automatically. This page (along with Scanner and Keywords) is available for SEO experts who want more granular control &mdash; to review generated anchor variations, check quality, or regenerate banks for specific pages.</p>';
    echo '</div>';

    // Explanation box
    echo '<div style="background:#f0f6fc;border:1px solid #c3c4c7;border-left:4px solid #2271b1;padding:12px 16px;margin:15px 0;">';
    echo '<p style="margin:0 0 8px;"><strong>What are Anchor Banks?</strong> &mdash; For each page that has keywords, AI generates 15-25 natural-sounding anchor text variations (exact matches, synonyms, descriptive phrases, contextual phrases, and generic terms). These give the Linker many more natural phrases to choose from when inserting internal links.</p>';
    echo '<p style="margin:0 0 8px;"><strong>Why separate from Keywords?</strong> &mdash; Each page requires 1 AI call for anchor bank generation. Running this together with keyword extraction (which also uses 1 AI call per page) caused timeouts on larger batches. Separating them lets you run each step independently.</p>';
    echo '<p style="margin:0;"><strong>How it works:</strong> Pages that already have keywords extracted but don\'t yet have anchor banks will be processed. Uses 1 AI call per page.</p>';
    echo '</div>';

    if ($err) echo '<div class="notice notice-error"><p>' . esc_html($err) . '</p></div>';
    if ($msg) echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';

    // Anchor bank status
    $bank_stats = InternalLinksTool_Strategy::get_anchor_bank_stats();
    echo '<p><strong>Anchor Bank Status:</strong> ';
    echo '<code>' . (int)$bank_stats['pages_with_banks'] . '</code> pages with banks, ';
    echo '<code>' . (int)$bank_stats['total_anchors'] . '</code> total anchors, ';
    echo '<code>' . (int)$bank_stats['pages_needing_banks'] . '</code> pages needing banks.';
    echo '</p>';

    // AJAX-based anchor bank generation with progress bar
    $ab_nonce = wp_create_nonce('internallinkstool_batch_anchor_banks');
    $init_with_banks = (int)$bank_stats['pages_with_banks'];
    $init_needing = (int)$bank_stats['pages_needing_banks'];
    $init_total_ab = $init_with_banks + $init_needing;
    ?>
    <div style="margin:10px 0;">
      <button type="button" id="ilt-ab-start" class="button button-primary button-hero">Generate Anchor Banks</button>
      <button type="button" id="ilt-ab-stop" class="button button-secondary" style="display:none;margin-left:8px;">Stop</button>
    </div>

    <div id="ilt-ab-progress-wrap" style="display:none;margin:15px 0 20px;">
      <div style="background:#e0e0e0;border-radius:4px;height:28px;width:100%;max-width:600px;overflow:hidden;">
        <div id="ilt-ab-bar" style="background:#2271b1;height:100%;width:0%;transition:width 0.3s;display:flex;align-items:center;justify-content:center;color:#fff;font-size:13px;font-weight:600;min-width:40px;">0%</div>
      </div>
      <p id="ilt-ab-stats" style="margin:6px 0;font-size:13px;color:#555;">Processed: 0 | Remaining: 0 | Errors: 0 | Elapsed: 0s</p>
    </div>

    <div id="ilt-ab-log-wrap" style="display:none;margin:10px 0;">
      <div id="ilt-ab-log" style="max-height:300px;overflow-y:auto;background:#1d2327;color:#c3c4c7;font-family:monospace;font-size:12px;padding:10px;border-radius:4px;line-height:1.6;"></div>
    </div>

    <div id="ilt-ab-summary" style="display:none;margin:15px 0;padding:12px 16px;background:#e7f5e7;border:1px solid #46b450;border-radius:4px;">
      <strong>Anchor bank generation complete!</strong>
      <span id="ilt-ab-summary-text"></span>
      &mdash; <a href="<?php echo esc_url(admin_url('admin.php?page=internallinkstool-anchor-banks')); ?>">Refresh Page</a>
    </div>

    <script type="text/javascript">
    (function(){
      var running = false;
      var startBtn = document.getElementById('ilt-ab-start');
      var stopBtn = document.getElementById('ilt-ab-stop');
      var progressWrap = document.getElementById('ilt-ab-progress-wrap');
      var bar = document.getElementById('ilt-ab-bar');
      var stats = document.getElementById('ilt-ab-stats');
      var logWrap = document.getElementById('ilt-ab-log-wrap');
      var log = document.getElementById('ilt-ab-log');
      var summary = document.getElementById('ilt-ab-summary');
      var summaryText = document.getElementById('ilt-ab-summary-text');
      var nonce = '<?php echo esc_js($ab_nonce); ?>';

      var initTotal = <?php echo (int)$init_total_ab; ?>;
      var processed = 0, errorCount = 0, startTime = 0;

      startBtn.addEventListener('click', function(){
        if (running) return;
        running = true;
        processed = 0;
        errorCount = 0;
        startTime = Date.now();

        startBtn.style.display = 'none';
        stopBtn.style.display = '';
        progressWrap.style.display = '';
        logWrap.style.display = '';
        summary.style.display = 'none';
        log.innerHTML = '';
        bar.style.width = '0%';
        bar.textContent = '0%';

        appendLog('Starting anchor bank generation...');
        processNext();
      });

      stopBtn.addEventListener('click', function(){
        running = false;
        stopBtn.disabled = true;
        stopBtn.textContent = 'Stopping...';
        appendLog('Stop requested — finishing current page...');
      });

      function processNext() {
        if (!running) {
          finish('Stopped by user.');
          return;
        }

        var formData = new FormData();
        formData.append('action', 'internallinkstool_process_single_anchor_bank');
        formData.append('_ajax_nonce', nonce);

        fetch(ajaxurl, { method: 'POST', body: formData })
          .then(function(r){ return r.json(); })
          .then(function(resp){
            if (!resp.success) {
              appendLog('ERROR: ' + (resp.data || 'Unknown error'), true);
              errorCount++;
              if (running) { setTimeout(processNext, 2000); }
              return;
            }

            var d = resp.data;

            if (d.done) {
              updateBar(d.progress);
              appendLog('All pages processed.');
              running = false;
              finish('All done!');
              return;
            }

            processed++;
            updateBar(d.progress);

            var line = '#' + d.doc_id + ' ' + d.url + ' — "' + d.primary + '" — ' + d.anchors_generated + ' anchors';
            if (!d.success) {
              line = '#' + d.doc_id + ' ' + d.url + ' — ERROR: ' + d.error;
              errorCount++;
            }
            appendLog(line, !d.success);

            if (running) { processNext(); }
            else { finish('Stopped by user.'); }
          })
          .catch(function(err){
            appendLog('Network error: ' + err.message + ' — retrying...', true);
            errorCount++;
            if (running) { setTimeout(processNext, 2000); }
            else { finish('Stopped after error.'); }
          });
      }

      function updateBar(progress) {
        if (!progress) return;
        var total = initTotal > 0 ? initTotal : (progress.pages_with_banks + progress.pages_needing_banks);
        var done = total > 0 ? total - progress.pages_needing_banks : 0;
        var pct = total > 0 ? Math.round((done / total) * 100) : 0;
        bar.style.width = pct + '%';
        bar.textContent = pct + '%';
        var elapsed = Math.round((Date.now() - startTime) / 1000);
        stats.textContent = 'Processed: ' + processed + ' | With banks: ' + progress.pages_with_banks +
          ' | Remaining: ' + progress.pages_needing_banks + ' | Errors: ' + errorCount + ' | Elapsed: ' + elapsed + 's';
      }

      function appendLog(text, isError) {
        var span = document.createElement('div');
        span.textContent = text;
        if (isError) span.style.color = '#e65054';
        log.appendChild(span);
        log.scrollTop = log.scrollHeight;
      }

      function finish(msg) {
        stopBtn.style.display = 'none';
        stopBtn.disabled = false;
        stopBtn.textContent = 'Stop';
        startBtn.style.display = '';
        var elapsed = Math.round((Date.now() - startTime) / 1000);
        summaryText.textContent = ' Processed ' + processed + ' pages, ' + errorCount + ' errors, ' + elapsed + 's elapsed.';
        summary.style.display = '';
        appendLog(msg + ' (' + processed + ' pages, ' + errorCount + ' errors, ' + elapsed + 's)');
      }
    })();
    </script>
    <?php

    // Reset Anchor Banks
    echo '<hr>';
    echo '<h2>Reset Anchor Banks</h2>';
    echo '<p class="description">Clear all generated anchor banks from the database. You can regenerate them afterwards by clicking "Generate Anchor Banks" above.</p>';
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'Are you sure you want to reset ALL anchor banks? This cannot be undone.\');">';
    echo '<input type="hidden" name="action" value="internallinkstool_reset_anchor_banks" />';
    wp_nonce_field('internallinkstool_reset_anchor_banks');
    submit_button('Reset All Anchor Banks', 'delete');
    echo '</form>';

    // Paginated anchor banks table
    echo '<hr>';
    echo '<h2>Generated Anchor Banks</h2>';
    self::render_paginated_anchor_banks();

    echo '</div>';
  }

  /**
   * Render paginated table of all anchor banks grouped by page.
   */
  private static function render_paginated_anchor_banks() {
    if (!class_exists('InternalLinksTool_DB')) {
      echo '<p>Database class not available.</p>';
      return;
    }

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');
    $docs  = InternalLinksTool_DB::table('documents');
    $kws   = InternalLinksTool_DB::table('keywords');

    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) {
      echo '<p><em>Anchor banks table not yet created. Generate anchor banks first.</em></p>';
      return;
    }

    // Pagination params
    $per_page = isset($_GET['ab_per_page']) ? max(1, min(200, (int)$_GET['ab_per_page'])) : 20;
    $paged    = isset($_GET['ab_paged']) ? max(1, (int)$_GET['ab_paged']) : 1;
    $offset   = ($paged - 1) * $per_page;

    // Total pages with banks
    $total = (int)$wpdb->get_var("SELECT COUNT(DISTINCT document_id) FROM {$banks}");

    if ($total === 0) {
      echo '<p><em>No anchor banks generated yet.</em></p>';
      return;
    }

    $total_pages = max(1, (int)ceil($total / $per_page));
    if ($paged > $total_pages) $paged = $total_pages;

    // Fetch paginated rows grouped by document
    $sql = $wpdb->prepare("
      SELECT b.document_id, d.url, d.post_id, k.primary_keyword,
             pm.meta_value AS yoast_focus_kw,
             GROUP_CONCAT(CASE WHEN b.anchor_type = 'exact' THEN b.anchor_text END SEPARATOR '||') AS exact_anchors,
             GROUP_CONCAT(CASE WHEN b.anchor_type = 'partial' THEN b.anchor_text END SEPARATOR '||') AS partial_anchors,
             GROUP_CONCAT(CASE WHEN b.anchor_type = 'descriptive' THEN b.anchor_text END SEPARATOR '||') AS descriptive_anchors,
             GROUP_CONCAT(CASE WHEN b.anchor_type = 'contextual' THEN b.anchor_text END SEPARATOR '||') AS contextual_anchors,
             GROUP_CONCAT(CASE WHEN b.anchor_type = 'generic' THEN b.anchor_text END SEPARATOR '||') AS generic_anchors,
             COUNT(*) AS total_anchors
      FROM {$banks} b
      INNER JOIN {$docs} d ON d.id = b.document_id
      LEFT JOIN {$kws} k ON k.document_id = b.document_id
      LEFT JOIN {$wpdb->postmeta} pm ON pm.post_id = d.post_id AND pm.meta_key = '_yoast_wpseo_focuskw'
      GROUP BY b.document_id, d.url, d.post_id, k.primary_keyword, pm.meta_value
      ORDER BY b.document_id DESC
      LIMIT %d OFFSET %d
    ", $per_page, $offset);

    $rows = $wpdb->get_results($sql, ARRAY_A);

    // Pagination controls
    $base_url = admin_url('admin.php?page=internallinkstool-anchor-banks');
    echo '<form method="get" action="' . esc_url($base_url) . '" style="margin:10px 0;display:flex;align-items:center;gap:10px;">';
    echo '<input type="hidden" name="page" value="internallinkstool-anchor-banks" />';
    echo '<input type="hidden" name="ab_paged" value="1" />';
    echo '<label>URLs per page: <input type="number" name="ab_per_page" value="' . $per_page . '" min="1" max="200" style="width:60px;" /></label>';
    echo '<button type="submit" class="button button-small">Apply</button>';
    echo '<span style="margin-left:10px;color:#666;">Showing page <strong>' . $paged . '</strong> of <strong>' . $total_pages . '</strong> (' . $total . ' pages total)</span>';
    echo '</form>';

    // Prev / Next
    echo '<div style="margin:5px 0 15px;">';
    if ($paged > 1) {
      $prev_url = add_query_arg(['ab_paged' => $paged - 1, 'ab_per_page' => $per_page], $base_url);
      echo '<a href="' . esc_url($prev_url) . '" class="button button-small">&laquo; Previous</a> ';
    }
    if ($paged < $total_pages) {
      $next_url = add_query_arg(['ab_paged' => $paged + 1, 'ab_per_page' => $per_page], $base_url);
      echo '<a href="' . esc_url($next_url) . '" class="button button-small">Next &raquo;</a>';
    }
    echo '</div>';

    if (empty($rows)) {
      echo '<p><em>No results for this page.</em></p>';
      return;
    }

    echo '<table class="widefat fixed striped">';
    echo '<thead><tr>';
    echo '<th style="width:16%;">URL</th>';
    echo '<th style="width:8%;">Yoast Focus KW</th>';
    echo '<th style="width:8%;">Primary KW</th>';
    echo '<th style="width:16%;">Exact Anchors</th>';
    echo '<th style="width:16%;">Partial/Semantic</th>';
    echo '<th style="width:14%;">Descriptive</th>';
    echo '<th style="width:12%;">Contextual</th>';
    echo '<th style="width:7%;">Generic</th>';
    echo '<th style="width:3%;">#</th>';
    echo '</tr></thead><tbody>';

    foreach ($rows as $r) {
      $url = esc_html((string)($r['url'] ?? ''));
      echo '<tr>';
      echo '<td style="word-break:break-all;font-size:11px;"><a href="' . esc_url($r['url']) . '" target="_blank">' . $url . '</a></td>';
      echo '<td style="font-size:11px;"><code>' . esc_html($r['yoast_focus_kw'] ?? '-') . '</code></td>';
      echo '<td style="font-size:11px;"><code>' . esc_html($r['primary_keyword'] ?? '-') . '</code></td>';

      // Render each anchor type as pills
      $types = ['exact_anchors', 'partial_anchors', 'descriptive_anchors', 'contextual_anchors', 'generic_anchors'];
      $colors = ['#d32f2f', '#1976d2', '#388e3c', '#7b1fa2', '#666'];
      foreach ($types as $ti => $col_name) {
        $raw = (string)($r[$col_name] ?? '');
        echo '<td style="font-size:10px;line-height:1.8;">';
        if ($raw !== '') {
          $items = array_filter(array_map('trim', explode('||', $raw)));
          foreach ($items as $item) {
            echo '<span style="background:' . $colors[$ti] . '10;border:1px solid ' . $colors[$ti] . '33;color:' . $colors[$ti] . ';padding:1px 5px;border-radius:3px;display:inline-block;margin:1px 2px;">' . esc_html($item) . '</span>';
          }
        } else {
          echo '<span style="color:#ccc;">-</span>';
        }
        echo '</td>';
      }

      echo '<td>' . (int)$r['total_anchors'] . '</td>';
      echo '</tr>';
    }

    echo '</tbody></table>';

    // Bottom pagination
    echo '<div style="margin:10px 0;">';
    if ($paged > 1) {
      $prev_url = add_query_arg(['ab_paged' => $paged - 1, 'ab_per_page' => $per_page], $base_url);
      echo '<a href="' . esc_url($prev_url) . '" class="button button-small">&laquo; Previous</a> ';
    }
    if ($paged < $total_pages) {
      $next_url = add_query_arg(['ab_paged' => $paged + 1, 'ab_per_page' => $per_page], $base_url);
      echo '<a href="' . esc_url($next_url) . '" class="button button-small">Next &raquo;</a>';
    }
    echo '</div>';
  }

  public static function handle_run_anchor_banks() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_run_anchor_banks')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-anchor-banks&err=' . rawurlencode('Nonce failed. Refresh and try again.')));
      exit;
    }

    $batch_size = isset($_POST['batch_size']) ? (int)$_POST['batch_size'] : 20;
    $batch_size = max(1, min(100, $batch_size));

    try {
      $res = self::run_batch($batch_size);
      $msg = 'Anchor bank generation complete. Processed: ' . (int)$res['processed'] . ', Banks generated: ' . (int)$res['banks_generated'] . ' anchors';
      if ($res['errors']) {
        $msg .= ', Errors: ' . (int)$res['errors'];
      }
      $msg .= '.';
      wp_redirect(admin_url('admin.php?page=internallinkstool-anchor-banks&batch_size=' . $batch_size . '&msg=' . rawurlencode($msg)));
      exit;
    } catch (Exception $e) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-anchor-banks&batch_size=' . $batch_size . '&err=' . rawurlencode('Anchor banks error: ' . $e->getMessage())));
      exit;
    }
  }

  public static function run_batch($batch_size = 20) {
    if (!class_exists('InternalLinksTool_DB')) throw new Exception('DB class not loaded.');
    global $wpdb;

    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');
    $banks = InternalLinksTool_DB::table('anchor_banks');

    // Get settings for filters
    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];

    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    $type_placeholders = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Ensure anchor_banks table exists before LEFT JOIN
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) {
      // Let generate_anchor_bank_for_page create it on first call
      // For now, just query pages with keywords
      $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, k.primary_keyword, k.secondary_keywords
              FROM {$docs} d
              INNER JOIN {$keywords} k ON k.document_id = d.id
              WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
                AND d.type IN ({$type_placeholders})
                AND d.status IN ({$status_placeholders})
                {$robots_sql}
              GROUP BY d.id
              ORDER BY d.id ASC
              LIMIT %d";
    } else {
      // Query documents that have keywords but no anchor banks yet
      $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, k.primary_keyword, k.secondary_keywords
              FROM {$docs} d
              INNER JOIN {$keywords} k ON k.document_id = d.id
              LEFT JOIN {$banks} b ON b.document_id = d.id
              WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
                AND b.id IS NULL
                AND d.type IN ({$type_placeholders})
                AND d.status IN ({$status_placeholders})
                {$robots_sql}
              GROUP BY d.id
              ORDER BY d.id ASC
              LIMIT %d";
    }

    $params = array_merge($types, $statuses, [(int)$batch_size]);
    $rows = $wpdb->get_results($wpdb->prepare($sql, $params), ARRAY_A);

    if (!is_array($rows)) $rows = [];

    $processed = 0; $banks_generated = 0; $errors = 0;

    foreach ($rows as $r) {
      $processed++;

      $doc_id = (int)($r['id'] ?? 0);
      $primary = trim((string)($r['primary_keyword'] ?? ''));
      $secondary = trim((string)($r['secondary_keywords'] ?? ''));
      $meta_title = trim((string)($r['meta_title'] ?? ''));
      $meta_desc = trim((string)($r['meta_desc'] ?? ''));

      try {
        $count = InternalLinksTool_Strategy::generate_anchor_bank_for_page($doc_id, $primary, $secondary, $meta_title, $meta_desc);
        $banks_generated += $count;
      } catch (Exception $e) {
        $errors++;
      }
    }

    return ['processed' => $processed, 'banks_generated' => $banks_generated, 'errors' => $errors];
  }

  /**
   * Process a single page for anchor bank generation.
   * Returns result array with success/error info, or 'none_remaining' error when done.
   */
  public static function process_single_page() {
    if (!class_exists('InternalLinksTool_DB')) throw new Exception('DB class not loaded.');
    global $wpdb;

    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];
    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    $type_placeholders = implode(',', array_fill(0, count($types), '%s'));
    $status_placeholders = implode(',', array_fill(0, count($statuses), '%s'));

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Check if anchor_banks table exists
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if (!$table_exists) {
      $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, k.primary_keyword, k.secondary_keywords
              FROM {$docs} d
              INNER JOIN {$keywords} k ON k.document_id = d.id
              WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
                AND d.type IN ({$type_placeholders})
                AND d.status IN ({$status_placeholders})
                {$robots_sql}
              GROUP BY d.id
              ORDER BY d.id ASC
              LIMIT 1";
    } else {
      $sql = "SELECT d.id, d.post_id, d.url, d.meta_title, d.meta_desc, k.primary_keyword, k.secondary_keywords
              FROM {$docs} d
              INNER JOIN {$keywords} k ON k.document_id = d.id
              LEFT JOIN {$banks} b ON b.document_id = d.id
              WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
                AND b.id IS NULL
                AND d.type IN ({$type_placeholders})
                AND d.status IN ({$status_placeholders})
                {$robots_sql}
              GROUP BY d.id
              ORDER BY d.id ASC
              LIMIT 1";
    }

    $params = array_merge($types, $statuses);
    $r = $wpdb->get_row($wpdb->prepare($sql, $params), ARRAY_A);

    if (!$r) {
      return ['success' => true, 'doc_id' => 0, 'url' => '', 'anchors_generated' => 0, 'primary' => '', 'error' => 'none_remaining'];
    }

    $doc_id = (int)($r['id'] ?? 0);
    $url = trim((string)($r['url'] ?? ''));
    $primary = trim((string)($r['primary_keyword'] ?? ''));
    $secondary = trim((string)($r['secondary_keywords'] ?? ''));
    $meta_title = trim((string)($r['meta_title'] ?? ''));
    $meta_desc = trim((string)($r['meta_desc'] ?? ''));

    try {
      $count = InternalLinksTool_Strategy::generate_anchor_bank_for_page($doc_id, $primary, $secondary, $meta_title, $meta_desc);
      return ['success' => true, 'doc_id' => $doc_id, 'url' => $url, 'anchors_generated' => (int)$count, 'primary' => $primary, 'error' => ''];
    } catch (Exception $e) {
      return ['success' => false, 'doc_id' => $doc_id, 'url' => $url, 'anchors_generated' => 0, 'primary' => $primary, 'error' => $e->getMessage()];
    }
  }

  /**
   * AJAX handler for single-page anchor bank processing (progress bar).
   */
  public static function ajax_process_single_anchor_bank() {
    if (!current_user_can('manage_options')) {
      wp_send_json_error('No permission');
    }

    if (!check_ajax_referer('internallinkstool_batch_anchor_banks', '_ajax_nonce', false)) {
      wp_send_json_error('Nonce verification failed');
    }

    $result = self::process_single_page();

    // If no pages remaining, return done
    if ($result['error'] === 'none_remaining') {
      $progress = InternalLinksTool_Strategy::get_anchor_bank_stats();
      wp_send_json_success([
        'done' => true,
        'progress' => $progress,
      ]);
    }

    $progress = InternalLinksTool_Strategy::get_anchor_bank_stats();

    wp_send_json_success([
      'done' => false,
      'doc_id' => $result['doc_id'],
      'url' => $result['url'],
      'primary' => $result['primary'],
      'anchors_generated' => $result['anchors_generated'],
      'success' => $result['success'],
      'error' => $result['error'],
      'progress' => $progress,
    ]);
  }

  public static function handle_reset_anchor_banks() {
    if (!current_user_can('manage_options')) wp_die('No permission');

    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'internallinkstool_reset_anchor_banks')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-anchor-banks&err=' . rawurlencode('Nonce failed. Refresh and try again.')));
      exit;
    }

    if (!class_exists('InternalLinksTool_DB')) {
      wp_redirect(admin_url('admin.php?page=internallinkstool-anchor-banks&err=' . rawurlencode('DB class not available.')));
      exit;
    }

    global $wpdb;
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    if ($table_exists) {
      $wpdb->query("TRUNCATE TABLE {$banks}");
    }

    wp_redirect(admin_url('admin.php?page=internallinkstool-anchor-banks&msg=' . rawurlencode('All anchor banks have been cleared.')));
    exit;
  }

  private static function get_allowed_types($settings) {
    $types = [];
    if (!empty($settings['include_posts'])) $types[] = 'post';
    if (!empty($settings['include_pages'])) $types[] = 'page';
    if (empty($types)) $types = ['post', 'page'];
    return $types;
  }

  private static function get_allowed_statuses($settings) {
    if (!empty($settings['scan_statuses']) && is_array($settings['scan_statuses'])) {
      $statuses = array_values(array_unique(array_map('sanitize_key', $settings['scan_statuses'])));
      $statuses = array_values(array_intersect($statuses, ['publish', 'draft']));
      if (!empty($statuses)) return $statuses;
    }
    return ['publish'];
  }

  /**
   * Get progress counts for anchor bank generation.
   * Returns total pages with keywords, done (have anchor banks), remaining.
   */
  public static function get_progress_counts() {
    if (!class_exists('InternalLinksTool_DB')) return ['total' => 0, 'done' => 0, 'remaining' => 0];
    global $wpdb;

    $docs = InternalLinksTool_DB::table('documents');
    $keywords = InternalLinksTool_DB::table('keywords');
    $banks = InternalLinksTool_DB::table('anchor_banks');

    $settings = class_exists('InternalLinksTool_Admin') ? InternalLinksTool_Admin::get_settings() : [];
    $types = self::get_allowed_types($settings);
    $statuses = self::get_allowed_statuses($settings);

    $type_ph = implode(',', array_fill(0, count($types), '%s'));
    $status_ph = implode(',', array_fill(0, count($statuses), '%s'));

    $robots_sql = '';
    if (!empty($settings['respect_robots'])) {
      $robots_sql = ' AND d.is_indexable = 1 AND d.is_robots_blocked = 0 ';
    }

    // Total = pages that have keywords (eligible for anchor bank generation)
    $params = array_merge($types, $statuses);
    $total = (int)$wpdb->get_var($wpdb->prepare(
      "SELECT COUNT(DISTINCT d.id) FROM {$docs} d
       INNER JOIN {$keywords} k ON k.document_id = d.id
       WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
         AND d.type IN ({$type_ph}) AND d.status IN ({$status_ph}) {$robots_sql}",
      $params
    ));

    // Done = pages that already have at least one anchor bank entry
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$banks}'");
    $done = 0;
    if ($table_exists) {
      $done = (int)$wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(DISTINCT d.id) FROM {$docs} d
         INNER JOIN {$keywords} k ON k.document_id = d.id
         INNER JOIN {$banks} b ON b.document_id = d.id
         WHERE k.primary_keyword IS NOT NULL AND k.primary_keyword != ''
           AND d.type IN ({$type_ph}) AND d.status IN ({$status_ph}) {$robots_sql}",
        $params
      ));
    }

    return ['total' => $total, 'done' => $done, 'remaining' => max(0, $total - $done)];
  }
}
