<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_Admin {

  const OPT_SETTINGS = 'internallinkstool_settings';

  public static function init() {
    // IMPORTANT: no admin_menu here — menus are owned by internal-links-tool.php
    add_action('admin_init', [__CLASS__, 'register_settings']);

    // migrate legacy settings once
    add_action('admin_init', [__CLASS__, 'maybe_migrate_legacy_settings'], 5);
  }

  public static function register_settings() {
    register_setting(self::OPT_SETTINGS . '_group', self::OPT_SETTINGS, [
      'type' => 'array',
      'sanitize_callback' => [__CLASS__, 'sanitize_settings'],
      'default' => [],
    ]);
  }

  public static function render_settings_page() {
    if (!current_user_can('manage_options')) return;

    $s = self::get_settings();

    echo '<div class="wrap">';
    echo '<h1>Link Settings</h1>';

    echo '<form method="post" action="options.php">';
    settings_fields(self::OPT_SETTINGS . '_group');

    echo '<table class="form-table">';

    // Max links per page
    echo '<tr><th scope="row">Max links per page</th><td>';
    echo '<input type="number" min="0" max="50" name="' . esc_attr(self::OPT_SETTINGS) . '[max_links_per_page]" value="' . esc_attr((int)$s['max_links_per_page']) . '"/>';
    echo '<p class="description">0 disables insertion.</p>';
    echo '</td></tr>';

    // Max links to same target per page
    echo '<tr><th scope="row">Max links to the same target per page</th><td>';
    echo '<input type="number" min="1" max="10" name="' . esc_attr(self::OPT_SETTINGS) . '[max_links_per_target]" value="' . esc_attr((int)$s['max_links_per_target']) . '"/>';
    echo '<p class="description">Example: 1 = do not link to the same page twice from one page.</p>';
    echo '</td></tr>';

    // Skip sentences
    echo '<tr><th scope="row">Skip sentences</th><td>';
    echo '<input type="number" min="0" max="50" name="' . esc_attr(self::OPT_SETTINGS) . '[skip_sentences]" value="' . esc_attr((int)$s['skip_sentences']) . '"/>';
    echo '<p class="description">Do not insert links within the first X sentences of a post/page.</p>';
    echo '</td></tr>';

    // Spread mode
    echo '<tr><th scope="row">Spread mode</th><td>';
    echo '<select name="' . esc_attr(self::OPT_SETTINGS) . '[spread_mode]">';
    $modes = [
      'start'  => 'Start (front-loaded)',
      'middle' => 'Middle',
      'end'    => 'End',
      'spread' => 'Spread (best)',
    ];
    foreach ($modes as $k => $label) {
      echo '<option value="' . esc_attr($k) . '" ' . selected($s['spread_mode'], $k, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select>';
    echo '</td></tr>';

    // Anchor mode
    echo '<tr><th scope="row">Anchor mode</th><td>';
    echo '<select name="' . esc_attr(self::OPT_SETTINGS) . '[anchor_mode]">';
    echo '<option value="exact" ' . selected($s['anchor_mode'], 'exact', false) . '>Exact</option>';
    echo '<option value="fuzzy" ' . selected($s['anchor_mode'], 'fuzzy', false) . '>Fuzzy (later with AI/synonyms)</option>';
    echo '</select>';
    echo '</td></tr>';

    // Anchor rewriting (AI)
    echo '<tr><th scope="row">AI anchor rewriting</th><td>';
    echo '<input type="hidden" name="' . esc_attr(self::OPT_SETTINGS) . '[anchor_rewriting]" value="0" />';
    echo '<label><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[anchor_rewriting]" value="1" ' . checked($s['anchor_rewriting'], 1, false) . '> Enable AI-powered anchor rewriting</label>';
    echo '<p class="description">When enabled, AI will suggest more natural anchor text in dry-run preview. Uses additional API calls.</p>';
    echo '</td></tr>';

    // Link title attribute
    echo '<tr><th scope="row">Link title attribute</th><td>';
    echo '<input type="hidden" name="' . esc_attr(self::OPT_SETTINGS) . '[link_title_attr]" value="0" />';
    echo '<label><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[link_title_attr]" value="1" ' . checked($s['link_title_attr'], 1, false) . '> Add a title attribute to inserted links</label>';
    echo '<p class="description">When enabled, each internal link will include a <code>title</code> HTML attribute set to the anchor text (e.g. <code>&lt;a href="..." title="anchor text"&gt;</code>).</p>';
    echo '</td></tr>';

    // Scanner: include statuses
    echo '<tr><th scope="row">Scan post statuses</th><td>';
    $scan_statuses = isset($s['scan_statuses']) && is_array($s['scan_statuses']) ? $s['scan_statuses'] : ['publish'];
    $all_statuses = [
      'publish' => 'Published',
      'draft'   => 'Draft',
    ];
    foreach ($all_statuses as $st => $label) {
      $checked = in_array($st, $scan_statuses, true) ? 'checked' : '';
      echo '<label style="display:inline-block;margin-right:14px;"><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[scan_statuses][]" value="' . esc_attr($st) . '" ' . $checked . '> ' . esc_html($label) . '</label>';
    }
    echo '<p class="description">Default is <strong>Published</strong> only. This controls what the Scanner considers eligible.</p>';
    echo '</td></tr>';

    // Include post types (hidden fields ensure unchecked = 0 is sent)
    echo '<tr><th scope="row">Include post types</th><td>';
    echo '<input type="hidden" name="' . esc_attr(self::OPT_SETTINGS) . '[include_posts]" value="0" />';
    echo '<label><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[include_posts]" value="1" ' . checked($s['include_posts'], 1, false) . '> Posts</label><br>';
    echo '<input type="hidden" name="' . esc_attr(self::OPT_SETTINGS) . '[include_pages]" value="0" />';
    echo '<label><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[include_pages]" value="1" ' . checked($s['include_pages'], 1, false) . '> Pages</label>';
    echo '</td></tr>';

    // Posts can link to
    echo '<tr><th scope="row">Posts can link to</th><td>';
    echo '<select name="' . esc_attr(self::OPT_SETTINGS) . '[post_link_targets]">';
    $plt_options = ['all' => 'All types', 'post' => 'Posts only', 'page' => 'Pages only', 'none' => 'No links'];
    foreach ($plt_options as $k => $label) {
      echo '<option value="' . esc_attr($k) . '" ' . selected($s['post_link_targets'], $k, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select>';
    echo '<p class="description">Restrict which post types a <strong>post</strong> can link to.</p>';
    echo '</td></tr>';

    // Pages can link to
    echo '<tr><th scope="row">Pages can link to</th><td>';
    echo '<select name="' . esc_attr(self::OPT_SETTINGS) . '[page_link_targets]">';
    $palt_options = ['all' => 'All types', 'post' => 'Posts only', 'page' => 'Pages only', 'none' => 'No links'];
    foreach ($palt_options as $k => $label) {
      echo '<option value="' . esc_attr($k) . '" ' . selected($s['page_link_targets'], $k, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select>';
    echo '<p class="description">Restrict which post types a <strong>page</strong> can link to.</p>';
    echo '</td></tr>';

    // Same category only (posts)
    echo '<tr><th scope="row">Same category only (posts)</th><td>';
    echo '<input type="hidden" name="' . esc_attr(self::OPT_SETTINGS) . '[same_category_only]" value="0" />';
    echo '<label><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[same_category_only]" value="1" ' . checked($s['same_category_only'], 1, false) . '> Only link between posts sharing categories</label>';
    echo '</td></tr>';

    // Respect robots
    echo '<tr><th scope="row">Respect robots (noindex/nofollow/robots blocked)</th><td>';
    echo '<input type="hidden" name="' . esc_attr(self::OPT_SETTINGS) . '[respect_robots]" value="0" />';
    echo '<label><input type="checkbox" name="' . esc_attr(self::OPT_SETTINGS) . '[respect_robots]" value="1" ' . checked($s['respect_robots'], 1, false) . '> Exclude pages marked noindex/nofollow or blocked</label>';
    echo '</td></tr>';

    // Negative slug words
    echo '<tr><th scope="row">Exclude slug words</th><td>';
    echo '<input type="text" style="width:520px" name="' . esc_attr(self::OPT_SETTINGS) . '[exclude_slug_words]" value="' . esc_attr($s['exclude_slug_words']) . '"/>';
    echo '<p class="description">Comma-separated. Example: privacy,terms,about,contact,login,signup,cookie,gdpr,legal</p>';
    echo '</td></tr>';

    // Negative URLs
    echo '<tr><th scope="row">Exclude URLs (contains)</th><td>';
    echo '<textarea rows="4" style="width:520px" name="' . esc_attr(self::OPT_SETTINGS) . '[exclude_urls]">' . esc_textarea($s['exclude_urls']) . '</textarea>';
    echo '<p class="description">One per line or comma-separated. If a page URL contains any entry, it will be excluded.</p>';
    echo '</td></tr>';

    // Include URLs (whitelist override)
    echo '<tr><th scope="row">Include URLs only (whitelist)</th><td>';
    echo '<textarea rows="4" style="width:520px" name="' . esc_attr(self::OPT_SETTINGS) . '[include_urls]">' . esc_textarea($s['include_urls']) . '</textarea>';
    echo '<p class="description">One per line or comma-separated. If a target URL contains any entry, it will be included. Leave empty to allow all targets.</p>';
    if (trim($s['include_urls']) !== '') {
      echo '<div class="notice notice-warning inline" style="margin:8px 0 0;padding:8px 12px;"><strong>Warning:</strong> Include URLs is active. Only pages whose URL matches an entry above will receive links. All other pages will be skipped as targets.</div>';
    }
    echo '</td></tr>';

    // Daily limit (future cron)
    echo '<tr><th scope="row">Daily insertion limit (future cron)</th><td>';
    echo '<input type="number" min="0" max="500" name="' . esc_attr(self::OPT_SETTINGS) . '[daily_page_limit]" value="' . esc_attr((int)$s['daily_page_limit']) . '"/>';
    echo '<p class="description">0 disables daily limit. Cron wiring later.</p>';
    echo '</td></tr>';

    // Auto-process new posts
    echo '<tr><th scope="row">New post/page automation</th><td>';
    echo '<select name="' . esc_attr(self::OPT_SETTINGS) . '[auto_process_new]">';
    $auto_modes = [
      'manual' => 'Manual — Do nothing automatically',
      'auto'   => 'Fully Automate — Scan, Keywords, Anchor Bank, Linker',
    ];
    foreach ($auto_modes as $k => $label) {
      echo '<option value="' . esc_attr($k) . '" ' . selected($s['auto_process_new'], $k, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select>';
    echo '<p class="description"><strong>Manual:</strong> New posts require running each step manually.<br>';
    echo '<strong>Fully Automate:</strong> When a new post/page is published, automatically: scan it, extract keywords (AI), generate anchor bank (AI), and run linker to add internal links.</p>';
    echo '</td></tr>';

    // OpenAI API key
    echo '<tr><th scope="row">OpenAI API Key</th><td>';
    echo '<input type="password" style="width:520px" name="' . esc_attr(self::OPT_SETTINGS) . '[openai_api_key]" value="' . esc_attr($s['openai_api_key']) . '"/>';
    echo '</td></tr>';

    echo '</table>';

    submit_button('Save Settings');

    echo '</form>';
    echo '</div>';
  }

  public static function get_settings() {
    $s = get_option(self::OPT_SETTINGS, []);
    if (!is_array($s)) $s = [];
    return self::normalize_settings($s);
  }

  public static function sanitize_settings($input) {
    if (!is_array($input)) $input = [];
    return self::normalize_settings($input);
  }

  private static function normalize_settings($s) {
    $defaults = [
      'max_links_per_page'   => 3,
      'max_links_per_target' => 1,
      'skip_sentences'       => 2,
      'spread_mode'          => 'spread',
      'anchor_mode'          => 'exact',
      'anchor_rewriting'     => 0,
      'link_title_attr'      => 1,
      'daily_page_limit'     => 20,
      'respect_robots'       => 1,
      'same_category_only'   => 0,
      'exclude_slug_words'   => 'privacy,terms,about,contact,login,signup,cookie,gdpr,legal,disclaimer',
      'exclude_urls'         => '',
      'include_urls'         => '',
      'include_posts'        => 1,
      'include_pages'        => 1,
      'post_link_targets'    => 'all',
      'page_link_targets'    => 'all',
      'scan_statuses'        => ['publish'],
      'openai_api_key'       => '',
      'auto_process_new'     => 'manual',
    ];

    $s = array_merge($defaults, is_array($s) ? $s : []);

    $s['max_links_per_page']   = max(0, (int)$s['max_links_per_page']);
    $s['max_links_per_target'] = max(1, (int)$s['max_links_per_target']);
    $s['skip_sentences']       = max(0, (int)$s['skip_sentences']);
    $s['daily_page_limit']     = max(0, (int)$s['daily_page_limit']);

    $m = sanitize_key($s['spread_mode']);
    if (!in_array($m, ['start','middle','end','spread'], true)) $m = 'spread';
    $s['spread_mode'] = $m;

    $a = sanitize_key($s['anchor_mode']);
    if (!in_array($a, ['exact','fuzzy'], true)) $a = 'exact';
    $s['anchor_mode'] = $a;

    $s['anchor_rewriting']   = (int)!empty($s['anchor_rewriting']);
    $s['link_title_attr']    = (int)!empty($s['link_title_attr']);
    $s['respect_robots']     = (int)!empty($s['respect_robots']);
    $s['same_category_only'] = (int)!empty($s['same_category_only']);
    $s['include_posts']      = (int)!empty($s['include_posts']);
    $s['include_pages']      = (int)!empty($s['include_pages']);

    $plt = sanitize_key($s['post_link_targets'] ?? 'all');
    if (!in_array($plt, ['all', 'post', 'page', 'none'], true)) $plt = 'all';
    $s['post_link_targets'] = $plt;

    $palt = sanitize_key($s['page_link_targets'] ?? 'all');
    if (!in_array($palt, ['all', 'post', 'page', 'none'], true)) $palt = 'all';
    $s['page_link_targets'] = $palt;

    // Scan statuses (Scanner eligibility)
    $allowed_statuses = ['publish','draft'];
    $scan = $s['scan_statuses'] ?? ['publish'];
    if (!is_array($scan)) $scan = ['publish'];
    $scan = array_values(array_unique(array_map('sanitize_key', $scan)));
    $scan = array_values(array_intersect($scan, $allowed_statuses));
    if (empty($scan)) $scan = ['publish'];
    $s['scan_statuses'] = $scan;

    $s['exclude_slug_words'] = trim((string)$s['exclude_slug_words']);
    $s['exclude_urls']       = trim((string)$s['exclude_urls']);
    $s['include_urls']       = trim((string)$s['include_urls']);
    $s['openai_api_key']     = trim((string)$s['openai_api_key']);

    // Auto-process new posts setting
    $auto = sanitize_key($s['auto_process_new'] ?? 'manual');
    if (!in_array($auto, ['manual', 'auto'], true)) $auto = 'manual';
    $s['auto_process_new'] = $auto;

    return $s;
  }

  /**
   * One-time migration from legacy scattered options (ilt_* + internallinkstool_*)
   * into internallinkstool_settings array.
   */
  public static function maybe_migrate_legacy_settings() {
    $existing = get_option(self::OPT_SETTINGS, null);
    if (is_array($existing) && !empty($existing)) return;

    // Detect legacy by one known key
    $legacy = get_option('ilt_max_links_per_page', null);
    $legacy2 = get_option('internallinkstool_openai_api_key', null);
    if ($legacy === null && $legacy2 === null) {
      // Ensure defaults exist
      if (!is_array($existing)) {
        update_option(self::OPT_SETTINGS, self::normalize_settings([]), false);
      }
      return;
    }

    $ipt = get_option('ilt_include_post_types', []);
    $include_posts = 1;
    $include_pages = 1;
    if (is_array($ipt) && !empty($ipt)) {
      $include_posts = isset($ipt['post']) ? 1 : 0;
      $include_pages = isset($ipt['page']) ? 1 : 0;
      if ($include_posts === 0 && $include_pages === 0) { $include_posts = 1; $include_pages = 1; }
    }

    $migrated = [
      'max_links_per_page'   => (int)get_option('ilt_max_links_per_page', 3),
      'max_links_per_target' => (int)get_option('internallinkstool_max_links_per_target', 1),
      'skip_sentences'       => (int)get_option('ilt_skip_sentences', 2),
      'spread_mode'          => (string)get_option('ilt_spread_mode', 'spread'),
      'anchor_mode'          => (string)get_option('ilt_anchor_mode', 'exact'),
      'daily_page_limit'     => (int)get_option('ilt_daily_page_limit', 20),
      'respect_robots'       => (int)get_option('ilt_respect_robots', 1),
      'same_category_only'   => (int)get_option('ilt_same_category_only', 0),
      'exclude_slug_words'   => (string)get_option('ilt_exclude_slug_words', 'privacy,terms,about,contact,login,signup,cookie,gdpr,legal,disclaimer'),
      'exclude_urls'         => (string)get_option('ilt_exclude_urls', ''),
      'include_posts'        => (int)$include_posts,
      'include_pages'        => (int)$include_pages,
      'scan_statuses'        => (array)get_option('internallinkstool_scan_statuses', ['publish']),
      'openai_api_key'       => (string)get_option('internallinkstool_openai_api_key', ''),
    ];

    update_option(self::OPT_SETTINGS, self::normalize_settings($migrated), false);
  }
}
