<?php
if (!defined('ABSPATH')) exit;

class InternalLinksTool_AddLink {

  public static function init() {
    add_action('admin_post_internallinkstool_add_single_link', [__CLASS__, 'handle_single_add']);
    add_action('admin_post_internallinkstool_csv_import_links', [__CLASS__, 'handle_csv_import']);
  }

  /**
   * Render the Add New Link admin page.
   */
  public static function render_page() {
    if (!current_user_can('manage_options')) return;

    $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : '';
    $err = isset($_GET['err']) ? sanitize_text_field($_GET['err']) : '';
    $generate_ab = !empty($_GET['generate_ab']);
    $ab_count = isset($_GET['count']) ? (int)$_GET['count'] : 0;
    $csv_success = isset($_GET['csv_success']) ? (int)$_GET['csv_success'] : 0;
    $csv_errors = isset($_GET['csv_errors']) ? (int)$_GET['csv_errors'] : 0;

    echo '<div class="wrap">';
    echo '<h1>Add New Link</h1>';

    echo '<div style="background:#f0f6fc;border:1px solid #c3c4c7;border-left:4px solid #2271b1;padding:12px 16px;margin:15px 0;">';
    echo '<p style="margin:0;"><strong>Add pages manually</strong> &mdash; Insert pages into the internal linking system as if they had been scanned, keyworded, and optionally had AI anchor banks generated. The Linker will consider these pages as targets in future runs.</p>';
    echo '</div>';

    if ($err) echo '<div class="notice notice-error"><p>' . esc_html($err) . '</p></div>';
    if ($msg) echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';

    // ── Single Add Form ──
    echo '<h2>Add Single Page</h2>';
    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
    echo '<input type="hidden" name="action" value="internallinkstool_add_single_link" />';
    wp_nonce_field('internallinkstool_add_single_link');

    echo '<table class="form-table">';

    // Page URL
    echo '<tr><th scope="row"><label for="ilt-url">Page URL</label></th><td>';
    echo '<input type="url" id="ilt-url" name="page_url" value="" class="regular-text" required placeholder="https://yoursite.com/page-slug" />';
    echo '<p class="description">Must be a URL on this WordPress site.</p>';
    echo '</td></tr>';

    // Primary Keyword
    echo '<tr><th scope="row"><label for="ilt-primary-kw">Primary Keyword</label></th><td>';
    echo '<input type="text" id="ilt-primary-kw" name="primary_keyword" value="" class="regular-text" required placeholder="e.g. best dog beds" />';
    echo '</td></tr>';

    // Secondary Keywords
    echo '<tr><th scope="row"><label for="ilt-secondary-kw">Secondary Keywords</label></th><td>';
    echo '<input type="text" id="ilt-secondary-kw" name="secondary_keywords" value="" class="regular-text" placeholder="e.g. memory foam, orthopedic, large breed" />';
    echo '<p class="description">Comma-separated. Optional.</p>';
    echo '</td></tr>';

    // Tier
    echo '<tr><th scope="row"><label for="ilt-tier">Tier</label></th><td>';
    echo '<select id="ilt-tier" name="tier">';
    echo '<option value="1">Tier 1 (highest priority)</option>';
    echo '<option value="2" selected>Tier 2 (default)</option>';
    echo '<option value="3">Tier 3 (lowest priority)</option>';
    echo '</select>';
    echo '</td></tr>';

    // Page Type
    echo '<tr><th scope="row"><label for="ilt-page-type">Page Type</label></th><td>';
    echo '<select id="ilt-page-type" name="page_type">';
    echo '<option value="post" selected>Post</option>';
    echo '<option value="page">Page</option>';
    echo '</select>';
    echo '</td></tr>';

    // Generate AI Anchor Banks
    echo '<tr><th scope="row">AI Anchor Banks</th><td>';
    echo '<label><input type="checkbox" name="generate_anchor_banks" value="1" checked /> Generate AI Anchor Banks for this page</label>';
    echo '<p class="description">Creates diverse anchor text variations. Uses 1 AI call.</p>';
    echo '</td></tr>';

    echo '</table>';

    submit_button('Add Page');
    echo '</form>';

    echo '<hr>';

    // ── CSV Import Form ──
    echo '<h2>Bulk Import via CSV</h2>';
    echo '<p>Upload a CSV file to add multiple pages at once.</p>';

    echo '<div style="background:#f9f9f9;border:1px solid #ccd0d4;border-radius:4px;padding:15px;margin:10px 0 20px;">';
    echo '<strong>CSV Format:</strong>';
    echo '<pre style="background:#1d2327;color:#c3c4c7;padding:10px;border-radius:4px;margin:8px 0;overflow-x:auto;">url,primary_keyword,secondary_keywords,tier,page_type
https://example.com/dog-beds,best dog beds,"memory foam,orthopedic",2,post
https://example.com/about,about us,,1,page</pre>';
    echo '<p class="description" style="margin:0;">Headers are flexible: <code>URL</code>, <code>url</code>, <code>primary keyword</code>, <code>primary_keyword</code>, <code>keyword</code>, etc. are all accepted. Only <code>url</code> and <code>primary_keyword</code> are required columns.</p>';
    echo '</div>';

    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" enctype="multipart/form-data">';
    echo '<input type="hidden" name="action" value="internallinkstool_csv_import_links" />';
    wp_nonce_field('internallinkstool_csv_import_links');

    echo '<table class="form-table">';

    echo '<tr><th scope="row"><label for="ilt-csv-file">CSV File</label></th><td>';
    echo '<input type="file" id="ilt-csv-file" name="csv_file" accept=".csv" required />';
    echo '</td></tr>';

    echo '<tr><th scope="row">AI Anchor Banks</th><td>';
    echo '<label><input type="checkbox" name="generate_anchor_banks" value="1" checked /> Generate AI Anchor Banks for imported pages</label>';
    echo '<p class="description">Processes anchor banks via progress bar after import. Uses 1 AI call per page.</p>';
    echo '</td></tr>';

    echo '</table>';

    submit_button('Import CSV');
    echo '</form>';

    // ── Anchor Bank Progress Bar (for CSV imports) ──
    if ($generate_ab && $ab_count > 0) {
      self::render_anchor_bank_progress($ab_count);
    }

    echo '</div>'; // .wrap
  }

  /**
   * Handle single page add form submission.
   */
  public static function handle_single_add() {
    if (!current_user_can('manage_options')) {
      wp_die('Unauthorized');
    }

    check_admin_referer('internallinkstool_add_single_link');

    $url              = isset($_POST['page_url']) ? esc_url_raw(trim($_POST['page_url'])) : '';
    $primary_keyword  = isset($_POST['primary_keyword']) ? sanitize_text_field(trim($_POST['primary_keyword'])) : '';
    $secondary_kw_raw = isset($_POST['secondary_keywords']) ? sanitize_text_field(trim($_POST['secondary_keywords'])) : '';
    $tier_raw         = isset($_POST['tier']) ? sanitize_text_field($_POST['tier']) : '2';
    $page_type        = isset($_POST['page_type']) ? sanitize_text_field($_POST['page_type']) : 'post';
    $generate_ab      = !empty($_POST['generate_anchor_banks']);

    $redirect_base = admin_url('admin.php?page=internallinkstool-addlink');

    // Validate required fields
    if ($url === '' || $primary_keyword === '') {
      wp_safe_redirect(add_query_arg('err', 'Page URL and Primary Keyword are required.', $redirect_base));
      exit;
    }

    // Validate URL is on this site
    $home = untrailingslashit(home_url());
    if (stripos($url, $home) !== 0) {
      wp_safe_redirect(add_query_arg('err', 'URL must be on this WordPress site (' . esc_html($home) . ').', $redirect_base));
      exit;
    }

    // Resolve URL to post ID
    $post_id = self::resolve_url_to_post_id($url);

    if ($post_id <= 0) {
      wp_safe_redirect(add_query_arg('err', 'URL not found as a WordPress post/page: ' . esc_html($url), $redirect_base));
      exit;
    }

    // Upsert document
    if (!class_exists('InternalLinksTool_DB') || !method_exists('InternalLinksTool_DB', 'upsert_document_from_post')) {
      wp_safe_redirect(add_query_arg('err', 'Database class not available.', $redirect_base));
      exit;
    }

    $doc_id = InternalLinksTool_DB::upsert_document_from_post($post_id);

    if (!$doc_id) {
      wp_safe_redirect(add_query_arg('err', 'Failed to create document record for post ID ' . $post_id . '.', $redirect_base));
      exit;
    }

    // Ensure tier columns exist and set tier
    self::ensure_tier_columns();
    $tier = self::normalize_tier($tier_raw);

    global $wpdb;
    $docs_table = InternalLinksTool_DB::table('documents');
    $wpdb->update($docs_table, ['tier' => $tier], ['id' => (int)$doc_id]);

    // Save keywords
    $secondary_arr = self::parse_comma_separated($secondary_kw_raw);
    self::save_keywords($doc_id, $primary_keyword, $secondary_arr);

    // Generate anchor banks if requested
    $ab_msg = '';
    if ($generate_ab) {
      if (class_exists('InternalLinksTool_Strategy') && method_exists('InternalLinksTool_Strategy', 'generate_anchor_bank_for_page')) {
        $meta_title = '';
        $meta_desc  = '';

        $doc_row = $wpdb->get_row($wpdb->prepare(
          "SELECT meta_title, meta_desc FROM {$docs_table} WHERE id = %d LIMIT 1",
          (int)$doc_id
        ), ARRAY_A);

        if ($doc_row) {
          $meta_title = $doc_row['meta_title'] ?? '';
          $meta_desc  = $doc_row['meta_desc'] ?? '';
        }

        $count = InternalLinksTool_Strategy::generate_anchor_bank_for_page(
          $doc_id,
          $primary_keyword,
          implode(', ', $secondary_arr),
          $meta_title,
          $meta_desc
        );

        $ab_msg = ' Anchor banks generated (' . (int)$count . ' anchors).';
      }
    }

    wp_safe_redirect(add_query_arg('msg', 'Page added successfully: ' . esc_html($url) . '.' . $ab_msg, $redirect_base));
    exit;
  }

  /**
   * Handle CSV bulk import form submission.
   */
  public static function handle_csv_import() {
    if (!current_user_can('manage_options')) {
      wp_die('Unauthorized');
    }

    check_admin_referer('internallinkstool_csv_import_links');

    $redirect_base = admin_url('admin.php?page=internallinkstool-addlink');
    $generate_ab   = !empty($_POST['generate_anchor_banks']);

    // Check file upload
    if (empty($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
      wp_safe_redirect(add_query_arg('err', 'No CSV file uploaded or upload error.', $redirect_base));
      exit;
    }

    $file = $_FILES['csv_file']['tmp_name'];

    if (!is_uploaded_file($file)) {
      wp_safe_redirect(add_query_arg('err', 'Invalid file upload.', $redirect_base));
      exit;
    }

    $handle = fopen($file, 'r');
    if (!$handle) {
      wp_safe_redirect(add_query_arg('err', 'Could not read CSV file.', $redirect_base));
      exit;
    }

    // Read header row
    $header_row = fgetcsv($handle);
    if (!$header_row || !is_array($header_row)) {
      fclose($handle);
      wp_safe_redirect(add_query_arg('err', 'CSV file is empty or has no header row.', $redirect_base));
      exit;
    }

    // Normalize headers
    $headers = array_map([__CLASS__, 'normalize_csv_header'], $header_row);

    // Map headers to column indices
    $col_url       = self::find_header_index($headers, ['url', 'page_url', 'link', 'page_link']);
    $col_primary   = self::find_header_index($headers, ['primary_keyword', 'primary', 'keyword', 'main_keyword']);
    $col_secondary = self::find_header_index($headers, ['secondary_keywords', 'secondary', 'other_keywords']);
    $col_tier      = self::find_header_index($headers, ['tier', 'priority']);
    $col_page_type = self::find_header_index($headers, ['page_type', 'type', 'post_type']);

    if ($col_url === false || $col_primary === false) {
      fclose($handle);
      wp_safe_redirect(add_query_arg('err', 'CSV must have at least "url" and "primary_keyword" columns. Found headers: ' . esc_html(implode(', ', $headers)), $redirect_base));
      exit;
    }

    if (!class_exists('InternalLinksTool_DB') || !method_exists('InternalLinksTool_DB', 'upsert_document_from_post')) {
      fclose($handle);
      wp_safe_redirect(add_query_arg('err', 'Database class not available.', $redirect_base));
      exit;
    }

    self::ensure_tier_columns();

    global $wpdb;
    $docs_table = InternalLinksTool_DB::table('documents');

    $success_count = 0;
    $error_count   = 0;
    $error_urls    = [];
    $row_num       = 1;

    while (($row = fgetcsv($handle)) !== false) {
      $row_num++;

      $url = isset($row[$col_url]) ? esc_url_raw(trim($row[$col_url])) : '';
      $primary = isset($row[$col_primary]) ? sanitize_text_field(trim($row[$col_primary])) : '';

      if ($url === '' || $primary === '') {
        $error_count++;
        $error_urls[] = 'Row ' . $row_num . ': missing URL or keyword';
        continue;
      }

      $secondary_raw = ($col_secondary !== false && isset($row[$col_secondary])) ? sanitize_text_field(trim($row[$col_secondary])) : '';
      $tier_raw      = ($col_tier !== false && isset($row[$col_tier])) ? sanitize_text_field(trim($row[$col_tier])) : '2';
      // page_type from CSV is informational; upsert_document_from_post reads the actual post type

      $post_id = self::resolve_url_to_post_id($url);

      if ($post_id <= 0) {
        $error_count++;
        $error_urls[] = 'Row ' . $row_num . ': URL not found — ' . $url;
        continue;
      }

      $doc_id = InternalLinksTool_DB::upsert_document_from_post($post_id);

      if (!$doc_id) {
        $error_count++;
        $error_urls[] = 'Row ' . $row_num . ': failed to create document — ' . $url;
        continue;
      }

      // Set tier
      $tier = self::normalize_tier($tier_raw);
      $wpdb->update($docs_table, ['tier' => $tier], ['id' => (int)$doc_id]);

      // Save keywords
      $secondary_arr = self::parse_comma_separated($secondary_raw);
      self::save_keywords($doc_id, $primary, $secondary_arr);

      $success_count++;
    }

    fclose($handle);

    // Build result message
    $msg_parts = [];
    if ($success_count > 0) {
      $msg_parts[] = $success_count . ' page(s) imported successfully';
    }
    if ($error_count > 0) {
      $msg_parts[] = $error_count . ' row(s) skipped';
    }
    $result_msg = implode('. ', $msg_parts) . '.';

    if ($error_count > 0 && count($error_urls) > 0) {
      $result_msg .= ' Errors: ' . implode('; ', array_slice($error_urls, 0, 10));
      if (count($error_urls) > 10) {
        $result_msg .= ' (and ' . (count($error_urls) - 10) . ' more)';
      }
    }

    // Redirect: if anchor banks requested and we have successes, trigger progress bar
    if ($generate_ab && $success_count > 0) {
      $args = [
        'msg'         => $result_msg,
        'generate_ab' => 1,
        'count'       => $success_count,
        'csv_success' => $success_count,
        'csv_errors'  => $error_count,
      ];
      wp_safe_redirect(add_query_arg($args, $redirect_base));
      exit;
    }

    // No anchor banks: just show results
    $param = ($error_count > 0 && $success_count === 0) ? 'err' : 'msg';
    wp_safe_redirect(add_query_arg($param, $result_msg, $redirect_base));
    exit;
  }

  // ─── Helper Methods ───

  /**
   * Ensure tier and page_type columns exist in documents table.
   */
  private static function ensure_tier_columns() {
    global $wpdb;
    $docs = InternalLinksTool_DB::table('documents');

    $col_tier = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'tier'");
    if (!$col_tier) {
      $wpdb->query("ALTER TABLE {$docs} ADD COLUMN tier VARCHAR(20) NULL");
    }

    $col_type = $wpdb->get_var("SHOW COLUMNS FROM {$docs} LIKE 'page_type'");
    if (!$col_type) {
      $wpdb->query("ALTER TABLE {$docs} ADD COLUMN page_type VARCHAR(50) NULL");
    }
  }

  /**
   * Normalize tier value to "tier1"/"tier2"/"tier3".
   */
  private static function normalize_tier($val) {
    $val = strtolower(trim((string)$val));
    if (in_array($val, ['tier1', 'tier2', 'tier3'], true)) {
      return $val;
    }
    if (in_array($val, ['1', '2', '3'], true)) {
      return 'tier' . $val;
    }
    return 'tier2'; // default
  }

  /**
   * Parse comma-separated string into trimmed, non-empty array.
   */
  private static function parse_comma_separated($str) {
    if (!is_string($str) || $str === '') return [];
    $parts = explode(',', $str);
    $parts = array_map('trim', $parts);
    $parts = array_filter($parts, function($v) { return $v !== ''; });
    return array_values($parts);
  }

  /**
   * Normalize a CSV header: lowercase, trim, replace spaces with underscores.
   */
  private static function normalize_csv_header($header) {
    $header = strtolower(trim((string)$header));
    // Remove BOM if present
    $header = preg_replace('/^\xEF\xBB\xBF/', '', $header);
    $header = str_replace(' ', '_', $header);
    return $header;
  }

  /**
   * Find the index of a header from a list of accepted names.
   */
  private static function find_header_index($headers, $accepted_names) {
    foreach ($headers as $idx => $h) {
      if (in_array($h, $accepted_names, true)) {
        return $idx;
      }
    }
    return false;
  }

  /**
   * Resolve URL to WordPress post ID. Tries multiple variations.
   */
  private static function resolve_url_to_post_id($url) {
    $url = trim((string)$url);
    if ($url === '') return 0;

    // Try as-is
    $post_id = url_to_postid($url);
    if ($post_id > 0) return $post_id;

    // Try with trailing slash
    $post_id = url_to_postid(trailingslashit($url));
    if ($post_id > 0) return $post_id;

    // Try without trailing slash
    $post_id = url_to_postid(untrailingslashit($url));
    if ($post_id > 0) return $post_id;

    // Try prefixing with home_url if it looks like a relative path
    if (strpos($url, 'http') !== 0) {
      $full_url = home_url($url);
      $post_id = url_to_postid($full_url);
      if ($post_id > 0) return $post_id;
    }

    return 0;
  }

  /**
   * Save keywords for a document (insert or update).
   */
  private static function save_keywords($doc_id, $primary_keyword, $secondary_arr) {
    global $wpdb;

    $keywords_table = InternalLinksTool_DB::table('keywords');

    $secondary_csv = implode(', ', $secondary_arr);

    $data = [
      'document_id'        => (int)$doc_id,
      'primary_keyword'    => $primary_keyword,
      'secondary_keywords' => $secondary_csv,
      'source'             => 'manual',
      'confidence'         => 1.0,
      'updated_at'         => current_time('mysql'),
    ];

    $exists = $wpdb->get_var($wpdb->prepare(
      "SELECT id FROM {$keywords_table} WHERE document_id = %d LIMIT 1",
      (int)$doc_id
    ));

    if ($exists) {
      unset($data['document_id']);
      $wpdb->update($keywords_table, $data, ['document_id' => (int)$doc_id]);
    } else {
      $wpdb->insert($keywords_table, $data);
    }
  }

  /**
   * Render the anchor bank progress bar (auto-starting) for CSV imports.
   * Reuses the existing internallinkstool_process_single_anchor_bank AJAX endpoint.
   */
  private static function render_anchor_bank_progress($count) {
    $ab_nonce = wp_create_nonce('internallinkstool_batch_anchor_banks');
    $init_total = (int)$count;

    ?>
    <hr>
    <h2>Generating Anchor Banks</h2>
    <p>Processing anchor banks for the <?php echo (int)$count; ?> imported page(s). This will auto-start.</p>

    <div style="margin:10px 0;">
      <button type="button" id="ilt-addlink-ab-stop" class="button button-secondary" style="margin-left:0;">Stop</button>
    </div>

    <div id="ilt-addlink-ab-progress-wrap" style="margin:15px 0 20px;">
      <div style="background:#e0e0e0;border-radius:4px;height:28px;width:100%;max-width:600px;overflow:hidden;">
        <div id="ilt-addlink-ab-bar" style="background:#2271b1;height:100%;width:0%;transition:width 0.3s;display:flex;align-items:center;justify-content:center;color:#fff;font-size:13px;font-weight:600;min-width:40px;">0%</div>
      </div>
      <p id="ilt-addlink-ab-stats" style="margin:6px 0;font-size:13px;color:#555;">Processed: 0 | Remaining: <?php echo (int)$count; ?> | Errors: 0 | Elapsed: 0s</p>
    </div>

    <div id="ilt-addlink-ab-log-wrap" style="margin:10px 0;">
      <div id="ilt-addlink-ab-log" style="max-height:300px;overflow-y:auto;background:#1d2327;color:#c3c4c7;font-family:monospace;font-size:12px;padding:10px;border-radius:4px;line-height:1.6;"></div>
    </div>

    <div id="ilt-addlink-ab-summary" style="display:none;margin:15px 0;padding:12px 16px;background:#e7f5e7;border:1px solid #46b450;border-radius:4px;">
      <strong>Anchor bank generation complete!</strong>
      <span id="ilt-addlink-ab-summary-text"></span>
      &mdash; <a href="<?php echo esc_url(admin_url('admin.php?page=internallinkstool-anchor-banks')); ?>">View Anchor Banks</a>
    </div>

    <script type="text/javascript">
    (function(){
      var running = true;
      var stopBtn = document.getElementById('ilt-addlink-ab-stop');
      var bar = document.getElementById('ilt-addlink-ab-bar');
      var stats = document.getElementById('ilt-addlink-ab-stats');
      var log = document.getElementById('ilt-addlink-ab-log');
      var summary = document.getElementById('ilt-addlink-ab-summary');
      var summaryText = document.getElementById('ilt-addlink-ab-summary-text');
      var nonce = '<?php echo esc_js($ab_nonce); ?>';

      var initTotal = <?php echo (int)$init_total; ?>;
      var processed = 0, errorCount = 0, startTime = Date.now();

      appendLog('Starting anchor bank generation for ' + initTotal + ' imported page(s)...');
      processNext();

      stopBtn.addEventListener('click', function(){
        running = false;
        stopBtn.disabled = true;
        stopBtn.textContent = 'Stopping...';
        appendLog('Stop requested — finishing current page...');
      });

      function processNext() {
        if (!running) {
          finish('Stopped by user.');
          return;
        }

        var formData = new FormData();
        formData.append('action', 'internallinkstool_process_single_anchor_bank');
        formData.append('_ajax_nonce', nonce);

        fetch(ajaxurl, { method: 'POST', body: formData })
          .then(function(r){ return r.json(); })
          .then(function(resp){
            if (!resp.success) {
              appendLog('ERROR: ' + (resp.data || 'Unknown error'), true);
              errorCount++;
              if (running) { setTimeout(processNext, 2000); }
              return;
            }

            var d = resp.data;

            if (d.done) {
              updateBar(d.progress);
              appendLog('All pages processed.');
              running = false;
              finish('All done!');
              return;
            }

            processed++;
            updateBar(d.progress);

            var line = '#' + d.doc_id + ' ' + d.url + ' — "' + d.primary + '" — ' + d.anchors_generated + ' anchors';
            if (!d.success) {
              line = '#' + d.doc_id + ' ' + d.url + ' — ERROR: ' + d.error;
              errorCount++;
            }
            appendLog(line, !d.success);

            if (running) { processNext(); }
            else { finish('Stopped by user.'); }
          })
          .catch(function(err){
            appendLog('Network error: ' + err.message + ' — retrying...', true);
            errorCount++;
            if (running) { setTimeout(processNext, 2000); }
            else { finish('Stopped after error.'); }
          });
      }

      function updateBar(progress) {
        if (!progress) return;
        var total = initTotal > 0 ? initTotal : (progress.pages_with_banks + progress.pages_needing_banks);
        var done = total > 0 ? total - progress.pages_needing_banks : 0;
        if (done > total) done = total;
        var pct = total > 0 ? Math.round((done / total) * 100) : 0;
        bar.style.width = pct + '%';
        bar.textContent = pct + '%';
        var elapsed = Math.round((Date.now() - startTime) / 1000);
        stats.textContent = 'Processed: ' + processed + ' | With banks: ' + progress.pages_with_banks +
          ' | Remaining: ' + progress.pages_needing_banks + ' | Errors: ' + errorCount + ' | Elapsed: ' + elapsed + 's';
      }

      function appendLog(text, isError) {
        var span = document.createElement('div');
        span.textContent = text;
        if (isError) span.style.color = '#e65054';
        log.appendChild(span);
        log.scrollTop = log.scrollHeight;
      }

      function finish(msg) {
        stopBtn.style.display = 'none';
        var elapsed = Math.round((Date.now() - startTime) / 1000);
        summaryText.textContent = ' Processed ' + processed + ' pages, ' + errorCount + ' errors, ' + elapsed + 's elapsed.';
        summary.style.display = '';
        appendLog(msg + ' (' + processed + ' pages, ' + errorCount + ' errors, ' + elapsed + 's)');
      }
    })();
    </script>
    <?php
  }

}
